"""DisuData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import dis_data_base
from xms.mf6.data.dis_data_base import DisDataBase
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import options_util, units_util
from xms.mf6.gui.options_defs import CheckboxField


class DisuData(DisDataBase):
    """A DISU package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'DISU6'
        self._grid_info = self._base._grid_info if self._base._grid_info is not None else GridInfo(dis=DisEnum.DISU)
        self.list_blocks = {}  # Dict of other list blocks in the package file (VERTICES, CELL2D)
        self.add_block('GRIDDATA', ['TOP', 'BOT', 'AREA', 'IDOMAIN'])
        self.add_block('CONNECTIONDATA', ['IAC', 'JA', 'IHC', 'CL12', 'HWVA', 'ANGLDEGX'])

    def get_units(self, array_name: str) -> str:
        """Returns a units string.

        Args:
            array_name (str): The name of an array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'TOP' | 'BOT':
                units_str = units_util.UNITS_LENGTH
            case 'AREA':
                units_str = units_util.UNITS_AREA
            case 'IDOMAIN':
                units_str = units_util.UNITS_UNITLESS
            case 'CL12':
                units_str = units_util.UNITS_LENGTH
            case 'HWVA':
                # Thus, values in the HWVA array contain dimensions of both length and area
                units_str = f'{units_util.UNITS_LENGTH} or {units_util.UNITS_AREA}'
            case 'ANGLDEGX':
                units_str = units_util.UNITS_DEGREES
            case _:
                units_str = units_util.UNITS_UNITLESS
        return units_str

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        int_props = {'IDOMAIN', 'IAC', 'JA', 'IHC'}
        return array_name in int_props

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name: The name of an array.

        Returns:
            (bool): True or False
        """
        # ANGLDEGX is required under certain circumstances so we just say it's always required
        return array_name.upper() != 'IDOMAIN'

    @override
    def can_be_layered(self, array_name: str) -> bool:
        """Returns True if the array can have the LAYERED option.

        Args:
            array_name (str): The name of an array.

        Returns:
            (bool): True or False
        """
        return False

    @override
    def can_be_dataset(self, array_name: str) -> bool:
        """Returns True if the array can be converted to a 3d dataset.

        Args:
            array_name: The name of an array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() in ['TOP', 'BOT', 'AREA', 'IDOMAIN', 'IAC']

    @override
    def array_size_and_layers(self, array_name: str, layered: bool) -> tuple[int, int, tuple]:
        """Returns a tuple of nvalues, layers_to_read based on array_name.

        Args:
            array_name: The name of an array.
            layered: The 'LAYERED' word was found next to the array.

        Returns:
            (tuple): tuple containing:
                - nvalues (int): Number of values to read.
                - layers_to_read (int): Number of layers to read.
        """
        nja_props = {'JA', 'IHC', 'CL12', 'HWVA', 'ANGLDEGX'}
        if array_name in nja_props:
            nvalues = self.grid_info().nja
        else:
            nvalues = self.grid_info().nodes

        layers_to_read = 1
        shape = (nvalues, 1)

        return nvalues, layers_to_read, shape

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Unstructured Discretization (DISU) Package'

    def make_copy(self) -> 'DisuData':
        """Returns a copy of this class."""
        data = copy.deepcopy(self)
        data.list_blocks = dis_data_base.copy_list_blocks(self.list_blocks)
        return data

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        text = ''
        if tab_text == 'TOP':
            text = 'Top elevation for each cell in the model grid.'
        elif tab_text == 'BOT':
            text = 'Bottom elevation for each cell.'
        elif tab_text == 'AREA':
            text = 'Cell surface area (in plan view).'
        elif tab_text == 'IDOMAIN':
            text = (
                'Optional array that characterizes the existence status of a cell. If the IDOMAIN array is not'
                ' specified, then all model cells exist within the solution. If the IDOMAIN value for a cell is 0, the'
                ' cell does not exist in the simulation. Input and output values will be read and written for the cell,'
                ' but internal to the program, the cell is excluded from the solution. If the IDOMAIN value for a cell'
                ' is 1 or greater, the cell exists in the simulation. IDOMAIN values of -1 cannot be specified for the'
                ' DISU Package.'
            )
        elif tab_text == 'IAC':
            text = (
                'Number of connections (plus 1) for each cell. The sum of all the entries in IAC must be equal to NJA.'
            )
        elif tab_text == 'JA':
            text = (
                'List of cell number (n) followed by its connecting cell numbers (m) for each of the m cells connected'
                ' to cell n. The number of values to provide for cell n is IAC(n). This list is sequentially provided'
                ' for the first to the last cell. The first value in the list must be cell n itself, and the remaining'
                ' cells must be listed in an increasing order (sorted from lowest number to highest). Note that the'
                ' cell and its connections are only supplied for the GWF cells and their connections to the other GWF'
                ' cells. ... To further ease readability of the file, the node number of the cell whose connectivity is'
                ' subsequently listed, may be expressed as a negative number, the sign of which is subsequently'
                ' converted to positive by the code.'
            )
        elif tab_text == 'IHC':
            text = (
                'Index array indicating the direction between node n and all of its m connections. If IHC = 0 then cell'
                ' n and cell m are connected in the vertical direction. Cell n overlies cell m if the cell number for n'
                ' is less than m; cell m overlies cell n if the cell number for m is less than n. If IHC = 1 then cell'
                ' n and cell m are connected in the horizontal direction. If IHC = 2 then cell n and cell m are'
                ' connected in the horizontal direction, and the connection is vertically staggered. A vertically'
                ' staggered connection is one in which a cell is horizontally connected to more than one cell in a'
                ' horizontal connection.'
            )
        elif tab_text == 'CL12':
            text = (
                'Array containing connection lengths between the center of cell n and the shared face with each'
                ' adjacent m cell.'
            )
        elif tab_text == 'HWVA':
            text = (
                'Symmetric array of size NJA. For horizontal connections, entries in HWVA are the horizontal width'
                ' perpendicular to flow. For vertical connections, entries in HWVA are the vertical area for flow.'
                ' Thus, values in the HWVA array contain dimensions of both length and area. Entries in the HWVA array'
                ' have a one-to-one correspondence with the connections specified in the JA array. Likewise, there is a'
                ' one-to-one correspondence between entries in the HWVA array and entries in the IHC array, which'
                ' specifies the connection type (horizontal or vertical). Entries in the HWVA array must be symmetric;'
                ' the program will terminate with an error if the value for HWVA for an n to m connection does not'
                ' equal the value for HWVA for the corresponding n to m connection.'
            )
        elif tab_text == 'ANGLDEGX':
            text = (
                'Angle (in degrees) between the horizontal x-axis and the outward normal to the face between a cell and'
                ' its connecting cells. The angle varies between zero and 360.0 degrees, where zero degrees points in'
                ' the positive x-axis direction, and 90 degrees points in the positive y-axis direction. ANGLDEGX is'
                ' only needed if horizontal anisotropy is specified in the NPF Package, if the XT3D option is used in'
                ' the NPF Package, or if the SAVE_SPECIFIC_DISCHARGE option is specified in the NPF Package. ANGLDEGX'
                ' does not need to be specified if these conditions are not met. ANGLDEGX is of size NJA; values'
                ' specified for vertical connections and for the diagonal position are not used. Note that ANGLDEGX is'
                ' read in degrees, which is different from MODFLOW-USG, which reads a similar variable (ANGLEX) in'
                ' radians.'
            )
        return text

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        options_block = super()._setup_options()
        options_block.definitions.append(
            CheckboxField(
                'VERTICAL_OFFSET_TOLERANCE',
                brief='Tolerance used to check if top of cell is higher than bottom of overlying cell',
                type_='float',
                value=0.0,
                read_only=False
            )
        )
        options_block.definitions.append(options_util.export_array_ascii_def())
        return options_block
