"""ArrayLayerWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.file_io import io_util
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.misc import log_util


class ArrayLayerWriter:
    """Writes an ArrayLayer."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()

        self.filename = ''

    def _copy_file(self, array_layer, stress_period, nper, filename, writer_options):
        """Copies an external file to be in the new location.

        Args:
            array_layer (ArrayLayer): The array data.
            stress_period (int): The 1-based stress period.
            nper (int): Number of stress periods.
            filename (str): File path to package file being written.
            writer_options: WriterOptions object.
        """
        if writer_options.use_open_close and not writer_options.dmi_sim_dir:
            path = os.path.join(os.path.dirname(filename), writer_options.open_close_dir)
        else:
            path = os.path.dirname(filename)

        remove_temp_file = False
        if array_layer.binary:  # Binary external file
            old_filename = array_layer.external_filename
            if not Path(old_filename).is_file():
                log = log_util.get_logger()
                log.warning('Binary external file not found for array "%s"', array_layer.name)
            new_base = os.path.basename(array_layer.external_filename)
            new_filename = os.path.join(path, new_base)
        elif array_layer.temp_external_filename:
            old_filename = array_layer.temp_external_filename
            external_filename, _ = io_util.get_external_filename(stress_period, nper, filename, array_layer.name)
            new_base = os.path.basename(external_filename)
            new_filename = os.path.join(path, new_base)
            remove_temp_file = True
        else:
            old_filename = array_layer.external_filename
            new_filename, _ = io_util.get_external_filename(stress_period, nper, filename, array_layer.name)
            new_base = os.path.basename(new_filename)
            new_filename = os.path.join(path, new_base)

        array_layer.external_filename = new_filename
        if old_filename != '':
            fs.copyfile(old_filename, new_filename)
        if remove_temp_file:
            fs.removefile(old_filename)
        array_layer.temp_external_filename = ''

    def write(self, array_layer, stress_period, nper, filename, out_file, writer_options: WriterOptions):
        """Writes the array_layer to disk.

        Args:
            array_layer (ArrayLayer): The array_layer.
            stress_period (int): The 1-based stress period.
            nper (int): Number of stress periods.
            filename (str): File path to package file being written.
            out_file (_io.TextIOWrapper): The file we are writing to.
            writer_options (WriterOptions): Options for writing the simulation.
        """
        caster = io_util.type_caster_from_string(array_layer.numeric_type)

        if array_layer.binary:
            orig_use_open_close = writer_options.use_open_close
            if not writer_options.use_open_close:
                # Temporarily change the option so we can write the binary file
                writer_options.use_open_close = True
            self._copy_file(array_layer, stress_period, nper, filename, writer_options)
            writer_options.use_open_close = orig_use_open_close
            filename = fs.compute_relative_path(writer_options.mfsim_dir, array_layer.external_filename)
            out_file.write(
                f'{io_util.mftab}{io_util.mftab}OPEN/CLOSE {io_util.quote(filename)} FACTOR'
                f' {str(caster(array_layer.factor))} (BINARY)\n'
            )
        elif array_layer.storage == 'CONSTANT':
            out_file.write(f'{io_util.mftab}{io_util.mftab}CONSTANT {str(caster(array_layer.constant))}\n')
        elif array_layer.storage == 'ARRAY':
            if writer_options.use_open_close:
                if not array_layer.temp_external_filename and not array_layer.external_filename and array_layer.values:
                    ArrayLayerWriter.write_array_layer(array_layer, array_layer.shape)  # Save values in RAM to disk
                self._copy_file(array_layer, stress_period, nper, filename, writer_options)
                filename = fs.compute_relative_path(writer_options.mfsim_dir, array_layer.external_filename)
                out_file.write(
                    f'{io_util.mftab}{io_util.mftab}OPEN/CLOSE {io_util.quote(filename)} FACTOR'
                    f' {str(caster(array_layer.factor))}\n'
                )
            else:
                out_file.write(f'{io_util.mftab}{io_util.mftab}INTERNAL FACTOR {str(caster(array_layer.factor))}\n')
                if array_layer.temp_external_filename:
                    io_util.write_file_internal(out_file, array_layer.temp_external_filename)
                elif array_layer.external_filename:
                    io_util.write_file_internal(out_file, array_layer.external_filename)
                else:  # This should only happen during testing
                    io_util.write_array_values(array_layer.values, array_layer.shape, out_file)

    @staticmethod
    def write_array_layer(array_layer: ArrayLayer, shape, temp_file_list=None):
        """Writes array_layer values in RAM to external files.

        Used when importing and editing. Values in RAM are deleted.

        Args:
            array_layer: The array_layer.
            shape: Array shape.
            temp_file_list (list of str): list of temporary file names
        """
        if array_layer.binary:
            return

        if array_layer.storage != 'ARRAY':
            log = log_util.get_logger()
            log.info('Do not call "array_layer_writer.write_array_layer" unless storage == "ARRAY"')
            return

        if array_layer.values is not None:
            temp_filename = io_util.get_temp_filename()
            with open(temp_filename, mode='w') as temp:
                io_util.write_array_values(array_layer.values, shape, temp)

            # Remove old temp file if one exists and update with new temp filename
            if array_layer.temp_external_filename:
                os.remove(array_layer.temp_external_filename)
            array_layer.temp_external_filename = temp_filename
            if temp_file_list is not None:
                temp_file_list.append(temp_filename)
            array_layer.values = None
