"""ArrayPackageWriter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.array_layer_writer import ArrayLayerWriter
from xms.mf6.file_io.package_writer_base import PackageWriterBase


class ArrayPackageWriter(PackageWriterBase):
    """Writes an array-based package file (RCH, EVT)."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._tab_names = []  # Will become ArrayPackageData._griddata_names + aux

    def _write_stress_periods(self, fp):
        """Writes all the stress period blocks.

        Args:
            fp (_io.TextIOWrapper): The file.
        """
        if not self._tab_names:
            self._tab_names = self._data.tab_names()

        array_names = self._tab_names
        nper = self._data.nper()
        for sp in self._data.period_data.keys():
            fp.write('\n')
            fp.write('BEGIN PERIOD {}\n'.format(sp))

            # Go through properties in order
            for array_name in array_names:
                array = self._data.get_array(sp, array_name)
                if array and array.storage != 'UNDEFINED':
                    fp.write(f'{io_util.mftab}{array.array_name}')
                    if array.storage == 'TIME-ARRAY SERIES':
                        fp.write(f' TIMEARRAYSERIES {array.time_array_series}\n')
                    else:
                        fp.write('\n')
                        writer = ArrayLayerWriter()
                        writer.write(array.layer(0), sp, nper, self._data.filename, fp, self._writer_options)
            fp.write('END PERIOD\n')

    def _write_package(self, data):
        """Writes the package file.

        Saves array data in external files using the OPEN/CLOSE option.
        """
        self._data = data
        with open(self._data.filename, 'w') as fp:
            self._write_comments(fp)
            self._write_options(fp)
            self._write_stress_periods(fp)
