"""OcReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import shlex
import sys

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import oc_data
from xms.mf6.data.oc_data import OcPresetOutputEnum
from xms.mf6.file_io import io_util
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.misc.settings import Settings


class OcReader(PackageReader):
    """Reads an OC package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='OC6')

    @staticmethod
    def dependent_variable_and_budget_files_from_oc_file(oc_filename: str | Path, model_ftype: str):
        """Reads part of the OC file and returns the dependent variable (head, concentration) and budget filenames.

        Args:
            oc_filename: File path of OC file.
            model_ftype: Ftype of the model ('GWF6', 'GWT6').

        Returns:
            (tuple): tuple containing:
                - dependent_variable_filename (str): The head or concentration filename, or '' if none specified.
                - budget_filename (str): The budget filename, or '' if none specified.
        """
        word_up = oc_data.oc_first_word(model_ftype)
        dv_filename = ''
        budget_filename = ''
        with open(oc_filename, 'r') as file:
            for line in file:
                if io_util.is_begin_x_line(line, 'OPTIONS'):
                    for line2 in file:
                        line2.strip()
                        words = shlex.split(line2, posix="win" not in sys.platform)
                        if words[1].upper() == 'FILEOUT':
                            if not dv_filename and words[0].upper() == word_up:
                                dv_filename = words[2]
                                if budget_filename:
                                    break
                            elif not budget_filename and words[0].upper() == 'BUDGET':
                                budget_filename = words[2]
                                if dv_filename:
                                    break
                        elif io_util.is_end_line(line2):
                            break
                    break

        return dv_filename, budget_filename

    def _read_settings(self):
        """Reads the package settings file if it exists."""
        super()._read_settings()
        if Path(Settings.settings_filename(self._data.filename)).exists():
            settings = Settings.read_settings(self._data.filename)
            self._data.preset_output = settings.get('preset_output', OcPresetOutputEnum.OC_USER_SPECIFIED)
