"""RchAndEvtReaderBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.gwf.array_package_reader import ArrayPackageReader
from xms.mf6.file_io.package_reader import PackageReader
from xms.mf6.file_io.package_reader_base import PackageReaderBase


class RchAndEvtReaderBase(PackageReaderBase):
    """Base class for RchReader and EvtReader."""
    def __init__(self):
        """Initializes the class."""
        super().__init__()
        self._array_data = None  # Gets set in derived classes: EvtArrayData, RchArrayData
        self._list_data = None  # Gets set in derived classes: EvtListData, RchListData

    def read(self, filename, **kwargs):
        """Reads the file.

        Args:
            filename (str): Path to file.
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis.

        Returns:
            The package data class.
        """
        array_based = io_util.file_has_readasarrays(filename)
        if array_based or kwargs.get('array_based', False):
            reader = ArrayPackageReader()
            kwargs['array_based'] = True
        else:
            reader = PackageReader(ftype=self.ftype)
        reader.ftype = self.ftype

        self._data = reader.read(filename, **kwargs)

        return self._data
