"""ObsReader class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shlex
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.file_io import io_util
from xms.mf6.file_io.package_reader import PackageReader


class ObsReader(PackageReader):
    """Reads an OBS package file."""
    def __init__(self):
        """Initializes the class."""
        super().__init__(ftype='OBS6')

    def _read_continuous(self, line):
        """Read the CONTINUOUS block from the file.

        Args:
            line (str): line from the package file
        """
        words = shlex.split(self._block_begin_line, posix="win" not in sys.platform)
        if len(words) < 4:
            self.invalid_line_in_file(line)
            return

        filename = os.path.basename(words[3])
        block_name = 'CONTINUOUS'
        external_file = self._read_list_block(line, block_name)
        # self._data.list_blocks[filename] = external_file
        if self._data.mfsim:  # This can be None in unit tests
            self._data.list_blocks[filename] = fs.resolve_relative_path(self._data.mfsim.dir_(), external_file)
        return True

    # @overrides
    def join_line(self, words, column_count, separator=io_util.mfsep):
        """Joins the words into a line and returns the line.

        Can be overridden.

        Args:
            words (list[str]): The words.
            column_count (int): Number of columns that should be written for this line.

        Returns:
            (str): The line.
        """
        sep = ','  # separator
        id_count = 1
        obstype = words[1].lower()
        # See Tables 30 and 31
        if obstype in {
            'conductance', 'delay-compaction', 'delay-estress', 'delay-gstress', 'delay-head', 'delay-preconstress',
            'delay-thickness', 'delay-theta', 'flow-ja-face', 'lak', 'lkt', 'maw', 'mwt', 'sft', 'uzt', 'water-content',
            'wetted-area'
        }:
            id_count = 2

        id = ''
        id2 = ''
        if id_count == 1:
            id = f"{' '.join([str(x) for x in words[2:]])}"
        else:
            # Figure out how to read the columns
            id_type = 1
            if obstype == 'flow-ja-face':
                ftype = self._data.parent_package_ftype
                if ftype in {'LKT6', 'SFT6', 'UZT'}:
                    id_type = 1  # pragma no cover
                else:
                    id_type = 0
            elif obstype in {
                'delay-head', 'delay-gstress', 'delay-estress', 'delay-preconstress', 'delay-compaction',
                'delay-thickness', 'delay-theta', 'lak', 'wetted-area', 'conductance', 'lkt', 'maw', 'mwt', 'sft',
                'uzt', 'water-content'
            }:
                id_type = 1

            # Read the columns
            if id_type == 0:
                # 'flow-ja-face': id=cellid, id2=cellid
                cellid_len = self._data.grid_info().cell_id_column_count()
                id = io_util.mfsep.join(words[2:2 + cellid_len])
                id2 = io_util.mfsep.join(words[2 + cellid_len:2 + (2 * cellid_len)])
            elif id_type == 1:
                # id=icsubno id2=idcellno
                id = words[2]
                if len(words) > 3:
                    id2 = io_util.mfsep.join(words[3:])

        record = f'{words[0]}{sep}{words[1]}{sep}{id}{sep}{id2}'

        return record
