"""PackageWriterBase class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import typing

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.block import Block
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.array_layer_writer import ArrayLayerWriter
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.misc import log_util


class PackageWriterBase:
    """Base class for package reader classes."""
    def __init__(self):
        """Initializes the class."""
        self._data = None  # A package data class.
        self._writer_options = WriterOptions()
        self._log = log_util.get_logger()
        self._write_log_message = True

    def _write_comments(self, fp: typing.TextIO) -> None:
        """Writes the comments at the top of the file.

        Args:
            fp: The file.
        """
        for comment in self._data.comments:
            fp.write('# {}\n'.format(comment))
        if self._data.comments:
            fp.write('\n')

    def _write_option(self, key, fp: typing.TextIO, options_dict) -> None:
        """Writes the options block.

        Args:
            key: The key.
            fp: The file.
            options_dict (dict): Optional dict containing the options.
        """
        if key.find('FILEIN') > -1:
            words = key.split()
            files = options_dict.get(key, [])
            self.write_file_list(files, words[0], words[1], fp, self._data.filename, self._writer_options)
        elif key.find('FILEOUT') > -1:
            words = key.split()
            file = options_dict.get(key, '')
            self.write_file_list([file], words[0], words[1], fp, self._data.filename, self._writer_options)
        else:
            io_util.write_option(options_dict, key, fp)

    def _write_options(self, fp: typing.TextIO) -> None:
        """Writes the options block.

        Args:
            fp: The file.
        """
        fp.write('BEGIN OPTIONS\n')
        for key, _ in self._data.options_block.dict().items():
            self._write_option(key, fp, self._data.options_block.dict())
        fp.write('END OPTIONS\n')

    def _compute_external_file_name(self, stress_period):
        """Given the stress period, computes and returns the external filename.

        Filename is of the form "<list_package.ext>_0001.txt"

        Args:
            stress_period (int): Stress period number (1-based).

        Returns:
            (str): The filename.
        """
        nper = self._data.nper()
        full_filename, _ = io_util.get_external_filename(stress_period, nper, self._data.filename, '')
        return full_filename

    def _copy_data_file(self, block_name, filename, filename_append):
        """Copies a file.

        Args:
            block_name (str): Name of the block.
            filename (str):
            filename_append (str): What to append to filename?

        Returns:
            (str): The new filename.
        """
        from_filename, to_filename = self._get_from_and_to_filenames(filename, filename_append)

        # Copy the file if the new and current paths are different
        fs.copyfile(from_filename, to_filename)

        return to_filename

    def _get_from_and_to_filenames(self, filename, filename_append):
        """Returns the from and to filenames to copy.

        Args:
            filename (str):
            filename_append (str): What to append to filename?

        Returns:
            (tuple(str, str)): 'from' and 'to' filenames
        """
        file_basename = f'{os.path.basename(self._data.filename)}{filename_append}'
        package_dir = os.path.dirname(self._data.filename)
        if package_dir == self._writer_options.open_close_dir:
            new_filename = os.path.join(package_dir, file_basename)
        else:
            if self._writer_options.dmi_sim_dir or not self._writer_options.open_close_dir:
                new_filename = os.path.join(package_dir, file_basename)
            else:
                new_filename = os.path.join(self._writer_options.open_close_dir, file_basename)
        file_dir = os.path.dirname(filename)
        old_base = os.path.basename(filename)
        # old_filename = os.path.join(fs.resolve_relative_path(package_dir, file_dir), old_base)
        old_filename = os.path.join(fs.resolve_relative_path(self._writer_options.mfsim_dir, file_dir), old_base)
        return old_filename, new_filename

    def _block_supports_open_close(self, block_name):
        """Returns True if the block information can be contained in a separate file.

        See mf6io.pdf Table A-1. Also, although that table shows GWT SSM SOURCES as supported, in reality MODFLOW
        doesn't support it. That may be a bug that they fix but it was that way in 6.3.0.

        That bug was fixed in 6.3.0.

        Args:
            block_name (str): Name of the block.

        Returns:
            (bool): See description.
        """
        # return block_name.upper() not in {'SOURCES'}
        return True

    def _write_list_block_extended(self, block_name, block_key, block_name_extra, fp: typing.TextIO) -> None:
        """Write a list to the file.

        Args:
            block_name (str): Name of the block.
            block_key (str): Key into self._data.list_blocks dict.
            block_name_extra (str): Stuff to follow the block name (number, FILEOUT etc)
            fp: The file.
        """
        if block_key not in self._data.list_blocks:
            return

        filename = self._data.list_blocks[block_key]
        fp.write('\n')
        if block_name_extra:
            block_name_extra = ' ' + block_name_extra
        fp.write(f'BEGIN {block_name}{block_name_extra}\n')
        if filename:
            if self._writer_options.use_open_close and self._block_supports_open_close(block_name):
                old_filename = filename
                block_key_strip = block_key.strip('\'')
                filename = self._copy_data_file(block_name, filename, f'_{block_key_strip}.txt')
                if old_filename != filename and io_util.is_temporary_file(old_filename):
                    os.remove(old_filename)
                self._data.list_blocks[block_key] = filename
                filename = fs.compute_relative_path(self._writer_options.mfsim_dir, filename)
                if filename.find(' ') > -1:
                    filename = f'\'{filename}\''
                fp.write(f'{io_util.mftab}OPEN/CLOSE {io_util.quote(filename)}\n')
            else:
                io_util.write_file_internal(fp, filename)
        fp.write(f'END {block_name}\n')

    def _write_list_block(self, block_name, fp: typing.TextIO) -> None:
        """Write a list to the file.

        Args:
            block_name (str): Name of the block.
            fp: The file.
        """
        self._write_list_block_extended(block_name, block_name.upper(), '', fp)

    def _write_griddata(self, fp: typing.TextIO) -> None:
        """Writes the griddata block.

        Args:
            fp: The file.
        """
        self._write_block(fp, self._data.block('GRIDDATA'))

    def _write_block(self, fp: typing.TextIO, block: Block) -> None:
        """Writes array properties to a block in a file.

        Args:
            fp: the file
            block: dictionary of array properties
        """
        fp.write('\n')
        block_name = block.name
        fp.write(f'BEGIN {block_name}\n')
        nper = self._data.nper()

        # Go in order of block.names
        for name in block.names:
            array = block.array(name)
            if not array or not array.defined:
                continue

            layered_str = '' if not array.layered else ' LAYERED'
            fp.write(f'{io_util.mftab}{array.array_name}{layered_str}\n')
            sp = -1
            for index, array_layer in enumerate(array.layers):
                array_layer_writer = ArrayLayerWriter()
                if array.layered:
                    array_layer.name = f'{array.array_name}_{index + 1}'
                array_layer_writer.write(array_layer, sp, nper, self._data.filename, fp, self._writer_options)

        fp.write(f'END {block_name}\n')

    def _create_unique_dir(self, package, base_dir, prefix='', running_io_tests=False, random=None):
        """Creates a directory in base_dir with a unique name with optional prefix.

        Args:
            package: The package data.
            base_dir (str): Directory where the new directory will be created.
            prefix (str): Becomes the starting string in the new directory name.
            running_io_tests: True if we're running tests.
            random: Random number generator.

        Returns:
            Path to the directory.
        """
        uuid_str = data_util.new_uuid(package, running_io_tests, random)
        path = os.path.join(base_dir, uuid_str)
        os.makedirs(path)
        return path

    def _get_new_filename(self, package, auto_name=''):
        """Sets the package filepath before writing to it.

        If self._writer_options.dmi_sim_dir is not empty, creates a directory for the package.

        Args:
            package: The package data.
            auto_name (str): The basename GMS is automatically giving it, if we are.

        Returns:
            The new package filepath.
        """
        filepath = package.filename
        # Create a separate directory for the file
        if self._writer_options.dmi_sim_dir:
            if (
                hasattr(self._data, 'gui_edit_active') and self._data.gui_edit_active
            ) or self._writer_options.just_name_file:
                dir_ = os.path.dirname(filepath)
            else:
                opts = self._writer_options
                dir_ = self._create_unique_dir(package, opts.dmi_sim_dir, '', opts.running_io_tests, opts._random)
        else:
            dir_ = os.path.dirname(self._data.filename)

        # Use fname if exporting, otherwise just save it to the component main_file.
        if self._writer_options.dmi_sim_dir:
            new_filepath = os.path.join(dir_, os.path.basename(filepath))
        elif auto_name:
            new_filepath = os.path.join(dir_, auto_name)
        else:
            new_filepath = os.path.join(dir_, os.path.basename(package.fname))

        if self._writer_options.dmi_sim_dir and not self._writer_options.just_name_file:
            WriterOptions._dmi_file_list.append(new_filepath)

        # Get path relative to mfsim.nam location
        relative_path = ''
        if self._writer_options.mfsim_dir:
            relative_path = fs.compute_relative_path(self._writer_options.mfsim_dir, new_filepath)
        return new_filepath, relative_path

    def write_file_list(self, file_list, card1, card2, fp: typing.TextIO, filename, writer_options, lake_id=-1) -> None:
        """Used with TS6 and OBS6 and similar file lists to write the filenames.

        Args:
            file_list (list of str): List of files.
            card1 (str): The 1st card to write to the file (i.e. 'TS6', 'OBS6')
            card2 (str): The 2nd card to write to the file (i.e. 'FILEIN', 'FILEOUT')
            fp: The file.
            filename (str): Filepath of file we are writing to.
            writer_options (WriterOptions): Options for writing the simulation.
            lake_id (int): id for the lake associated with a TAB6 file
        """
        # OBS6 holds the obs package class not just a filename. So we get the file names and process them
        # and then we write the obs files
        if not file_list:
            self._log.info('File list not defined. Do not call write_file_list.')
            return
        try:
            orig_list = file_list.copy()
            for i, list_filename in enumerate(file_list):
                base_name = os.path.basename(list_filename)
                package_path = os.path.dirname(filename)
                new_path = package_path
                if card2.upper() != 'FILEOUT' and writer_options.use_open_close and not writer_options.dmi_sim_dir:
                    # Make name unique using package name because it will go in 'input' folder with others
                    if os.path.basename(filename) not in base_name:
                        base_name = f'{os.path.basename(filename)}_{base_name}'
                    new_path = os.path.join(new_path, writer_options.open_close_dir)
                if card2.upper() == 'FILEOUT' and writer_options.use_output_dir:
                    new_path = os.path.join(new_path, writer_options.output_dir)
                new_filepath = os.path.join(new_path, base_name)
                if card2.upper() != 'FILEOUT' and os.path.isfile(list_filename):
                    fs.copyfile(list_filename, new_filepath)

                relative_path = fs.compute_relative_path(writer_options.mfsim_dir, new_filepath)
                if ' ' in relative_path:
                    relative_path = f'\'{relative_path}\''
                card1_out = card1
                if lake_id > -1:
                    card1_out = f'{lake_id} {card1}'
                fp.write(f'{io_util.mftab}{card1_out} {card2} {relative_path}\n')
                file_list[i] = new_filepath

            card_upper = card1.upper()
            if card_upper in {'OBS6', 'TAS6', 'TS6', 'TAB6', 'ATS6', 'TVA6'}:
                for i, item in enumerate(orig_list):
                    reader = io_factory.reader_from_ftype(card_upper)
                    if not Path(item).is_file():
                        self._log.error(f'File not found: {card1.upper()} {card2.upper()} {item}')
                        continue
                    data = reader.read(
                        item, mfsim=self._data.mfsim, model=self._data.model, parent_package_ftype=self._data.ftype
                    )
                    data.filename = file_list[i]
                    writer = io_factory.writer_from_ftype(card_upper, writer_options)
                    writer.write(data)
                file_list[:] = orig_list.copy()
        except Exception as error:
            self._log.info(f'Error occurred: {error}.')

    def write_settings(self, data):
        """If writing to components area, save fname to a settings file.

        Args:
            data: The package data.
        """
        pass

    def _write_package(self, data):
        """Writes the package.

        You should override this method.

        Args:
            data: Something derived from BaseFileData
        """
        raise NotImplementedError()

    def _write_gms_options(self, data) -> None:
        """Writes the persistent data file.

        Args:
            data: Something derived from BaseFileData
        """
        # Don't write when exporting
        if not data.gms_options or not self._writer_options.dmi_sim_dir:
            return

        parent_dir = Path(data.filename).parent
        io_util.write_gms_options(parent_dir, data.gms_options)

    def write(self, data):
        """Writes the package.

        Args:
            data: Package data class.
        """
        try:
            if self._write_log_message:
                self._log.info(f'Writing \"{data.filename}\"')
            self._write_package(data)
            self.write_settings(data)
            self._write_gms_options(data)
        except Exception as error:
            raise RuntimeError(f'Error writing {data.filename}: {str(error)}')
