"""Geometry functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.grid.geometry import geometry  # Some functions will be moved to here in later versions

# 4. Local modules


def polyline_stationing(polyline):
    """Returns a list of the distances along the polyline of each point, starting at 0.0 for the first point.

    From gmPolyLineStationing().

    Args:
        polyline (list): List of x,y points defining the polyline.

    Returns:
        (list): See description.
    """
    stations = [0.0] if polyline else []
    length = 0.0
    for i in range(1, len(polyline)):
        length += geometry.distance_2d(polyline[i - 1], polyline[i])
        stations.append(length)
    return stations


def distance_2d(point1, point2):
    """Wraps geometry.distance_2d which seems to move around.

    Args:
        point1: First point.
        point2: Second point.

    Returns:
        (float): Distance between the points.
    """
    return geometry.distance_2d(point1, point2)


def point_in_polygon_2d(polygon, point):
    """Wraps geometry.point_in_polygon_2d which seems to move around.

    See gmPointInPolygon2D()

    Args:
        polygon: The polygon.
        point: The point.

    Returns:
        (int): 1 = in, -1 = out, 0 = on. (What geometry.point_in_polygon_2d returns).
    """
    return geometry.point_in_polygon_2d(polygon, point)


def on_line_and_between_endpoints_2d(line_point1, line_point2, point, tol):
    """Wraps geometry.on_line_and_between_endpoints_2d which seems to move around.

    See gmOnLineAndBetweenEndpoints()

    Args:
        line_point1: First line endpoint.
        line_point2: Second line endpoint.
        point: Point to test.
        tol: Tolerance

    Returns:
        Whatever geometry.on_line_and_between_endpoints_2d returns.
    """
    return geometry.on_line_and_between_endpoints_2d(line_point1, line_point2, point, tol)


def normal_xyz(vector):
    """Returns the vector as normalized (all values between 0.0 and 1.0).

    See gmNormalizeXYZ().

    Args:
        vector: The list-like vector.

    Returns:
        (list): See description.
    """
    magnitude = math.sqrt(vector[0]**2 + vector[1]**2 + vector[2]**2)
    if math.isclose(magnitude, 0.0):
        return [0.0, 0.0, 0.0]
    one_over_magnitude = 1.0 / magnitude
    return [v * one_over_magnitude for v in vector]


def normalize_xyz(vector):
    """Returns the vector as normalized (all values between 0.0 and 1.0).

    See gmNormalizeXYZ().

    Args:
        vector: The list-like vector.

    Returns:
        (list): See description.
    """
    magnitude = math.sqrt(vector[0]**2 + vector[1]**2 + vector[2]**2)
    if math.isclose(magnitude, 0.0):
        vector[0] = 0.0
        vector[1] = 0.0
        vector[2] = 0.0
        return
    one_over_magnitude = 1.0 / magnitude
    vector[0] *= one_over_magnitude
    vector[1] *= one_over_magnitude
    vector[2] *= one_over_magnitude


def vector_add(vector_1, vector_2):
    """Returns a vector that is the sum of the two vectors, element-wise.

    See gmVectorAdd().

    Args:
        vector_1: First vector
        vector_2: Second vector

    Returns:
        See description.
    """
    # map might be slightly faster but vectors are usually size 3 (xyz) and the list comprehension is more readable
    # If we have very big vectors, we should not use this but use numpy. https://stackoverflow.com/questions/18713321
    # return list(map(operator.add, vector_1, vector_2))
    return [a + b for a, b in zip(vector_1, vector_2)]
