"""TreeButtonDelegate class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, Qt
from PySide2.QtGui import QBrush, QColor, QPixmap
from PySide2.QtWidgets import QDialog, QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg

# 4. Local modules
from xms.mf6.gui import gui_util


class TreeButtonDelegate(QStyledItemDelegate):
    """Qt delegate for a tree item selector button."""
    def __init__(self, dialog_title, pe_tree, parent=None, query=None, multiple=False):
        """Initializes the class.

        Args:
            dialog_title (str): Title for the dialog.
            pe_tree (TreeNode): The project explorer tree.
            parent (QObject): The parent object.
            query (xms.api.dmi.Query): Object for communicating with GMS. If not None, the
                dialog that appears will allow for selecting time steps (query is needed for that).
            multiple (bool): If true, multiple items can be selected.
        """
        super().__init__(parent)
        self.dialog_title = dialog_title
        self.pe_tree = pe_tree
        self.parent_dlg = parent
        self.query = query
        self.multiple = multiple

        self.uuid_to_tree_name = {}

        self._fill_uuid_to_tree_name(self.pe_tree)

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton()
            uuid_data = index.data()
            btn.setText('(none selected)')
            if uuid_data:
                uuids = uuid_data.split()
                text = ' '.join([self.uuid_to_tree_name.get(tree_uuid, '') for tree_uuid in uuids])
                btn.setText(text)
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())
            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            painter.fillRect(option.rect, QColor(240, 240, 240))

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the XY series editor button is clicked.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                tree_uuid = index.data()
                dialog = TreeItemSelectorDlg(
                    title=self.dialog_title,
                    target_type=ComponentItem,
                    pe_tree=self.pe_tree,
                    previous_selection=tree_uuid,
                    override_icon=gui_util.get_icon_path,
                    parent=self.parent_dlg,
                    allow_multi_select=self.multiple
                )
                if dialog.exec() == QDialog.Accepted:
                    uuid_data = dialog.get_selected_item_uuid()
                    if isinstance(uuid_data, list):
                        uuid_data = ' '.join(uuid_data)
                    model.setData(index, uuid_data)
                return True
        return super().editorEvent(event, model, option, index)

    def _fill_uuid_to_tree_name(self, node):
        """Fills in a dictionary of UUIDs to names in the project explorer tree.

        Args:
            node (TreeNode): The current node in the project explorer tree.
        """
        if not node:
            return
        if node.uuid:
            self.uuid_to_tree_name[node.uuid] = node.name
        for child in node.children:
            self._fill_uuid_to_tree_name(child)
