"""DisDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtGui import QTextCursor, QTextOption
from PySide2.QtWidgets import QPlainTextEdit
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.components.idomain_mapper_runner import IdomainMapperRunner
from xms.mf6.gui.griddata_dialog import GriddataDialog

_MAX_LINES_DISPLAYED = 100  # Max number of lines to display


class DisDialogBase(GriddataDialog):
    """A base class dialog used for the DIS* package."""
    def __init__(self, dlg_input, read_only_tabs: list[str] | None = None, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            read_only_tabs: List of tabs that should be read only.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, read_only_tabs, parent)

    @override
    def _setup_base_ui(self):
        """Set up the UI for the base dialog - things that only should be done once - not for the dynamic sections."""
        super()._setup_base_ui()

        # Signals
        self.ui.btn_activate_cells.clicked.connect(self._on_btn_activate_cells)

        # Enable stuff
        locked = self.dlg_input.locked
        self.ui.btn_activate_cells.setVisible(not locked)

    def _on_btn_activate_cells(self) -> None:
        """Set IDOMAIN array based on polygons."""
        runner = IdomainMapperRunner(self)
        if runner.run_with_feedback_dialog():
            self.dlg_input.restore_on_cancel = True
            self.griddata_widget.load_arrays()
            self.griddata_widget.flag_array_as_modified('IDOMAIN')
            msg = 'IDOMAIN updated.'
            message_box.message_with_ok(parent=self, message=msg, icon='Information')


def create_plain_text_edit() -> QPlainTextEdit:
    """Create the text edit field."""
    w = QPlainTextEdit()
    w.setReadOnly(True)
    w.setWordWrapMode(QTextOption.NoWrap)  # This is supposed to make it faster (but doesn't seem to)
    w.setLineWrapMode(QPlainTextEdit.NoWrap)
    return w


def fill_edt_with_file(
    file_name: Pathlike, widget: QPlainTextEdit, header: str, max_lines: int = _MAX_LINES_DISPLAYED
) -> None:
    """Fills a plain text widget with the contents of a file.

    See widget_builder._fill_edt_with_file(). This one doesn't replace spaces with tabs, and only reads some lines.

    Args:
        file_name: file name
        widget: text edit
        header: header for the data in the edit
        max_lines: Maximum number of lines from the file to display.
    """
    widget.setReadOnly(True)
    widget.appendPlainText(header)
    if file_name and Path(file_name).is_file():
        text = ''
        with open(file_name, 'r') as file:
            for i, line in enumerate(file):
                if i >= max_lines:
                    break
                text += line
        widget.appendPlainText(text)
        if i == max_lines:
            widget.appendPlainText(f'[Only first {max_lines} lines displayed]')
    widget.moveCursor(QTextCursor.Start)
