"""GwxGwxDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog import PackageDialogBase
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget


class GwxGwxDialogBase(PackageDialogBase):
    """A dialog used for the GWF-GWF exchange."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'EXCHANGEDATA']
        self.default_sections = ['EXCHANGEDATA']

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'EXCHANGEDATA':
            self.setup_exchangedata_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The vertical layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def change_columns(self, use_aux):
        """Changes a table's columns given the list of auxiliary variables.

        Args:
            use_aux (bool): True to include AUXILIARY variables.
        """
        if self.sp_widget:
            self.sp_widget.change_aux_variables(block='', use_aux=use_aux)

    def setup_exchangedata_section(self):
        """Sets up the EXCHANGEDATA section."""
        section = 'EXCHANGEDATA'
        self.add_group_box_to_scroll_area(section)

        # Table
        w = self.uix[section]['tbl_wgt'] = ListBlockTableWidget(self, self.dlg_input, '')
        self.uix[section]['layout'].addWidget(w)

        self.get_table_widget(section).load_list_block(section)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.uix['EXCHANGEDATA']['tbl_wgt']

    def _get_aux_widget(self):
        """Returns the widget that includes aux columns (PeriodListWidget or ListBlockTableWidget).
        """
        return self.get_table_widget('EXCHANGEDATA')

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.save_list_blocks_to_temp()
