"""SsmDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QPushButton
from typing_extensions import override

# 3. Aquaveo modules
from xms.guipy.dialogs import message_box
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules
from xms.mf6.gui.gwt.flow_model_data_finder import FlowModelDataFinder
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase
from xms.mf6.gui.widgets.list_block_table_widget import ListBlockTableWidget


class SsmDialog(PackageDialogBase):
    """The SSM package dialog."""
    def __init__(self, dlg_input, parent=None):
        """Initializes the dialog.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)

        self.setup_ui()

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections, and self.default_sections should be set here.
        """
        self.sections = ['COMMENTS', 'OPTIONS', 'SOURCES']
        self.default_sections = ['SOURCES']

    def setup_section(self, section_name):
        """Sets up a section of widgets.

        Args:
            section_name (str): name of the section
        """
        if section_name == 'SOURCES':
            self.setup_sources_section()
        else:
            super().setup_section(section_name)

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_sources_section(self):
        """Sets up the SOURCES section."""
        section = 'SOURCES'
        self.add_group_box_to_scroll_area(section)

        # Table
        w = self.uix[section]['tbl_wgt'] = ListBlockTableWidget(self, self.dlg_input, '')
        self.uix[section]['layout'].addWidget(w)
        self.get_table_widget(section).load_list_block(section)

        # Button
        w = self.uix[section]['btn_setup_from_flow_model'] = QPushButton('Set Up From Flow Model')
        w.clicked.connect(self.on_btn_setup_from_flow_model)
        self.uix[section]['layout'].addWidget(w)
        w.setEnabled(not self.dlg_input.locked)

    def on_btn_setup_from_flow_model(self):
        """Called when the 'Setup From Flow Model' button is clicked."""
        finder = FlowModelDataFinder('GWF6', self.dlg_input, self)
        sources_data = finder.find_data()
        if sources_data is None:
            message_box.message_with_ok(parent=self, message=f'No {finder.flow_ftype} found.')
            return
        if not sources_data:  # User hit cancel or didn't select anything
            return
        if 'pnames' not in sources_data or not sources_data['pnames']:
            message = 'No flow packages with an AUXILIARY variable named "CONCENTRATION" found.'
            message_box.message_with_ok(parent=self, message=message)
            return

        self._update_model(sources_data)

    def _update_model(self, sources_data):
        """Updates the model for the table.

        Args:
            sources_data (dict): The pname, srctype, and auxname lists.
        """
        section = 'SOURCES'
        table_widget = self.get_table_widget(section)
        data_frame = pd.DataFrame(
            {
                'PNAME': sources_data['pnames'],
                'SRCTYPE': sources_data['srctypes'],
                'AUXNAME': sources_data['auxnames']
            }
        )
        data_frame.index += 1
        model = QxPandasTableModel(data_frame)
        table_widget.set_model(model, section)

    # @overrides
    def get_table_widget(self, block):
        """Returns the table view from the given block.

        Args:
            block: The block.
        """
        return self.uix[block]['tbl_wgt']

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.save_list_blocks_to_temp()
