"""GeneratePestObsDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QListWidget

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.file_io.pest.pest_obs_util import ObsCovData
from xms.mf6.gui import gui_util
from xms.mf6.gui.pest.generate_pest_obs_dialog_ui import Ui_GeneratePestObsDialog


class GeneratePestObsDialog(XmsDlg):
    """A dialog that appears when creating a new simulation."""
    def __init__(self, model_ftype: str, dep_var_cov_data: ObsCovData, flow_cov_data: ObsCovData, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            model_ftype: 'GWF6', 'GWT6' etc.
            dep_var_cov_data: Dict of dependent variable (head, concentration etc.) obs coverages and their att files.
            flow_cov_data: Dict of flow observation coverages and their att files.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.mf6.gui.generate_pest_obs_dialog')
        self._model_ftype = model_ftype
        self._dep_var_cov_data = dep_var_cov_data
        self._flow_cov_data = flow_cov_data

        self.ui = Ui_GeneratePestObsDialog()
        self.ui.setupUi(self)

        self.ui.buttonBox.helpRequested.connect(self.help_requested)

        self.help_getter = gui_util.help_getter('xms.mf6.gui.pest.generate_pest_obs_dialog')
        self._setup_dep_var_list(dep_var_cov_data)
        self._setup_flow_list(flow_cov_data)
        self.ui.spn_nearest.setValue(4)

    def _setup_dep_var_list(self, cov_data: ObsCovData) -> None:
        """Set up the dependent variable (head, concentration, or temperature) coverage list.

        Args:
            cov_data: Dict of dependent variable observation coverages and their att files.
        """
        # By default, the text says "Head observation coverages:"
        if self._model_ftype == 'GWT6':
            self.ui.txt_dep_var.setText('Concentration observation coverages:')
        elif self._model_ftype == 'GWE6':
            self.ui.txt_dep_var.setText('Temperature observation coverages:')
        _add_coverages_to_list_widget(cov_data, self.ui.lst_dep_var)

    def _setup_flow_list(self, cov_data: ObsCovData) -> None:
        """Set up the Flow list.

        Args:
            cov_data: Dict of flow observation coverages and their att files.
        """
        if self._model_ftype != 'GWF6':
            self.ui.lst_flow.setVisible(False)
            self.ui.txt_flow.setVisible(False)
            return
        _add_coverages_to_list_widget(cov_data, self.ui.lst_flow)

    def get_coverage_data(self, which_list: str) -> ObsCovData:
        """Returns a dict of the checked coverage tree items and their list of att files.

        Args:
            which_list: 'dep_var' or 'flow'.

        Returns:
            (ObsCovData): See description.
        """
        if which_list not in {'dep_var', 'flow'}:
            raise ValueError('Error getting coverage data.')
        list_widget = self.ui.lst_dep_var if which_list == 'dep_var' else self.ui.lst_flow
        cov_data = self._dep_var_cov_data if which_list == 'dep_var' else self._flow_cov_data
        selected_data = {}
        for row, key in enumerate(cov_data.keys()):
            list_item = list_widget.item(row)
            if list_item.checkState() == Qt.Checked:
                selected_data[key] = cov_data[key]
        return selected_data

    def nearest_n_points(self):
        """Returns the value in the nearest points field.

        Returns:
            (int): See description.
        """
        return self.ui.spn_nearest.value()

    def accept(self):
        """Called when the OK button is clicked. Makes sure a Ugrid is selected."""
        super().accept()

    def reject(self):
        """Called when the Cancel button is clicked."""
        super().reject()


def _add_coverages_to_list_widget(cov_data: ObsCovData, list_widget: QListWidget) -> None:
    """Set up the list widget.

    Args:
        cov_data: Dict of observation coverages and their att files.
        list_widget: The QListWidget.
    """
    # Add coverage names to list
    for coverage in cov_data.keys():
        list_widget.addItem(coverage.name)
    # Make list checkable
    for row in range(list_widget.count()):
        list_item = list_widget.item(row)
        list_item.setFlags(list_item.flags() | Qt.ItemIsUserCheckable)
        list_item.setCheckState(Qt.Unchecked)
