"""SimDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (QCheckBox, QHBoxLayout, QLabel, QLineEdit, QSpinBox)
from typing_extensions import override

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import ComponentItem
from xms.api.tree import tree_util, TreeNode
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_table_view import QxTableView

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.gui import gui_util
from xms.mf6.gui.delegates.tree_button_delegate import TreeButtonDelegate
from xms.mf6.gui.options_gui import OptionsGui
from xms.mf6.gui.package_dialog_base import PackageDialogBase


class SimDialog(PackageDialogBase):
    """A dialog used for the IC, NPF and STO packages."""

    # Columns in the MODELS table
    MTYPE_COL = 0
    MFNAME_COL = 1
    MNAME_COL = 2

    # Columns in the EXCHANGES table
    EXGTYPE_COL = 0
    EXGFILE_COL = 1
    EXGMNAMEA_COL = 2
    EXGMNAMEB_COL = 3

    # Columns in the SOLUTIONGROUPS table
    SLNTYPE_COL = 0
    SLNFNAME_COL = 1
    SLNMNAMES_COL = 2

    def __init__(self, dlg_input, parent=None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input (DialogInput): Information needed by the dialog.
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(dlg_input, parent)
        self._models_str = 'MODELS'
        self._exchanges_str = 'EXCHANGES'
        self._solutiongroups_str = 'SOLUTIONGROUPS'

        self.setup_ui()

    @override
    def setup_ui(self) -> None:
        """Set up everything dealing with sections."""
        super().setup_ui()
        self._init_auto_name_option()  # Do this after setting everything up

    def define_sections(self):
        """Defines the sections that appear in the list of sections.

        self.sections should be set here.
        """
        self.sections = [
            'GMS Options', 'COMMENTS', 'OPTIONS', 'TDIS', self._models_str, self._exchanges_str,
            self._solutiongroups_str
        ]
        self.default_sections = ['GMS Options', self._models_str]
        self.blocks_requiring_spacer.extend(['GMS Options', 'TDIS'])

    def setup_section(self, section_name):
        """Sets up a block of widgets.

        Args:
            section_name (str): name of the block
        """
        if section_name == 'GMS Options':
            self._setup_gms_options_section()
        elif section_name == 'TDIS':
            self.setup_tdis_section()
        elif section_name == self._models_str:
            self.setup_models_section()
        elif section_name == self._exchanges_str:
            self.setup_exchanges_section()
        elif section_name == self._solutiongroups_str:
            self.setup_solutiongroups_section()
        else:
            super().setup_section(section_name)

    def _setup_gms_options_section(self):
        """Sets up the GMS Options section."""
        name = 'GMS Options'
        self.add_group_box_to_scroll_area(name)
        self._setup_auto_name_option()

    def _setup_auto_name_option(self):
        name = 'GMS Options'
        str = 'Automatically name all files using Project Explorer names'
        chk = self.uix[name]['chk_auto_name'] = QCheckBox(str)
        self.uix[name]['layout'].addWidget(chk)

        chk.stateChanged.connect(self._on_chk_auto_name)

    def _auto_file_naming(self):
        """Returns True if the Auto name checkbox is checked.

        Returns:
            (bool): See description.
        """
        if 'GMS Options' in self.uix and 'chk_auto_name' in self.uix['GMS Options']:
            return self.uix['GMS Options']['chk_auto_name'].isChecked()
        return False

    def _on_chk_auto_name(self, state):
        """Enables/disables the flow simulation stuff based on the checkbox.

        Args:
            state (int): The state of the check box.
        """
        self._handle_tdis_auto_file_naming()
        self._handle_models_auto_file_naming()
        self._handle_exchanges_auto_file_naming()
        self._handle_solutiongroups_auto_file_naming()

    def _on_chk_mxiter(self, state) -> None:
        """Enables/disables the MXITER spinbox based on the checkbox.

        Args:
            state (int): The state of the check box.
        """
        self.uix[self._solutiongroups_str]['spn_mxiter'].setEnabled(state == Qt.Checked)

    def _handle_tdis_auto_file_naming(self):
        gui_util.set_read_only_and_grey(self.uix['TDIS']['edt_tdis_filename'], self._auto_file_naming())
        if self._auto_file_naming() and self.dlg_input.data.tree_node:  # tree_node can be None when testing
            node_name = self.dlg_input.data.tree_node.name
            self.uix['TDIS']['edt_tdis_filename'].setText(data_util.auto_file_name(node_name, 'TDIS6'))
        else:
            self.uix['TDIS']['edt_tdis_filename'].setText(os.path.basename(self.dlg_input.data.tdis.fname))

    def _handle_models_auto_file_naming(self):
        if self._models_table():
            self._setup_models_model(self._models_table())

    def _handle_exchanges_auto_file_naming(self):
        if self._exchanges_table():
            self._setup_exchanges_model(self._exchanges_table())

    def _handle_solutiongroups_auto_file_naming(self):
        if self._solutiongroups_table():
            self._setup_solutiongroups_model(self._solutiongroups_table())

    def _init_auto_name_option(self):
        """Sets the auto-name checkbox state."""
        auto_name = self.dlg_input.data.gms_options.get('AUTO_FILE_NAMING', False)
        if auto_name:
            self.uix['GMS Options']['chk_auto_name'].setChecked(True)

    def _save_auto_name_option(self):
        """Saves the auto-name checkbox state."""
        self.dlg_input.data.gms_options['AUTO_FILE_NAMING'] = self._auto_file_naming()

    # @overrides
    def setup_options(self, vlayout):
        """Sets up the options section, which is defined dynamically, not in the ui file.

        Args:
            vlayout (QVBoxLayout): The layout that the option widgets will be added to.
        """
        self.options_gui = OptionsGui(self)
        self.options_gui.setup(vlayout)

    def setup_tdis_section(self):
        """Sets up the TDIS block."""
        name = 'TDIS'
        self.add_group_box_to_scroll_area(name)

        hlay = self.uix[name]['hlay_tdis_filename'] = QHBoxLayout()
        self.uix[name]['layout'].addLayout(hlay)
        w = self.uix[name]['txt_tdis_filename'] = QLabel('TDIS filename:')
        hlay.addWidget(w)
        w = self.uix[name]['edt_tdis_filename'] = QLineEdit(os.path.basename(self.dlg_input.data.tdis.fname))
        hlay.addWidget(w)

    def _save_tdis(self):
        """Saves the data in the TDIS section from the widgets to the data structures."""
        filename = self.uix['TDIS']['edt_tdis_filename'].text()
        # f = os.path.join(os.path.dirname(self.dlg_input.data.tdis.filename), filename)
        # self.dlg_input.data.tdis.filename = f
        self.dlg_input.data.tdis.fname = filename

    def setup_models_section(self) -> None:
        """Sets up the MODELS section."""
        self.add_group_box_to_scroll_area(self._models_str)
        table = self._add_models_table()
        self._setup_models_model(table)

    def _models_table(self) -> QxTableView | None:
        """Returns the models table."""
        if self._models_str in self.uix and 'tbl_models' in self.uix[self._models_str]:
            return self.uix[self._models_str]['tbl_models']
        return None

    def _add_models_table(self) -> QxTableView:
        """Create, add, and return the QxTableView."""
        w = self.uix[self._models_str]['tbl_models'] = gui_util.new_table_view()
        self.uix[self._models_str]['layout'].addWidget(w)
        w = self.uix[self._models_str]['txt_mnames'] = \
            QLabel('("mname" is the model\'s Project Explorer item name and cannot be changed here.)')
        self.uix[self._models_str]['layout'].addWidget(w)
        return self.uix[self._models_str]['tbl_models']

    def _setup_models_model(self, table: QxTableView) -> QxPandasTableModel:
        """Create and return a QxPandasTableModel properly initialized for the table view.

        Args:
            table: The table view.

        Returns:
            See description.
        """
        model = self._create_models_model_and_add_data(table)
        self._set_models_read_only_columns(model)
        _add_models_header_tool_tips(model)

    def _create_models_model_and_add_data(self, table: QxTableView) -> QxPandasTableModel:
        # Load the package data into the table using a dataframe
        mtypes = []
        mfnames = []
        mnames = []
        for model in self.dlg_input.data.models:
            mtypes.append(model.mtype)
            mfnames.append(os.path.basename(model.fname))
            mnames.append(model.mname)
        if self._auto_file_naming():
            mfnames = data_util.compute_auto_names(self.dlg_input.data.models, '')
        df_dict = {'MTYPE': mtypes, 'MFNAME': mfnames, 'MNAME': mnames}
        df = pd.DataFrame(df_dict)
        model = QxPandasTableModel(df)
        table.setModel(model)
        table.verticalHeader().hide()
        widget_builder.resize_columns_to_contents(table)
        return model

    def _set_models_read_only_columns(self, model: QxPandasTableModel) -> None:
        """Set the readonly columns on the model.

        Args:
            model: The QxPandasTableModel.
        """
        columns = {self.MTYPE_COL, self.MNAME_COL}
        if self.dlg_input.locked or self._auto_file_naming():
            columns.update({self.MFNAME_COL})
        model.set_read_only_columns(columns)

    def _save_models(self) -> None:
        """Saves the data in the MODELS section from the widgets to the data structures."""
        model = self.uix[self._models_str]['tbl_models'].model()
        for row in range(model.rowCount()):
            filename = model.index(row, 1).data()
            # f = os.path.join(os.path.dirname(self.dlg_input.data.models[row].filename), filename)
            # self.dlg_input.data.models[row].filename = f
            self.dlg_input.data.models[row].fname = filename

    def setup_exchanges_section(self):
        """Sets up the EXCHANGES block."""
        self.add_group_box_to_scroll_area(self._exchanges_str)
        table = self._add_exchanges_table()
        self._setup_exchanges_model(table)

    def _exchanges_table(self):
        """Returns the exchanges table."""
        if self._exchanges_str in self.uix and 'table_exchanges' in self.uix[self._exchanges_str]:
            return self.uix[self._exchanges_str]['table_exchanges']
        return None

    def _add_exchanges_table(self) -> QxTableView:
        table = self.uix[self._exchanges_str]['table_exchanges'] = gui_util.new_table_view()
        self.uix[self._exchanges_str]['layout'].addWidget(table)
        return table

    def _setup_exchanges_model(self, table: QxTableView) -> None:
        """Create and return a QxPandasTableModel properly initialized for the table view.

        Args:
            table: The table view.

        Returns:
            See description.
        """
        model = self._create_exchanges_model_and_add_data(table)
        self._add_exchanges_delegates(table)
        self._set_exchanges_read_only_columns(model)
        _add_exchanges_header_tool_tips(model)

    def _create_exchanges_model_and_add_data(self, table: QxTableView) -> QxPandasTableModel:
        # Load the package data into the table using a dataframe
        exgtypes = []
        exgfiles = []
        exgmnameas = []
        exgmnamebs = []
        for exchange in self.dlg_input.data.exchanges:
            exgtypes.append(exchange.exgtype)
            exgfiles.append(os.path.basename(exchange.fname))
            exgmnameas.append(os.path.basename(exchange.exgmnamea))
            exgmnamebs.append(os.path.basename(exchange.exgmnameb))
        if self._auto_file_naming() and self.dlg_input.data.tree_node:  # tree_node can be None when testing
            exgfiles = data_util.compute_auto_names(self.dlg_input.data.exchanges, self.dlg_input.data.tree_node.name)
        df_dict = {'EXGTYPE': exgtypes, 'EXGFILE': exgfiles, 'EXGMNAMEA': exgmnameas, 'EXGMNAMEB': exgmnamebs}
        df = pd.DataFrame(df_dict)
        model = QxPandasTableModel(df)
        table.setModel(model)
        table.verticalHeader().hide()
        return model

    def _add_exchanges_delegates(self, table: QxTableView) -> None:
        # Add delegates
        query = self.dlg_input.query
        if query:
            project_tree = query.project_tree if query else None
            trimmed_tree = self._filter_tree(project_tree)
            tree_picker_delegate = TreeButtonDelegate(
                dialog_title='Select Model', pe_tree=trimmed_tree, parent=self, query=query, multiple=False
            )
            table.setItemDelegateForColumn(self.EXGMNAMEA_COL, tree_picker_delegate)
            table.setItemDelegateForColumn(self.EXGMNAMEB_COL, tree_picker_delegate)

    def _set_exchanges_read_only_columns(self, model: QxPandasTableModel) -> None:
        columns = {self.EXGTYPE_COL}
        if self._auto_file_naming():
            columns.update({self.EXGFILE_COL})
        if self.dlg_input.locked:
            columns.update({self.EXGFILE_COL, self.EXGMNAMEA_COL, self.EXGMNAMEB_COL})
        model.set_read_only_columns(columns)

    def _save_exchanges(self) -> None:
        """Saves the data in the EXCHANGES section from the widgets to the data structures."""
        name = 'EXCHANGES'
        model = self.uix[name]['table_exchanges'].model()
        for row in range(model.rowCount()):
            filename = model.index(row, 1).data()
            exgmnamea = model.index(row, 2).data()
            exgmnameb = model.index(row, 3).data()
            # f = os.path.join(os.path.dirname(self.dlg_input.data.exchanges[row].filename), filename)
            # self.dlg_input.data.exchanges[row].filename = f
            self.dlg_input.data.exchanges[row].fname = filename
            self.dlg_input.data.exchanges[row].exgmnamea = exgmnamea
            self.dlg_input.data.exchanges[row].exgmnameb = exgmnameb

    def setup_solutiongroups_section(self):
        """Sets up the SOLUTIONGROUPS block."""
        self.add_group_box_to_scroll_area(self._solutiongroups_str)
        self._add_mxiter_spin_box()
        table = self._add_solutiongroups_table()
        self._setup_solutiongroups_model(table)

    def _add_mxiter_spin_box(self):
        hlay = self.uix[self._solutiongroups_str]['hlay_mxiter'] = QHBoxLayout()
        self.uix[self._solutiongroups_str]['layout'].addLayout(hlay)

        # MXITER checkbox
        chk = self.uix[self._solutiongroups_str]['chk_mxiter'] = QCheckBox('MXITER:')
        hlay.addWidget(chk)

        # MXITER spin control
        spn = self.uix[self._solutiongroups_str]['spn_mxiter'] = QSpinBox()
        spn.setMinimum(1)
        spn.setMaximum(1000000)
        spn.setValue(self.dlg_input.data.solution_groups[0].mxiter)
        hlay.addWidget(spn)
        hlay.addStretch()

        # Set chk state and enable/disable spinbox
        chk.stateChanged.connect(self._on_chk_mxiter)
        chk.setChecked(self.dlg_input.data.solution_groups[0].use_mxiter)
        self._on_chk_mxiter(chk.checkState())  # So the spinbox gets enabled/disabled appropriately

    def _solutiongroups_table(self) -> QxTableView | None:
        """Returns the models table."""
        if self._solutiongroups_str in self.uix and 'table_solutiongroups' in self.uix[self._solutiongroups_str]:
            return self.uix[self._solutiongroups_str]['table_solutiongroups']
        return None

    def _add_solutiongroups_table(self) -> QxTableView:
        # Table
        table = self.uix[self._solutiongroups_str]['table_solutiongroups'] = gui_util.new_table_view()
        self.uix[self._solutiongroups_str]['layout'].addWidget(table)
        return table

    def _setup_solutiongroups_model(self, table: QxTableView) -> QxPandasTableModel:
        """Create and return a QxPandasTableModel properly initialized for the table view.

        Args:
            table: The table view.

        Returns:
            See description.
        """
        model = self._create_solutiongroups_model_and_add_data(table)
        self._add_solutiongroups_delegates(table)
        self._set_solutiongroups_read_only_columns(model)
        _add_solutiongroups_header_tool_tips(model)

    def _create_solutiongroups_model_and_add_data(self, table: QxTableView) -> QxPandasTableModel:
        # Load the package data into the table using a dataframe
        slntypes = []
        slnfnames = []
        slnmnames = []
        all_ims = []
        for solution_group in self.dlg_input.data.solution_groups:
            for ims in solution_group.ims_list:
                all_ims.append(ims)
                slntypes.append(ims.ftype)
                slnfnames.append(os.path.basename(ims.fname))
                slnmnames.append(' '.join(ims.slnmnames))
        if self._auto_file_naming() and self.dlg_input.data.tree_node:  # tree_node can be None when testing:
            slnfnames = data_util.compute_auto_names(all_ims, self.dlg_input.data.tree_node.name)
        df_dict = {'SLNTYPE': slntypes, 'SLNFNAME': slnfnames, 'SLNMNAMES': slnmnames}
        df = pd.DataFrame(df_dict)
        model = QxPandasTableModel(df)
        table.setModel(model)
        table.verticalHeader().hide()
        table.horizontalHeader().setStretchLastSection(True)
        return model

    def _add_solutiongroups_delegates(self, table: QxTableView) -> None:
        # Add delegate
        query = self.dlg_input.query
        if query:
            project_tree = query.project_tree if query else None
            trimmed_tree = self._filter_tree(project_tree)
            tree_picker_delegate = TreeButtonDelegate(
                dialog_title='Select Model(s)', pe_tree=trimmed_tree, parent=self, query=query, multiple=True
            )
            table.setItemDelegateForColumn(SimDialog.SLNMNAMES_COL, tree_picker_delegate)

    def _set_solutiongroups_read_only_columns(self, model: QxPandasTableModel) -> None:
        columns = {self.SLNTYPE_COL}
        if self._auto_file_naming():
            columns.update({self.SLNFNAME_COL})
        if self.dlg_input.locked:
            columns.update({self.SLNFNAME_COL, self.SLNMNAMES_COL})
        model.set_read_only_columns(columns)

    def _keep_node_condition(self, node):
        """Method used by tree_util.filter_project_explorer to filter the tree.

        Args:
            node (TreeNode): A node.

        Returns:
            (bool): True if the node should be kept.
        """
        if type(node.data) is ComponentItem:
            if node.unique_name in data_util.model_ftypes():
                return True
            return False
        return True

    def _filter_tree(self, project_tree):
        """Returns a trimmed project explorer containing only the packages we're looking for and their ancestors.

        Returns:
            (TreeNode): The trimmed tree.
        """
        tree_copy = TreeNode(other=project_tree)  # Create a copy of the project explorer tree to filter.
        if tree_copy:
            tree_util.filter_project_explorer(tree_copy, self._keep_node_condition)
        return tree_copy

    def _save_solution_groups(self):
        """Saves the data in the SOLUTIONGROUPS section from the widgets to the data structures."""
        name = 'SOLUTIONGROUPS'
        model = self._solutiongroups_table().model()
        self.dlg_input.data.solution_groups[0].use_mxiter = self.uix[name]['chk_mxiter'].isChecked()
        self.dlg_input.data.solution_groups[0].mxiter = self.uix[name]['spn_mxiter'].value()
        for row in range(model.rowCount()):
            slnfname = model.index(row, SimDialog.SLNFNAME_COL).data()
            slnmnames = model.index(row, SimDialog.SLNMNAMES_COL).data()
            # ims_filename = self.dlg_input.data.solution_groups[0].ims_list[row].filename
            # f = os.path.join(os.path.dirname(ims_filename), slnfname)
            # self.dlg_input.data.solution_groups[0].ims_list[row].filename = f
            self.dlg_input.data.solution_groups[0].ims_list[row].fname = slnfname
            self.dlg_input.data.solution_groups[0].ims_list[row].slnmnames = slnmnames.split()

    @override
    def widgets_to_data(self) -> None:
        """Set dlg_input.data from widgets."""
        super().widgets_to_data()
        if not self.dlg_input.locked:
            self.dlg_input.data.gui_edit_active = True
            self._save_auto_name_option()
            self._save_tdis()
            self._save_models()
            self._save_exchanges()
            self._save_solution_groups()


def _add_models_header_tool_tips(model: QxPandasTableModel):
    model.set_horizontal_header_tooltips(
        {
            0: 'Type of model ("GWF6", "GWT6" etc.)',
            1: 'File name of the model name file',
            2:
                'User-assigned name of the model, restricted to 16 characters. No spaces are allowed.'
                ' Same as the name of the item in the Project Explorer.'
        }
    )


def _add_exchanges_header_tool_tips(model: QxPandasTableModel):
    model.set_horizontal_header_tooltips(
        {
            0: 'Exchange type',
            1: 'Input file for the exchange',
            2: 'Name of the first model that is part of this exchange',
            3: 'Name of the second model that is part of this exchange'
        }
    )


def _add_solutiongroups_header_tool_tips(model: QxPandasTableModel):
    model.set_horizontal_header_tooltips(
        {
            0: 'Type of solution',
            1: 'Name of file containing solution input',
            2: 'Array of model names to add to this solution'
        }
    )
