"""GriddataWidget class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from enum import IntEnum
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt
from PySide2.QtWidgets import (
    QApplication, QCheckBox, QComboBox, QFileDialog, QHBoxLayout, QLabel, QLineEdit, QPushButton, QSizePolicy, QSpinBox,
    QTabWidget, QToolButton, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy.dialogs import dialog_util, message_box
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.mf6.data.array import Array
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.event_filter import EventFilter
from xms.mf6.gui.gui_util import set_read_only_and_grey
from xms.mf6.misc import util
from xms.mf6.misc.settings import Settings


class ConstantLayersEnum(IntEnum):
    """Enumeration for what to do with CONSTANT layers when doing Dataset to Array."""
    PRESERVE = 0
    OVERWRITE = 1
    CANCEL = 2


class GriddataWidget(QWidget):
    """A dialog used for the DIS, IC, NPF, STO and Zone packages."""

    ACTIVE_TAB_INDEX = 'active-tab-index'

    def __init__(self, dlg_input: DialogInput, read_only_tabs: list[str] | None = None, parent: QWidget | None = None):
        """Initializes the class, sets up the ui, and loads the package.

        Args:
            dlg_input: Information needed by the dialog.
            read_only_tabs: List of read_only_tabs
            parent: The parent window.
        """
        super().__init__(parent)
        self.dlg_input = dlg_input
        self._package = self.dlg_input.data if dlg_input else None  # for short
        self._griddata = self._package.block('GRIDDATA') if self._package else None  # for short
        self.models = {}  # Dictionary of classes derived from QAbstractTreeModel
        self.loading_arrays = False  # True in load_arrays
        self.previous_layer = []  # -1 to indicate it's initially unset per tab
        self.modified_arrays = set()  # Set of array names where arrays have been modified. Temporary.
        self.changed_arrays = set()  # Set of array names where arrays have been modified.
        self.ui_tabs = {}  # Widgets added to the tab control
        self.uix = {}  # dictionary of widgets for sections
        self.toolbar = None
        self.event_filter = EventFilter(dialog=self)
        self.read_only_tabs = read_only_tabs  # data tabs that are read only like DELR
        self.grid_info = self._package.grid_info() if self._package else None

    def setup_signals(self):
        """Sets up any needed signals."""
        self.uix['tab_widget'].currentChanged.connect(self.on_tab_changed)

    def setup(self):
        """Sets up the widget."""
        vlayout = QVBoxLayout()
        self.setLayout(vlayout)

        self.uix['tab_widget'] = QTabWidget()
        vlayout.addWidget(self.uix['tab_widget'])

        self.create_tabs()
        self._restore_active_tab()
        self.load_arrays()
        self.setup_signals()
        self.do_enabling()
        # self.adjustSize()

    def create_tabs(self):
        """Creates the tabs for the dialog."""
        for i, name in enumerate(self._griddata.names):
            self.previous_layer.append(1)  # initialize all previous_layer to 1
            widgets = {}
            page = QWidget()
            page.setUpdatesEnabled(False)  # To avoid signals (doesn't seem to work).
            page.setAccessibleName(f'{name} tab')
            vlayout = QVBoxLayout()

            # read only message
            if self.read_only_tabs and name in self.read_only_tabs:
                txt = f'{name} is defined by the grid and is not editable.'
                widgets['txt_read_only'] = QLabel(txt)
                widgets['txt_read_only'].setStyleSheet('font-weight: bold; color: black')
                vlayout.addWidget(widgets['txt_read_only'])

            # Combo box row
            hlayout = QHBoxLayout()
            widgets['chk_define'] = QCheckBox('Define')
            widgets['chk_define'].clicked.connect(self.on_chk_define)
            widgets['chk_define'].setAccessibleName('Define checkbox')
            hlayout.addWidget(widgets['chk_define'])
            if self._package.is_required_array(name):
                widgets['chk_define'].setChecked(True)
                widgets['chk_define'].setVisible(False)

            self._add_binary_file_widgets(hlayout, widgets)

            widgets['chk_layered'] = QCheckBox('Layered')
            widgets['chk_layered'].clicked.connect(self.on_chk_layered)
            widgets['chk_layered'].setAccessibleName('Layered checkbox')
            widgets['spn'] = QSpinBox()
            widgets['spn'].setMinimumWidth(60)
            widgets['spn'].setMinimum(1)
            widgets['spn'].setAccessibleName('Layer spin box')
            if self.grid_info.nlay != -1:
                widgets['spn'].setMaximum(self.grid_info.nlay)
            else:
                widgets['spn'].setMaximum(1)
            widgets['spn'].setKeyboardTracking(False)
            widgets['spn'].valueChanged.connect(self.on_spn_layer)

            widgets['cbx'] = QComboBox()
            widgets['lbl_constant'] = QLabel('Constant:')
            widgets['edt_constant'] = QLineEdit()
            widgets['lbl_units'] = QLabel('Units:')
            widgets['edt_constant'].editingFinished.connect(self.on_constant)
            widgets['edt_constant'].installEventFilter(self.event_filter)
            widgets['edt_constant'].setAccessibleName('Constant/Factor line edit')
            hlayout.addWidget(widgets['chk_layered'])
            hlayout.addWidget(widgets['spn'])
            hlayout.addWidget(widgets['cbx'])
            hlayout.addWidget(widgets['lbl_constant'])
            hlayout.addWidget(widgets['edt_constant'])
            hlayout.addWidget(widgets['lbl_units'])
            hlayout.addStretch()

            # Add combo box items
            widgets['cbx'].addItems(['CONSTANT', 'ARRAY'])

            widgets['cbx'].currentTextChanged.connect(self.on_cbx)
            vlayout.addLayout(hlayout)

            # Table
            widgets['tbl'] = gui_util.new_table_view()
            widgets['tbl'].allow_drag_fill = True
            widgets['tbl'].setMinimumHeight(150)  # I just think this looks better
            vlayout.addWidget(widgets['tbl'])

            if name not in ['DELR', 'DELC']:
                self._add_dataset_to_x_buttons(vlayout, widgets)

            page.setLayout(vlayout)
            self.uix['tab_widget'].addTab(page, name)
            self.uix['tab_widget'].setTabToolTip(i, self._package.get_tool_tip(name))

            self.ui_tabs[name] = widgets
            page.setUpdatesEnabled(True)  # To enable signals.

    def _add_dataset_to_x_buttons(self, vlayout: QVBoxLayout, widgets: dict[str, QWidget]) -> None:
        """Adds the 'Dataset to Layer' and 'Dataset to Array' buttons.

        Args:
            vlayout: Vertical layout of tab.
            widgets: dict of widget names to widgets.
        """
        hlay_dset = QHBoxLayout()
        vlayout.addLayout(hlay_dset)

        # Dataset to Layer
        if self.grid_info.dis_enum != DisEnum.DISU:
            w = widgets['btn_dataset_to_layer'] = QPushButton('Dataset to Layer...')
            hlay_dset.addWidget(w)
            w.clicked.connect(self.on_btn_dataset_to_layer)

        # Dataset to Array
        w = widgets['btn_dataset_to_array'] = QPushButton('Dataset to Array...')
        hlay_dset.addWidget(w)
        w.clicked.connect(self.on_btn_dataset_to_array)

    def _add_binary_file_widgets(self, hlayout, widgets: dict) -> None:
        """Adds the widgets for the external binary file stuff.

        Args:
            hlayout: Horizontal layout that the widgets are added to.
            widgets (dict): Dict of widgets.
        """
        widgets['chk_binary'] = QCheckBox('Binary')
        widgets['chk_binary'].clicked.connect(self.on_chk_binary)

        widgets['txt_binary_file'] = QLabel('File:')

        edt = widgets['edt_binary_file'] = QLineEdit()
        set_read_only_and_grey(edt, True)
        size_policy = edt.sizePolicy()
        size_policy.setHorizontalPolicy(QSizePolicy.Expanding)  # Big as possible
        size_policy.setHorizontalStretch(2)
        edt.setSizePolicy(size_policy)

        btn = widgets['btn_binary_file'] = QToolButton()  # QToolButton used because it's narrower
        btn.setToolButtonStyle(Qt.ToolButtonTextOnly)
        btn.setText('...')
        btn.clicked.connect(self.on_btn_browse)

        hlayout.addWidget(widgets['chk_binary'])
        hlayout.addWidget(widgets['txt_binary_file'])
        hlayout.addWidget(widgets['edt_binary_file'])
        hlayout.addWidget(widgets['btn_binary_file'])

    def to_layered(self):
        """Called when the Layered checkbox is checked."""
        if self.grid_info.dis_enum == DisEnum.DISU:
            return

        tab_name, array, _, _ = self.current_tab_data()
        nvalues, _, shape = self._package.array_size_and_layers(tab_name, True)
        array.to_layered(shape)
        array.dump_to_temp_files()
        self._load_array(array)
        self.do_enabling()

    def to_unlayered(self):
        """Called when the Layered checkbox is unchecked."""
        if self.grid_info.dis_enum == DisEnum.DISU:
            return

        tab_name, array, _, _ = self.current_tab_data()
        array.to_unlayered()
        array.dump_to_temp_files()
        self._load_array(array)
        self.do_enabling()

    def on_chk_define(self):
        """Called when define checkbox is clicked."""
        tab_name = self.current_tab_name()
        define = self.ui_tabs[tab_name]['chk_define'].isChecked()
        if define:
            if self._griddata.array(tab_name) is None:
                layered = self.ui_tabs[tab_name]['chk_layered'].isChecked()
                self._griddata.add_array(self._package.new_array(tab_name, layered))
                self.flag_array_as_modified(tab_name)
            else:
                if self._griddata.array(tab_name).defined != define:
                    self.flag_array_as_modified(tab_name)
                self._griddata.array(tab_name).defined = True
        else:
            if self._griddata.array(tab_name) is not None:
                if self._griddata.array(tab_name).defined != define:
                    self.flag_array_as_modified(tab_name)
                self._griddata.array(tab_name).defined = False
        self.do_enabling()

    def on_btn_browse(self):
        """Called when the browse button is clicked to pick a binary, external, file."""
        tab_name = self.current_tab_name()
        file_name = self.ui_tabs[tab_name]['edt_binary_file'].text()
        dir_ = Path(file_name).parent if file_name and Path(file_name).is_file() else ''
        file_name, _ = QFileDialog.getOpenFileName(parent=self, caption='Binary File', dir=str(dir_))
        if file_name and Path(file_name).is_file():
            self.ui_tabs[tab_name]['edt_binary_file'].setText(file_name)
            array = self._griddata.array(tab_name)
            if array:
                array_layer = array.layer(0)
                array_layer.external_filename = file_name

    def on_chk_binary(self):
        """Called when binary checkbox is clicked."""
        if self.loading_arrays:
            return

        tab_name = self.current_tab_name()
        self.flag_array_as_modified(tab_name)

        # Update the data from the control
        binary = self.ui_tabs[tab_name]['chk_binary'].isChecked()
        array = self._griddata.array(tab_name)
        if array:
            array_layer = array.layer(0)
            array_layer.binary = binary
        self.do_enabling()

    def on_chk_layered(self):
        """Called when layered checkbox is clicked."""
        self.save_array_to_temp()
        tab_name = self.current_tab_name()
        layered = self.ui_tabs[tab_name]['chk_layered'].isChecked()
        if layered:
            self.to_layered()
        else:
            self.to_unlayered()
        self.do_enabling()
        # self.on_cbx(text)

    def on_spn_layer(self, layer):
        """Saves any changes to the current stress period and loads the next stress period.

        Args:
            layer (int): Layer number (1-based).
        """
        if self.loading_arrays:
            return

        self.save_array_to_temp()
        self.load_arrays()
        tab_index = self.uix['tab_widget'].currentIndex()
        self.previous_layer[tab_index] = layer
        self.do_enabling()

    def on_tab_changed(self):
        """Called when the current tab changes."""
        if not self.loading_arrays:  # This is here to stop it while we're building the tabs
            self.do_enabling()

    def flag_array_as_modified(self, name):
        """Adds name to two sets of modified arrays.

        self.modified_arrays gets cleared in save_array_to_temp.
        self.changed_arrays doesn't get cleared so that we can know what's been changed in accept.

        Args:
            name (str): Name of the array.
        """
        self.modified_arrays.add(name)  # Gets wiped out right away
        self.changed_arrays.add(name)  # Stays around until OK

    def on_constant(self):
        """Called when finished editing the constant/factor field, and by the event filter."""
        if self.loading_arrays:
            return

        tab_name = self.current_tab_name()
        text = self.ui_tabs[tab_name]['edt_constant'].text()
        if not text or not util.is_number(text):
            self.do_enabling()  # This will put the text back in the edit field
            return

        self.flag_array_as_modified(tab_name)

        # Update the data from the control
        layer = self.ui_tabs[tab_name]['spn'].value()
        array = self._griddata.array(tab_name)
        if array:
            array_layer = array.layer(layer - 1)
            if self.ui_tabs[tab_name]['cbx'].currentText() == 'CONSTANT':
                array_layer.constant = float(text)
            else:
                array_layer.factor = float(text)

    def _size_is_sufficient_or_warn(self, ts_data) -> bool:
        """Returns True if len(ts_data) is big enough.

        Args:
            ts_data: Dataset values.

        Returns:
            See description.
        """
        tab_name, array, _, layer = self.current_tab_data()
        nvalues, layers_to_read, shape = self._package.array_size_and_layers(tab_name, True)
        if len(ts_data) < (layer * nvalues):  # Assuming a 3D dataset
            message_box.message_with_ok(parent=self.parent(), message='Dataset size is too small. Aborting.')
            return False
        return True

    def _sizes_match_or_warn(self, ts_data) -> bool:
        """Returns True if len(ts_data) is the correct size.

        Args:
            ts_data: Dataset values.

        Returns:
            See description.
        """
        if len(ts_data) != self.grid_info.cell_count():
            message_box.message_with_ok(
                parent=self.parent(), message='Dataset size does not match array size. Aborting.'
            )
            return False
        return True

    def _dataset_to_layer(self, ts_data) -> None:
        """Applies the dataset values to the array.

        Args:
            ts_data (list of float): Values from an XMS dataset.
        """
        tab_name, array, array_layer, layer = self.current_tab_data()
        nvalues, _, shape = self._package.array_size_and_layers(tab_name, True)
        array.dataset_to_layer(ts_data, layer, shape)
        array.dump_to_temp_files()
        self.flag_array_as_modified(tab_name)
        self._load_array(array)
        self.do_enabling()

    def _dataset_to_array(self, ts_data, keep_const_layers: bool):
        """Applies the dataset values to the array.

        Args:
            ts_data (list of float): Values from an XMS dataset.
            keep_const_layers: True to not change layers that are CONSTANT.
        """
        tab_name, array, _, _ = self.current_tab_data()
        nvalues, _, shape = self._package.array_size_and_layers(tab_name, array.layered)
        array.dataset_to_array(ts_data, shape, keep_const_layers)
        array.dump_to_temp_files()
        self.flag_array_as_modified(tab_name)
        self._load_array(array)
        self.do_enabling()

    def _overwrite_constant_layers(self) -> ConstantLayersEnum:
        """Returns PRESERVE to preserve CONSTANT layers, OVERWRITE to overwrite CONSTANT layers, CANCEL to cancel.

        Returns:
            See description.
        """
        tab_name = self.current_tab_name()
        if not self._can_be_layered(tab_name) or not self.ui_tabs[tab_name]['chk_layered'].isChecked():
            return ConstantLayersEnum.OVERWRITE

        # See if any layers are CONSTANT
        array = self._griddata.array(tab_name)
        constant_layers_exist = any(array_layer.storage == 'CONSTANT' for array_layer in array.layers)
        if not constant_layers_exist:
            return ConstantLayersEnum.OVERWRITE

        # Ask user what to do with CONSTANT layers
        msg = 'Some layers are specified as CONSTANT. Apply dataset to these layers too?'
        buttons = ['Preserve CONSTANT layers', 'Overwrite CONSTANT layers', 'Cancel']
        rv = message_box.message_with_n_buttons(self, msg, 'GMS', button_list=buttons, default=0, escape=2)
        return ConstantLayersEnum(rv)

    def on_btn_dataset_to_layer(self) -> None:
        """Lets user pick an XMS dataset and copies the values to the array."""
        cell_count = self.grid_info.cell_count()
        dataset_uuid = gui_util.select_dataset_dialog(self, self.dlg_input.query.project_tree, '', cell_count)
        if dataset_uuid:
            dataset = self.dlg_input.query.item_with_uuid(dataset_uuid)
            ts_data = dataset.values[0]

            # Make sure size is sufficient
            if not self._size_is_sufficient_or_warn(ts_data):
                return

            self._dataset_to_layer(ts_data)
            message_box.message_with_ok(parent=self.parent(), message='Dataset applied.')

    def on_btn_dataset_to_array(self):
        """Lets user pick an XMS dataset and copies the values to the array."""
        cell_count = self.grid_info.cell_count()
        dataset_uuid = gui_util.select_dataset_dialog(self, self.dlg_input.query.project_tree, '', cell_count)
        if dataset_uuid:
            dataset = self.dlg_input.query.item_with_uuid(dataset_uuid)
            ts_data = dataset.values[0]

            # Make sure sizes match
            if not self._sizes_match_or_warn(ts_data):
                return

            # If there are CONSTANT layers, ask if we should apply the dataset to those too
            rv: ConstantLayersEnum = self._overwrite_constant_layers()
            if rv == ConstantLayersEnum.CANCEL:
                return

            self._dataset_to_array(ts_data, rv == ConstantLayersEnum.PRESERVE)
            message_box.message_with_ok(parent=self.parent(), message='Dataset applied.')

    def current_tab_name(self):
        """Returns the name of the current tab."""
        tab_index = self.uix['tab_widget'].currentIndex()
        tab_name = self.uix['tab_widget'].tabText(tab_index)
        return tab_name

    def tab_data(self, tab_name, layer):
        """Returns the Array and the Array for the tab and current layer.

        Returns:
            See description.
        """
        array = self._griddata.array(tab_name)
        array_layer = array.layer(layer - 1)
        return array, array_layer

    def current_tab_data(self) -> tuple[str, Array, ArrayLayer, int]:
        """Returns a bunch of stuff for the current tab.

        Returns:
            See description.
        """
        tab_name = self.current_tab_name()
        layer = self.ui_tabs[tab_name]['spn'].value()
        array, array_layer = self.tab_data(tab_name, layer)
        return tab_name, array, array_layer, layer

    def on_cbx(self, text):
        """Called when a combo box changes.

        Args:
            text (str): The new text.
        """
        if self.loading_arrays:
            return

        tab_name, array, array_layer, _ = self.current_tab_data()
        self.flag_array_as_modified(tab_name)

        # Create an array if necessary
        if text == 'ARRAY':
            if not array_layer.external_filename and not array_layer.temp_external_filename:
                default_value = 1 if array_layer.numeric_type == 'int' else 0.0
                if array_layer.constant:
                    default_value = array_layer.constant
                data_frame = gui_util.default_dataframe(array_layer.numeric_type, array_layer.shape, default_value)
                self.setup_model(tab_name, data_frame)
        array_layer.storage = text
        self.do_enabling()

    def _can_be_layered(self, tab_name: str) -> bool:
        """Returns True if the data can be layered.

        Args:
            tab_name: Name of the current tab.

        Returns:
            See description.
        """
        if self._griddata.array(tab_name) is not None:
            return self._package.can_be_layered(tab_name)
        return True

    def do_enabling(self):
        """Enables and disables the widgets appropriately."""
        not_locked = not self.dlg_input.locked

        tab_index = self.uix['tab_widget'].currentIndex()
        if tab_index < 0:
            return

        tab_name = self.uix['tab_widget'].tabText(tab_index)
        editable_tab = not self.read_only_tabs or (tab_name not in self.read_only_tabs)
        defined = self.ui_tabs[tab_name]['chk_define'].isChecked()
        self.ui_tabs[tab_name]['chk_define'].setEnabled(not_locked and editable_tab)
        # self.ui_tabs[tab_name]['chk_define'].setVisible(editable_tab)

        # binary external file
        binary = self.ui_tabs[tab_name]['chk_binary'].isChecked()
        self.ui_tabs[tab_name]['chk_binary'].setEnabled(not_locked and defined and editable_tab)
        self.ui_tabs[tab_name]['txt_binary_file'].setVisible(binary)
        self.ui_tabs[tab_name]['txt_binary_file'].setEnabled(binary and not_locked and defined and editable_tab)
        self.ui_tabs[tab_name]['edt_binary_file'].setVisible(binary)
        self.ui_tabs[tab_name]['edt_binary_file'].setEnabled(binary and not_locked and defined and editable_tab)
        self.ui_tabs[tab_name]['btn_binary_file'].setVisible(binary)
        self.ui_tabs[tab_name]['btn_binary_file'].setEnabled(binary and not_locked and defined and editable_tab)

        # Layered checkbox
        can_be_layered = self._can_be_layered(tab_name)
        self.ui_tabs[tab_name]['chk_layered'].setEnabled(not_locked and defined and can_be_layered and editable_tab)
        # self.ui_tabs[tab_name]['chk_layered'].setVisible(can_be_layered and editable_tab)
        self.ui_tabs[tab_name]['chk_layered'].setVisible(not binary and can_be_layered)

        # Enable/disable spin box
        layered = self.ui_tabs[tab_name]['chk_layered'].isChecked()
        spn = self.ui_tabs[tab_name]['spn']
        spn.setEnabled(layered and defined and editable_tab)
        spn.setVisible(not binary and can_be_layered)

        # Get cbx text
        cbx_text = self.ui_tabs[tab_name]['cbx'].currentText()
        self.ui_tabs[tab_name]['cbx'].setEnabled(not_locked and defined and editable_tab)
        self.ui_tabs[tab_name]['cbx'].setVisible(not binary)

        # Get array
        if self._griddata.array(tab_name) is not None and self._griddata.array(tab_name).defined:
            if binary:
                array_layer = self._griddata.array(tab_name).layer(0)
            else:
                layer = spn.value()
                array_layer = self._griddata.array(tab_name).layer(layer - 1)

            # Change constant/factor label
            text = 'Constant:' if array_layer.storage == 'CONSTANT' and not array_layer.binary else 'Factor:'
            self.ui_tabs[tab_name]['lbl_constant'].setText(text)

            # Set constant/factor edit field text
            constant_or_factor = array_layer.constant if cbx_text == 'CONSTANT' and not binary else array_layer.factor
            if array_layer.numeric_type == 'int':
                self.ui_tabs[tab_name]['edt_constant'].setText(str(int(constant_or_factor)))
            else:
                self.ui_tabs[tab_name]['edt_constant'].setText(str(constant_or_factor))

            # Units
            units_str = self._package.get_units(tab_name)
            units_str = units_util.string_from_units(self._package, units_str)
            self.ui_tabs[tab_name]['lbl_units'].setText(f'Units: {units_str}')

        self.ui_tabs[tab_name]['lbl_constant'].setEnabled(not_locked and defined and editable_tab)
        self.ui_tabs[tab_name]['edt_constant'].setEnabled(not_locked and defined and editable_tab)

        # Enable/disable table
        self.ui_tabs[tab_name]['tbl'].setEnabled(cbx_text != 'CONSTANT' and defined and not binary)

        # Dataset to X buttons
        if 'btn_dataset_to_layer' in self.ui_tabs[tab_name]:
            w = self.ui_tabs[tab_name]['btn_dataset_to_layer']
            w.setEnabled(defined and not_locked and not binary and ((can_be_layered and layered) or tab_name == 'TOP'))

        if 'btn_dataset_to_array' in self.ui_tabs[tab_name]:
            dis_enum = self.grid_info.dis_enum
            w = self.ui_tabs[tab_name]['btn_dataset_to_array']
            w.setEnabled(defined and not_locked and not binary and (tab_name != 'TOP' or dis_enum == DisEnum.DISU))

    def setup_model(self, array_name: str, data_frame: pd.DataFrame):
        """Creates a model and connects it to the table view and some signals.

        Args:
            array_name: The name of the array ('RECHARGE' etc)
            data_frame: The DataFrame.
        """
        table_view = self.ui_tabs[array_name]['tbl']
        if data_frame is None:
            if array_name in self.models:
                del self.models[array_name]
            table_view.setModel(None)
            return

        model = self.models[array_name] = QxPandasTableModel(data_frame)
        # _, defaults = self.column_info()
        # self.model.set_default_values(defaults)
        model.dataChanged.connect(self.on_data_changed)
        if self.dlg_input.locked or (self.read_only_tabs and array_name in self.read_only_tabs):
            model.set_read_only_columns(set(range(model.columnCount())))
        table_view.setModel(model)
        # self.ui.tbl_surface.selectionModel().selectionChanged.connect(self.on_selection_changed)

    def dataframe_to_temp_file(self, dataframe: pd.DataFrame, array_layer: ArrayLayer):
        """Writes the dataframe to disk and saves the filename into array_layer.

        Also deletes any old file that may be saved with array_layer.

        Args:
            dataframe: The DataFrame.
            array_layer: The Array.
        """
        QApplication.setOverrideCursor(Qt.WaitCursor)
        gui_util.array_dataframe_to_temp_file(dataframe, array_layer)
        QApplication.restoreOverrideCursor()

    def save_array_to_temp(self):
        """If there have been changes to the array, saves the changes to a temporary csv file."""
        for array_name in self.modified_arrays:
            tab_index = self._griddata.names.index(array_name)
            layer = self.previous_layer[tab_index]
            array_layer = self._package.block('GRIDDATA').array(array_name).layer(layer - 1)
            if array_name in self.models:
                self.dataframe_to_temp_file(self.models[array_name].data_frame, array_layer)

        self.modified_arrays.clear()  # Clear this, but leave self.changed_arrays alone

    def load_arrays(self):
        """Loads the arrays."""
        for tab_index in range(self.uix['tab_widget'].count()):
            tab_name = self.uix['tab_widget'].tabText(tab_index)
            array = self._package.block('GRIDDATA').array(tab_name)
            if array:
                self._load_array(array)
        self.do_enabling()

    def _load_array(self, array: Array) -> None:
        """Load the array into the widgets.

        Args:
            array: The array.
        """
        self.loading_arrays = True
        tab_name = array.array_name
        tab = self.ui_tabs[tab_name]

        # Defined checkbox
        tab['chk_define'].setChecked(True)

        # Binary checkbox
        binary = array.layer(0).binary
        tab['chk_binary'].setChecked(binary)
        if binary:
            tab['edt_binary_file'].setText(array.layer(0).external_filename)

        # Layered checkbox
        tab['chk_layered'].setChecked(array.layered)

        layer = tab['spn'].value()
        cbx = tab['cbx']
        array_layer = array.layer(layer - 1)

        # Constant edit field
        # Set the validator
        if array_layer.numeric_type == 'int':
            validator = QxIntValidator(parent=tab['edt_constant'])
        else:
            validator = QxDoubleValidator(parent=tab['edt_constant'])
        tab['edt_constant'].setValidator(validator)

        # Units
        units_str = self._package.get_units(tab_name)
        units_str = units_util.string_from_units(self._package, units_str)
        tab['lbl_units'].setText(f'Units: {units_str}')

        # Change constant/factor label
        if binary:
            pass
        elif array_layer.storage == 'CONSTANT':
            cbx.setCurrentText('CONSTANT')
            self.setup_model(tab_name, None)
        elif array_layer.storage == 'ARRAY':
            cbx.setCurrentText('ARRAY')

            # Read the file
            with dialog_util.wait_cursor_context():
                data_frame = array.dataframe_from_layer(layer - 1, self.grid_info)

            self.setup_model(tab_name, data_frame)
        self.loading_arrays = False

    def on_cbx_length_units(self, units_string: str) -> None:
        """Called when the length unit changes.

        Args:
            units_string (str): The final units string.
        """
        tab_name = self.current_tab_name()
        self.ui_tabs[tab_name]['lbl_units'].setText(f'Units: {units_string}')

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        del top_left_index, bottom_right_index  # Unused parameters
        tab_index = self.uix['tab_widget'].currentIndex()
        tab_name = self.uix['tab_widget'].tabText(tab_index)
        self.flag_array_as_modified(tab_name)

    def _restore_active_tab(self):
        """Restores the active tab from the previous session."""
        active_tab_index = Settings.get(self._package.filename, self.ACTIVE_TAB_INDEX)
        if active_tab_index:
            self.uix['tab_widget'].setCurrentIndex(active_tab_index)

    def _save_active_tab_index(self):
        """Saves the current active tab, so we can restore it later."""
        Settings.set(self._package.filename, self.ACTIVE_TAB_INDEX, self.uix['tab_widget'].currentIndex())

    def accept(self):
        """Called when user hits the OK button. Saves the widget state and closes the dialog."""
        if not self.dlg_input.locked:
            self._save_active_tab_index()
            self.save_array_to_temp()
