"""PackageBuilder class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
import shapefile  # From pyshp

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.grid_info import DisEnum
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.data.gwf.array_package_data import ArrayPackageData
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.mapping.cell_adder import CellAdder
from xms.mf6.mapping.cell_adder_arcs import CellAdderArcs
from xms.mf6.mapping.cell_adder_polys import CellAdderPolys
from xms.mf6.mapping.cell_adder_wel import CellAdderWel
from xms.mf6.mapping.cell_polygon_calculator import CellPolygonCalculator
from xms.mf6.mapping.chd_from_polys_builder import ChdFromPolysBuilder
from xms.mf6.mapping.map_cell_iterator import MapCellIterator
from xms.mf6.mapping.package_builder_base import (
    add_list_package_period_data,
    cell_active_and_no_chd,
    cell_has_chd,
    CovMapInfo,
    PackageBuilderBase,
    PackageBuilderInputs,
    update_chd_cells_from_package,
)
from xms.mf6.mapping.sfr_package_builder import SfrPackageBuilder
from xms.mf6.misc import util
from xms.mf6.misc.util import XM_NODATA

FEATURE_TYPE_ORDER = ['polygons', 'arcs', 'points']


class PackageBuilder(PackageBuilderBase):
    """Builds a package during map from coverage."""
    def __init__(self, inputs: PackageBuilderInputs):
        """Initializes the class.

        Args:
            inputs: Everything needed to build the package.
        """
        super().__init__(inputs)
        self._hfb_disu_warning = False
        self._cell_adder = None  # None unless the package needs to do something special

    @property
    def outputs(self):
        """Get the output data to add to the GMS Query after building the package."""
        if self._package.ftype in data_util.dis_ftypes():
            return {
                'dis': self._package,
                'ugrid_uuid': self.cogrid.uuid,
            }
        return {}

    def build(self):
        """Maps the coverage to the package."""
        self.log.info('Building package')

        # Pipeline
        self._set_up()
        for cov_map_info in self._cov_map_info_list:
            self._set_up_for_current_coverage(cov_map_info)
            for self._feature_type in FEATURE_TYPE_ORDER:
                self._build_package(self._feature_type)
        self._write_package()
        self._save_groups_with_package()
        return self._package

    def _set_up(self) -> None:
        """Do stuff to get set up."""
        self._set_up_period_times()
        self._handle_append_or_replace()
        self._max_group = self._find_max_cellgrp()
        self._turn_on_boundnames()
        self._add_cellgrp_aux()
        self._load_chd_cells()

    def _handle_append_or_replace(self):
        """If replacing, wipes out existing data."""
        if self._replace:
            if isinstance(self._package, ArrayPackageData):  # Array-based EVT or RCH
                self._package.period_data = {}
            elif isinstance(self._package, GriddataBase):  # Griddata package: NPF, STO, IC
                # Delete all griddata except ICELLTYPE (NPF) and IDOMAIN (DIS*), if they exist. Always keep those.
                # for key in list(self._package.griddata):
                #     if key not in ['ICELLTYPE', 'IDOMAIN']:
                #         del self._package.griddata(key)

                # The above wipes out ICONVERT and SS in the STO package (both are required). SY was the only
                # attribute that was defined in the conceptual model, so that's all that existed in the STO
                # package after building the package. So I got rid of the above code. SY is replaced with the
                # new SY data in self._package.new_array() if self._replace is true. Append vs Replace with
                # griddata... what does that mean?
                pass
            elif self._package.periods_db and os.path.isfile(self._package.periods_db):  # List package
                fs.removefile(self._package.periods_db)
                for key, value in self._package.period_files.items():
                    fs.removefile(value)
                    self._package.period_files[key] = ''
            self._remove_cellgrp_info()
        else:
            self._read_cellgrp_info()

    def _build_package(self, feature_type: str) -> None:
        """Build the package.

        Args:
            feature_type: 'points', 'arcs', or 'polygons'.
        """
        if feature_type == 'points' and self._ix_info.points and isinstance(self._package, ListPackageData):
            self._build_list_package_from_points()
        elif feature_type == 'arcs' and self._ix_info.arcs:
            if self._package.ftype == 'SFR6':
                self._build_sfr_package()
            elif isinstance(self._package, ListPackageData):
                self._build_list_package_from_arcs()
        elif feature_type == 'polygons' and self._ix_info.polys:
            if self._package.ftype == 'NPF6':
                self._build_npf_package_from_polygons()
            elif self._package.ftype in {'DIS6', 'DISV6', 'STO6', 'MDT6'}:
                self._build_grid_package_from_polygons()
            else:
                if isinstance(self._package, ArrayPackageData):  # RCH and EVT
                    self._build_array_package_from_polygons()
                else:
                    self._build_list_package_from_polygons()

        # Update the set of CHD cells, so we don't add any twice
        if self._package.ftype == 'CHD6':
            update_chd_cells_from_package(self.grid_info, self._package, self._chd_cells)

    def _write_package(self) -> None:
        """Writes the package to disk."""
        try:
            # Write data to disk, replacing old file
            if self._package.mfsim:
                mfsim_dir = os.path.dirname(self._package.mfsim.filename)
            else:
                mfsim_dir = os.path.dirname(self._package.filename)  # Testing
            options = WriterOptions(mfsim_dir=mfsim_dir, use_periods_db=True, dmi_sim_dir=os.path.join(mfsim_dir, '..'))
            writer = io_factory.writer_from_ftype(self._package.ftype, options)
            writer.write(self._package)
        except Exception as error:
            # Restore backup copy
            self.log.error(str(error))
            # fs.copyfile(backup, self._package.filename)

        # fs.removefile(backup)

    def _add_row_to_period(self, record_dict, period_rows):
        """Adds rows to the periods for the cell.

        Args:
            record_dict: Shapefile record for the current shape.
            period_rows (list): List of rows for a stress period.
        """
        cell_adder = self._cell_adder if self._cell_adder else CellAdder(self)
        cell_adder.set_record(self._cellidx, XM_NODATA, record_dict, period_rows)
        cell_adder.add_bc_to_cell()

    def _add_row_to_period_from_arcs(self, cell_idx1, cell_idx2, length, t_value_avg, record, period_rows):
        """Adds rows to periods for the cell.

        If steady state, row will be added to period 1. If transient, a row will be added to each stress period.

        Args:
            cell_idx1: Cell index 1.
            cell_idx2: Cell index 2.
            length (float): Intersected length of the arc in the cell.
            t_value_avg (float): Average t_value for the cell
            record: Shapefile record for the current arc.
            period_rows (list): List of rows for a stress period.
        """
        cell_adder = CellAdderArcs(self, cell_idx1, cell_idx2, record, period_rows, length, t_value_avg)
        cell_adder.add_bc_to_cell()

    def _add_row_to_period_from_polys(self, cell_idx: int, ix_area: float, record, period_rows):
        """Adds rows to periods for the cell.

        If steady state, row will be added to period 1. If transient, a row will be added to each stress period.

        Args:
            cell_idx: Cell index 1.
            ix_area: Intersected area of the cell.
            record: Shapefile record for the current polygon.
            period_rows (list): List of rows for a stress period.
        """
        cell_adder = CellAdderPolys(self, cell_idx, record, period_rows, ix_area)
        cell_adder.add_bc_to_cell()

    def _get_initial_stuff(self, feature_type):
        """Returns some things we will need and avoids duplicating code.

        Args:
            feature_type (str): 'points', 'arcs', or 'polygons'.

        Returns:
            (tuple): tuple containing:
                - (str): Shapefile name
                - (str): String used in att table for the attribute Type column (i.e. 'well', 'drain', 'river' etc)
                - (list(recarray)): The intersection recarrays
                - (dict): Dict with shapefile or att table fields.
        """
        shapefile_name, att_type, ix_recs, map_info = super()._get_initial_stuff(feature_type)
        if self._package.ftype == 'WEL6':
            self._cell_adder = CellAdderWel(
                self, self._package, self.cogrid.get_cell_tops(), self.cogrid.get_cell_bottoms()
            )
        return shapefile_name, att_type, ix_recs, map_info

    def _build_list_package_from_points(self):
        """Creates a list package from intersection info and a shapefile."""
        shapefile_name, att_type, ix_recs, map_info = self._get_initial_stuff(self._feature_type)
        self.trans_data = self._read_transient_data(list(map_info.keys()), shapefile_name)

        with shapefile.Reader(shapefile_name) as reader:
            _ensure_field_in_shapefile(reader, 'Type')  # raises if not 'Type' field
            records = reader.records()

            # Do it one period at a time so as not to accumulate too much in RAM before we dump it
            period_list = [1] if not self.trans_data else list(self.trans_data.keys())
            for self.period in period_list:
                period_rows = []

                for i, record in enumerate(records):
                    if record['Type'] != att_type:
                        continue  # Shape intersected no cells or the feature att type doesn't match. Skip it.

                    self._shape = reader.shape(i)
                    self._add_cells(ix_recs[i].cellids.tolist(), record, period_rows)

                if period_rows:
                    add_list_package_period_data(self._package, self.period, period_rows)

    def _add_cells(self, cell_idxs: list[int], record, period_rows):
        for cell_idx in cell_idxs:
            # Add a bc to every layer in the layer range
            if self._package.ftype == 'WEL6':
                self._cell_adder.compute_well_screen_overlaps(cell_idx, record.as_dict())
            map_cell_iterator = MapCellIterator(cell_idx, self, record)
            for self._cellidx in map_cell_iterator:
                if cell_active_and_no_chd(self._idomain, self.period, self._cellidx, self._chd_cells):
                    self._add_row_to_period(record.as_dict(), period_rows)

    def _build_list_package_from_arcs(self) -> None:
        """Creates a list package from intersection info and a shapefile."""
        shapefile_name, att_type, _ix_recs, map_info = self._get_initial_stuff(self._feature_type)
        self.trans_data = self._read_transient_data(list(map_info.keys()), shapefile_name)

        with shapefile.Reader(shapefile_name) as reader:
            # Aerial coverage polys can have arcs with no Type. Not an error, but we don't want to map those.
            if not _field_in_shapefile(reader, 'Type'):
                return

            records = reader.records()

            # Do it one period at a time so as not to accumulate too much in RAM before we dump it
            period_list = [1] if not self.trans_data else list(self.trans_data.keys())
            for self.period in period_list:
                period_rows = []

                for shape_idx, record in enumerate(records):
                    if record['Type'] != att_type:
                        continue  # Shape intersected no cells or the feature att type doesn't match. Skip it.

                    self._shape = reader.shape(shape_idx)
                    if self._package.ftype == 'HFB6':
                        self._create_hfb_bcs_from_arcs(record, shape_idx, self.period, period_rows)
                    else:
                        self._create_bcs_from_arcs(record, shape_idx, self.period, period_rows)

                if period_rows:
                    add_list_package_period_data(self._package, self.period, period_rows)

    def _create_bcs_from_arcs(self, record, shape_idx: int, period: int, period_rows: list) -> None:
        """Create BCs at cells.

        Args:
            record: A record from the shapefile.
            shape_idx (int): Index of the shape.
            period (int): The stress period.
            period_rows (list): List of rows for a stress period.
        """
        cell_idxs = self._ix_info.arcs[shape_idx].cellids.tolist()
        lengths = self._ix_info.arcs[shape_idx].lengths.tolist()
        t_values = self._ix_info.arc_t_vals

        for j, cell_idx in enumerate(cell_idxs):
            map_cell_iterator = MapCellIterator(cell_idx, self, record.as_dict())
            for self._cellidx in map_cell_iterator:
                if cell_active_and_no_chd(self._idomain, period, self._cellidx, self._chd_cells):
                    # Avg t_value from 1st & last
                    t_value_avg = (t_values[shape_idx][j][0] + t_values[shape_idx][j][-1]) * 0.5
                    self._add_row_to_period_from_arcs(
                        self._cellidx, XM_NODATA, lengths[j], t_value_avg, record.as_dict(), period_rows
                    )

    def _create_hfb_bcs_from_arcs(self, record, shape_idx: int, period: int, period_rows: list) -> None:
        """Create HFB BCs at cells.

        Args:
            record: A record from the shapefile.
            shape_idx (int): Index of the shape.
            period (int): The stress period.
            period_rows (list): List of rows for a stress period.
        """
        hfb_cell_faces = self._ix_info.arc_hfb_cell_faces[shape_idx]
        for cell_face_tuple in hfb_cell_faces:
            cell_idx2 = self.ugrid.get_cell_3d_face_adjacent_cell(cell_face_tuple[0], cell_face_tuple[1])
            if cell_idx2 is None or cell_idx2 < 0:
                continue

            cell_idx1 = cell_face_tuple[0]

            # Add a bc to every layer in the layer range
            map_cell_iterator1 = MapCellIterator(cell_idx1, self, record.as_dict())
            map_cell_iterator2 = MapCellIterator(cell_idx2, self, record.as_dict())
            for cell_idx1, cell_idx2 in zip(map_cell_iterator1, map_cell_iterator2):
                if (
                    cell_active_and_no_chd(self._idomain, period, cell_idx1, self._chd_cells) and  # noqa W503
                    cell_active_and_no_chd(self._idomain, period, cell_idx2, self._chd_cells)
                ):  # noqa W503
                    self._add_row_to_period_from_arcs(cell_idx1, cell_idx2, 1.0, 1.0, record.as_dict(), period_rows)

    def _build_list_package_from_polygons(self):
        """Creates a list based package from intersection info and a shapefile."""
        shapefile_name, att_type, ix_recs, map_info = self._get_initial_stuff(self._feature_type)
        self.trans_data = self._read_transient_data(list(map_info.keys()), shapefile_name)
        # cell_count = self.grid_info.cell_count()

        with shapefile.Reader(shapefile_name) as reader:
            if att_type:
                _ensure_field_in_shapefile(reader, 'Type')
            records = reader.records()

            if self._package.ftype != 'CHD6':
                # cell_polys = self._compute_cell_polygons(
                #     cell_count, ix_recs, records, highest_active_cell=True, att_type=att_type
                # )

                # Do it one period at a time so as not to accumulate too much in RAM before we dump it
                period_list = [1] if not self.trans_data else list(self.trans_data.keys())
                for self.period in period_list:
                    period_rows = []

                    for shape_idx, record in enumerate(records):
                        if att_type and record['Type'] != att_type:
                            continue  # Shape intersected no cells or the feature att type doesn't match. Skip it.

                        self._shape = reader.shape(shape_idx)
                        self._create_bcs_from_polygon(record, shape_idx, self.period, period_rows)

                    if period_rows:
                        add_list_package_period_data(self._package, self.period, period_rows)
            else:
                builder = ChdFromPolysBuilder(self, ix_recs, records, reader)
                builder.build()

    def _create_bcs_from_polygon(self, record, shape_idx: int, period: int, period_rows: list):
        """Create BCs at cells.

        Args:
            record: A record from the shapefile.
            shape_idx: Index of the shape.
            period: The stress period.
            period_rows: List of rows for a stress period.
        """
        cell_idxs = self._ix_info.polys[shape_idx].cellids.tolist()
        ix_areas: list[float] = self._ix_info.polys[shape_idx].areas

        # Iterate through the cells intersected by this polygon
        for j, cell_idx in enumerate(cell_idxs):
            map_cell_iterator = MapCellIterator(cell_idx, self, record.as_dict())
            for self._cellidx in map_cell_iterator:
                if not cell_has_chd(period, self._cellidx, self._chd_cells):
                    self._add_row_to_period_from_polys(self._cellidx, ix_areas[j], record.as_dict(), period_rows)

    def _build_grid_package_from_polygons(self):
        """Creates a grid based package from intersection info and a shapefile.

        Returns:
            (dict): MODFLOW names to short names (names found in the shapefile).
        """
        shapefile_name, _att_type, ix_recs, map_info = self._get_initial_stuff(self._feature_type)
        cell_count = self.grid_info.cell_count()

        mf_to_short_names = {}  # Dict of MODFLOW names to short names (names found in shapefile)
        with shapefile.Reader(shapefile_name) as reader:
            records = reader.records()
            cell_polys = self._compute_cell_polygons(cell_count, ix_recs, records, highest_active_cell=False)

            # Get set of fields
            fields = reader.fields
            fields_set = {field[0] for field in fields}

            # Initialize the arrays for the period to nodata
            arrays = {}
            for short_name, mapping in map_info.items():
                if short_name in fields_set:
                    mf_name = mapping.mf_name if mapping else short_name
                    arrays[mf_name] = [util.XM_NODATA] * cell_count  # sizeof cell_count, value in each cell
                    mf_to_short_names[mf_name] = short_name

            # Fill in the array values for each cell
            for array_name, value_list in arrays.items():
                short_name = mf_to_short_names[array_name]
                for cellidx in range(len(cell_polys)):
                    record_index = cell_polys[cellidx]
                    if record_index >= 0:
                        value_list[cellidx] = records[record_index][short_name]

                # TOP is special. It's size of ncpl for DIS, DISV
                if array_name == 'TOP' and self.grid_info.dis_enum in [DisEnum.DIS, DisEnum.DISV]:
                    value_list = value_list[:self.grid_info.cells_per_layer()]

                # Add the array to the griddata package
                # We decided to always "append" to the griddata arrays. We don't want to "replace" array values.
                array = self._package.add_array('GRIDDATA', array_name, layered=False, replace_existing=False)
                _, _, shape = self._package.array_size_and_layers(array_name, layered=array.layered)
                array.set_values(values=value_list, shape=shape, combine=True)

        return mf_to_short_names

    def _build_npf_package_from_polygons(self):
        """Creates a grid based package from intersection info and a shapefile."""
        mf_to_short_names = self._build_grid_package_from_polygons()

        # For NPF, turn on options for HANI and VANI if needed
        griddata = self._package.block('GRIDDATA')
        for array in griddata.arrays:
            if mf_to_short_names.get(array.array_name, '') == 'HANI':
                self._package.options_block.set('K22OVERK', True, None)
            if mf_to_short_names.get(array.array_name, '') == 'VANI':
                self._package.options_block.set('K33OVERK', True, None)

    def _build_array_package_from_polygons(self):
        """Creates an array based package from intersection info and a shapefile."""
        shapefile_name, _att_type, ix_recs, map_info = self._get_initial_stuff(self._feature_type)
        self.trans_data = self._read_transient_data(list(map_info.keys()), shapefile_name)
        cell_count = self.grid_info.cell_count()
        ncpl = self.grid_info.cells_per_layer()

        # Get dict of modflow names -> short names (names in the shapefile/transient data file)
        mf_to_short_names = {}
        for short_name, mapping in map_info.items():
            mf_name = mapping.mf_name if mapping else short_name
            mf_to_short_names[mf_name] = short_name

        # Create arrays
        with shapefile.Reader(shapefile_name) as reader:
            records = reader.records()
            cell_polys = self._compute_cell_polygons(cell_count, ix_recs, records, highest_active_cell=True)

            # Do it one period at a time so as not to accumulate too much in RAM before we dump it
            first = True
            period_list = [1] if not self.trans_data else list(self.trans_data.keys())
            for self.period in period_list:

                # Initialize the arrays for the period to nodata
                arrays = {}
                for short_name, mapping in map_info.items():
                    mf_name = mapping.mf_name if mapping else short_name
                    arrays[mf_name] = [util.XM_NODATA] * ncpl  # sizeof ncpl, value in each cell

                # Compute IEVT/IRCH only once, but add it for all stress periods if we need to
                if first:
                    ievt_or_irch = self._get_ievt_or_irch(cell_polys, ncpl)
                    first = False
                if ievt_or_irch:
                    name = 'IEVT' if self._package.ftype == 'EVT6' else 'IRCH'
                    arrays[name] = ievt_or_irch

                cell_polys_1_layer = self._collapse_cell_polys(cell_polys, ncpl, ievt_or_irch)

                # Calculate the array values in each cell
                if self.trans_data:
                    for index in range(len(cell_polys_1_layer)):
                        record_index = cell_polys_1_layer[index]
                        if record_index >= 0:
                            feature_id = records[record_index]['ID']
                            feature_data = self._get_feature_data(self.period, feature_id)
                            if feature_data:
                                for short_name, mapping in map_info.items():
                                    mf_name = mapping.mf_name if mapping else short_name
                                    value = feature_data[short_name]
                                    if value is not None:
                                        arrays[mf_name][index] = value
                else:
                    for array_name, value_list in arrays.items():
                        if array_name in ['IEVT', 'IRCH']:
                            continue
                        short_name = mf_to_short_names[array_name]
                        for index in range(len(cell_polys_1_layer)):
                            record_index = cell_polys_1_layer[index]
                            if record_index >= 0:
                                value_list[index] = records[record_index][short_name]
                self._add_array_based_period(self.period, arrays)

    def _get_feature_data(self, period, feature_id):
        """Returns the dict for the period and feature."""
        feature_data = None
        period_data = self.trans_data.get(period)
        if period_data:
            feature_data = period_data.get(feature_id)
        return feature_data

    def _collapse_cell_polys(self, cell_polys, ncpl, ievt_or_irch):
        """Collapse the cell_polys list down to just one layer using the ievt_or_irch array.

        Args:
            cell_polys (list): The polygon index with the largest intersecting area for each cell.
            ncpl (int):  Number of cells per layer.
            ievt_or_irch (list): The IEVT or IRCH list.

        Returns:
            (list): The polygon index with the largest intersecting area for each cell.
        """
        cell_polys_1_layer = cell_polys[:ncpl]
        if ievt_or_irch:
            index = 0
            for _layer in range(1, self.grid_info.nlay):  # Start on layer 2
                for i in range(ncpl):
                    if cell_polys[index] > -1:
                        cell_polys_1_layer[i] = cell_polys[index]
                    index += 1
        return cell_polys_1_layer

    def _get_ievt_or_irch(self, cell_polys, ncpl):
        """Returns the IEVT or IRCH array if we need one, and None if we don't.

        Args:
            cell_polys (list): The polygon index with the largest intersecting area for each cell.
            ncpl (int):  Number of cells per layer.

        Returns:
            (list): IEVT or IRCH list, size of ncpl.
        """
        if self.grid_info.nlay == 1:
            return  # No need for IEVT or IRCH

        ievt_or_irch = [1] * ncpl  # Initialize to layer 1
        index = ncpl
        found = False
        for layer in range(1, self.grid_info.nlay):  # Start on layer 2
            for i in range(ncpl):
                if cell_polys[index] > -1:
                    ievt_or_irch[i] = layer + 1
                    found = True
                index += 1

        if not found:  # All in layer 1. No need for IEVT/IRCH
            return
        return ievt_or_irch

    def _compute_cell_polygons(self, cell_count, ix_recs, records, highest_active_cell=True, att_type=''):
        """Computes and returns the array, size of cell_count, that tells which polygon goes with each cell.

        Args:
            cell_count (int): Number of cells in the grid.
            ix_recs (list): The intersection info.
            records: The records from the shapefile.
            highest_active_cell (bool): True if we only want to include cells in the top layer (not for DISU).
            att_type (str): String used in att table for the attribute Type column (i.e. 'well', 'drain', 'river' etc)

        Returns:
            (list): The polygon index with the largest intersecting area for each cell.
        """
        calculator = CellPolygonCalculator(self, ix_recs, records, att_type)
        return calculator.compute_cell_polygons(cell_count, highest_active_cell)

    def _add_array_based_period(self, period: int, arrays):
        """Adds a stress period with the values.

        Args:
            period: The stress period.
            arrays (dict): Dict of array name -> list of values
        """
        self._package.add_period(period, replace_existing=False)
        for array_name, value_list in arrays.items():
            array = self._package.add_transient_array(period, array_name, replace_existing=False)
            layer_indicator = self._package.is_layer_indicator(array_name)
            constant = 1 if layer_indicator else 0.0
            array_layer = array.ensure_layer_exists(make_int=layer_indicator, name=array_name, constant=constant)
            array_layer.name = array_name
            _, shape = ArrayLayer.number_of_values_and_shape(layered=True, grid_info=self.grid_info)
            # array.set_values(value_list, shape)
            array_layer.shape = shape
            if array_name not in ['IEVT', 'IRCH'] or self._replace:
                # Keep old 'IEVT'/'IRCH' if appending.
                array_layer.combine_values(value_list, shape)
            array.storage = array_layer.storage

    def _build_sfr_package(self):
        """Builds the SFR package from the arc data."""
        cov_map_info = CovMapInfo(self._coverage_uuid, self._shapefile_names, self._ix_info)
        inputs = PackageBuilderInputs(self._package, int(self._replace), [cov_map_info], self.cogrid, self._idomain)
        builder = SfrPackageBuilder(inputs)
        builder.build()


def _field_in_shapefile(reader, field_name: str) -> bool:
    """Return True if field is in the shapefile, else return False.

    Args:
        reader (shapefile.Reader): shapefile reader class
        field_name: field that must be present

    Returns:
        See description.
    """
    return any(field_name == field[0] for field in reader.fields)


def _ensure_field_in_shapefile(reader, field_name) -> None:
    """Raises an exception if the field is not in the shapefile.

    Args:
        reader (shapefile.Reader): shapefile reader class
        field_name (str): field that must be present
    """
    if not _field_in_shapefile(reader, field_name):
        raise RuntimeError('Missing "Type" field in shapefile.')
