"""BcComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util, TreeNode
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import Parameter, Section
from xms.gmi.gui.dataset_callback import DatasetRequest
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.rsm.data.mesh_data_def import generic_model
from xms.rsm.dmi.dataset_tree import cell_dataset_selection_tree


class MeshDataComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base class."""
    def _get_section(self, target: TargetType) -> Section:
        return generic_model().section_from_target_type(target)

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> str | TreeNode:
        """
        Handle a request for information when picking a dataset.

        This should match `xms.gmi.gui.dataset_callback.DatasetCallback`.

        Notes:
            - `self._query` will be initialized and available before this method is called.
            - If `request` is `DatasetRequest.GetTree`, the returned node will be a clean copy of the entire project
              tree. Derived classes can safely mutate it for filtering purposes.

        Args:
            request: The requested operation.
            parameter: The parameter the request is for. If `request` is `DatasetRequest.GetLabel`, then
                `parameter.value` will be initialized to the UUID of the dataset to get the label for.

        Returns:
            - If `request` is `DatasetRequest.GetLabel`, returns a label to identify the dataset with to the user.
            - If `request` is `DatasetRequest.GetTree`, returns a tree for picking a new dataset for the parameter.
        """
        if self._project_tree is None:
            self._project_tree = cell_dataset_selection_tree(self._query, None)
        if request == DatasetRequest.GetLabel:
            return tree_util.build_tree_path(self._project_tree, parameter.value)
        if request == DatasetRequest.GetTree:
            return self._project_tree
        raise AssertionError('Unknown request.')
