"""canal data definition methods."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

BC_VAL_FLT = 'Constant (<const>)'
BC_VAL_CURVE = 'Curve (<csv>)'
BC_VAL_DSS = 'DSS file (<dss>)'
BC_VAL_RC = 'RC (rule curve) (<rc>)'


def add_bc_val_to_group(group, prefix='', label_replace='', parent=None, skip_csv=False, skip_rc=False):
    """Add the boundary condition value to the group.

    Args:
        group (xms.gmi.data.generic_model.Group): The group to add the boundary condition value to.
        prefix (str): The prefix to add to the group name.
        label_replace (str): The label to replace "Boundary condition" with.
        parent (xms.gmi.data.generic_model.Parameter, optional): The parent parameter for uniformity. Defaults to None.
        skip_csv (bool): If True, skip the CSV option. Defaults to False.
        skip_rc (bool): If True, skip the RC option. Defaults to False.
    """
    label_part = 'Boundary condition'
    if label_replace:
        label_part = label_replace
    bc_val_opts = [BC_VAL_FLT, BC_VAL_CURVE, BC_VAL_DSS, BC_VAL_RC]
    if skip_csv:
        bc_val_opts.remove(BC_VAL_CURVE)
    if skip_rc:
        bc_val_opts.remove(BC_VAL_RC)
    label = f'{label_part} value type (<const>, <csv>, <dss>, <rc>)'
    if skip_csv:
        label = label.replace(' <csv>,', '')
    opt_par = group.add_option(name=f'{prefix}bc_val_type', label=label, default=bc_val_opts[0], options=bc_val_opts)
    if parent is not None:
        opt_par.add_dependency(parent=parent, flags={True: False, False: True})

    label = f'{label_part} value (<const>)'
    p = group.add_float(name=f'{prefix}bc_val_flt', label=label, default=0.0)
    flags_bc_val_float = {opt: False for opt in bc_val_opts}
    flags_bc_val_float[BC_VAL_FLT] = True
    p.add_dependency(parent=opt_par, flags=flags_bc_val_float)

    if not skip_csv:
        label = f'{label_part} time series (<csv>)'
        p = group.add_curve(name=f'{prefix}bc_val_curve', label=label, use_dates=True, axis_titles=['Time', 'Value'])
        flags_bc_val_curve = {opt: False for opt in bc_val_opts}
        flags_bc_val_curve[BC_VAL_CURVE] = True
        p.add_dependency(parent=opt_par, flags=flags_bc_val_curve)

    label = f'{label_part} DSS time series (<file>)'
    file_filter = 'DSS files (*.dss);;All files (*)'
    p = group.add_input_file(name=f'{prefix}bc_val_dss', label=label, default='', file_filter=file_filter)
    flags_bc_val_dss = {opt: False for opt in bc_val_opts}
    flags_bc_val_dss[BC_VAL_DSS] = True
    p.add_dependency(parent=opt_par, flags=flags_bc_val_dss)
    p = group.add_text(name=f'{prefix}bc_val_dss_path', label='DSS path (<pn>)', default='')
    p.add_dependency(parent=opt_par, flags=flags_bc_val_dss)
    p = group.add_float(name=f'{prefix}bc_val_dss_mult', label='DSS multiplier (<mult>)', default=1.0)
    p.add_dependency(parent=opt_par, flags=flags_bc_val_dss)
    p = group.add_text(name=f'{prefix}bc_val_dss_units', label='DSS units (<units>)', default='')
    p.add_dependency(parent=opt_par, flags=flags_bc_val_dss)
    dss_type_opts = ['Not specified', 'INST-VAL', 'INST-CUM', 'PER-CUM', 'PER-AVE']
    pname = f'{prefix}bc_val_dss_type'
    plabel = 'DSS type of data (<type>)'
    p = group.add_option(name=pname, label=plabel, default=dss_type_opts[0], options=dss_type_opts)
    p.add_dependency(parent=opt_par, flags=flags_bc_val_dss)

    if not skip_rc:
        label = f'{label_part} rule curve name (<rc>)'
        p = group.add_text(name=f'{prefix}bc_val_rc', label=label, default='')
        flags_bc_val_rc = {opt: False for opt in bc_val_opts}
        flags_bc_val_rc[BC_VAL_RC] = True
        p.add_dependency(parent=opt_par, flags=flags_bc_val_rc)
