"""Simulation data definition."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel

# 4. Local modules
from xms.rsm.data.mesh_data_def import add_mesh_data_to_section


class _RsmModelControlDataDef:
    """Class to build the GenericModel class for the model control data."""
    def __init__(self):
        """Constructor."""
        self.gm = GenericModel()
        self._build_global_parameters()
        self._grp = None
        self._add_layer = False

    def _build_global_parameters(self):
        """Fill in the global_parameters in the GenericModel class."""
        self._add_control_data_to_group()
        add_mesh_data_to_section(self.gm.global_parameters)
        self._add_output_data_to_group()

    def _add_control_data_to_group(self):
        """Add model control data to the GenericModel class."""
        gm_gp = self.gm.global_parameters
        gp = gm_gp.add_group(group_name='control', label='Control')
        gp.is_active = True
        gp.add_text(name='run_descriptor', label='Model run description (runDescriptor)')
        gp.add_integer(name='start_time', label='Start time (starttime)', default=0, low=0, high=9999)
        gp.add_integer(name='end_time', label='End time (endtime)', default=2400, low=0, high=9999)
        gp.add_integer(name='timestep_length', label='Timestep length (tslen)', default=24, low=1)
        opts = ['second', 'minute', 'hour', 'day', 'week']
        gp.add_option(name='timestep_type', label='Timestep type (tstype)', options=opts, default=opts[3])
        label = 'Timestep weighting factor (alpha)'
        gp.add_float(name='timestep_weighting', label=label, default=0.9, low=0.0, high=1.0)
        dt = datetime(year=1990, month=1, day=1)
        dt1 = datetime(year=1990, month=12, day=31)
        gp.add_date_time(name='start_date', label='Start date (startdate)', default=dt, calendar_popup=False)
        gp.add_date_time(name='end_date', label='End date (startdate)', default=dt1, calendar_popup=False)
        opts = ['gmres', 'cg', 'cgs', 'bi-cg-stab', 'tfqmr', 'richardson', 'chebychev', 'other']
        gp.add_option(name='solver_method', label='Solver method (solver)', options=opts, default=opts[0])
        opts = ['aschwarz', 'bjacobi', 'jacobi', 'ilu', 'icc', 'lu', 'other']
        gp.add_option(
            name='solver_preconditioner', label='Solver preconditioner (precond)', options=opts, default=opts[1]
        )

    def _add_output_data_to_group(self):
        """Add output data to the GenericModel class."""
        gm_gp = self.gm.global_parameters
        gp = gm_gp.add_group(group_name='output', label='Output')
        gp.is_active = True

        parent = gp.add_boolean(name='wb_budget', label='Output WB budget <wbbudgetpackage>', default=False)
        p = gp.add_text(name='wb_budget_filename', label='', default='wb_budget.nc')
        p.add_dependency(parent=parent, flags={True: True, False: False})
        p = gp.add_integer(name='wb_dbintl', label='WB budget interval (<dbintl>)', default=0)
        p.add_dependency(parent=parent, flags={True: True, False: False})

        parent = gp.add_boolean(name='hpm_budget', label='Output HPM budget <hpmbudgetpackage>', default=False)
        p = gp.add_text(name='hpm_budget_filename', label='', default='hpm_budget.nc')
        p.add_dependency(parent=parent, flags={True: True, False: False})
        p = gp.add_integer(name='hpm_dbintl', label='HPM budget interval (<dbintl>)', default=0)
        p.add_dependency(parent=parent, flags={True: True, False: False})


def generic_model():
    """Gets a generic model for the boundary conditions coverage.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    dd = _RsmModelControlDataDef()
    return dd.gm
