"""CanalComponentHelper class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.gui.group_set_dialog import GroupSetDialog

# 4. Local modules
from xms.rsm.data.canal_data_def import generic_model, generic_model_default_canal


class CanalComponentHelper:
    """Helper class for the canal component."""
    def __init__(self, comp) -> None:
        """Initializes the class.

        Args:
            comp (CanalComponent): The canal component
        """
        self._comp = comp
        self._generic_model = generic_model()
        self._gm_default_canal = generic_model_default_canal()

        if 'DEFAULT_CANAL' in self._comp.data.info.attrs:
            default_arc_gm = GenericModel(definitions=self._comp.data.info.attrs['DEFAULT_CANAL'])
            def_canal = default_arc_gm.arc_parameters.group('default_canal')
            default_canal_group = self._gm_default_canal.arc_parameters.group('default_canal')
            canal_group = self._generic_model.arc_parameters.group('specified_canal')
            par_names = list(set(def_canal.parameter_names) & set(default_canal_group.parameter_names))
            for name in par_names:
                def_par = def_canal.parameter(name)
                for canal in [default_canal_group, canal_group]:
                    par = canal.parameter(name)
                    par.default = def_par.value
                    par.value = def_par.value

    @property
    def generic_model(self) -> GenericModel:
        """Gets the generic model.

        Returns:
            (xms.gmi.data.generic_model.GenericModel): the generic model class
        """
        return self._generic_model

    @property
    def gm_default_canal(self) -> GenericModel:
        """Gets the generic model for the default canal.

        Returns:
            (xms.gmi.data.generic_model.GenericModel): the generic model class
        """
        return self._gm_default_canal

    def open_default_canal(self, query, params, win_cont):
        """Opens the Assign arc dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list[dict]): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            (tuple(list,list)):
                messages (list[tuple(str,str)]): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (list[ActionRequest]): List of actions for XMS to perform.
        """
        ap = self._gm_default_canal.arc_parameters
        dlg_name = 'xms.rsm.canal_component.default_canal'
        dlg = GroupSetDialog(
            parent=win_cont,
            section=ap,
            get_curve=None,
            add_curve=None,
            is_interior=False,
            dlg_name=dlg_name,
            window_title='Default Canal Properties',
            multi_select_message='',
            show_groups=False
        )
        if dlg.exec():
            self._comp.data.info.attrs['DEFAULT_CANAL'] = self._gm_default_canal.to_string()
            self._comp.data.commit()
        return [], []


def helper_generic_model(component):
    """Gets a generic model for the canal component.

    Returns:
        (xms.gmi.data.generic_model.GenericModel): the generic model class
    """
    if hasattr(component, 'canal_helper'):
        return component.canal_helper.generic_model
    return generic_model()
