"""Class for writing mesh properties to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from functools import cached_property
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import hpm_data_def as hpmdd


class HpmDataWriter:
    """Writer class for the RSM control file."""
    def __init__(self, writer_data, elem, mesh_property_list, id_str):
        """Constructor.

        Args:
            writer_data (WriterData): Class with information needed to writer model input files.
            elem (xml.etree.cElementTree.Element): the parent element to write to
            mesh_property_list (list of MeshPropertyData): the list of hpm data to write
            id_str (list(str)): strings to identify the polygons with the mesh properties
        """
        self._wd = writer_data
        self._rc_lookup = self._wd.rule_curve_label_id
        self._hpm_root_xml = elem
        self._prop_list = mesh_property_list
        self._id_str = id_str
        self._cur_id_str = ''
        self._cur_hpm = None
        self._atts = {}
        self._cur_hpm_xml = None
        self._ramcc = False
        self._ns_methods = {
            hpmdd.NS_TYPE_LAY1: self._write_layer1nsm,
            hpmdd.NS_TYPE_UNSAT: self._write_unsat,
            hpmdd.NS_TYPE_NAM: self._write_nam,
            hpmdd.NS_TYPE_LAY5: self._write_layer5,
            hpmdd.NS_TYPE_LAYPC: self._write_layerpc
        }
        self._ag_methods = {
            hpmdd.AG_TYPE_AFS: self._write_afsirs,
            hpmdd.AG_TYPE_RAM: self._write_ramcc,
            hpmdd.AG_TYPE_PD: self._write_pumped_ditch,
            hpmdd.AG_TYPE_IMP: self._write_agimp,
            hpmdd.AG_TYPE_EAA: self._write_eaa,
        }
        self._urban_methods = {
            hpmdd.URBAN_TYPE_IMPERV: self._write_imperv,
            hpmdd.URBAN_TYPE_MBRCELL: self._write_mbrcell,
            hpmdd.URBAN_TYPE_CU: self._write_cu,
            hpmdd.URBAN_TYPE_URBANDET: self._write_urbandet
        }

    def write_hpm(self):
        """Write the mesh data portion of the control file."""
        for i, mp in enumerate(self._prop_list):
            # if mp.hpm.hpm_type == hpmdd.HPM_TYPE_NONE:
            #     continue
            self._cur_hpm = mp.hpm
            self._cur_id_str = self._id_str[i]
            self._atts = {'id': f'{mp.index_value}'}
            if mp.label:
                self._atts['label'] = mp.label
            self._cur_hpm_xml = Et.SubElement(self._hpm_root_xml, 'hpmEntry', self._atts)
            if mp.hpm.hpm_type == hpmdd.HPM_TYPE_NS:
                self._ns_methods[self._cur_hpm.ns.type]()
            elif mp.hpm.hpm_type == hpmdd.HPM_TYPE_AG:
                self._ag_methods[self._cur_hpm.ag.ag_type]()
            else:  # elif mp.hpm.hpm_type == hpmdd.HPM_TYPE_URBAN:
                self._urban_methods[self._cur_hpm.urban.urban_type]()

    @cached_property
    def _water_body_label_id(self):
        """Get a lookup of water body label to id."""
        wblid = {}
        for k, v in self._wd.water_body_info.items():
            if k == 'canal_junction':
                continue
            for wbi in v:
                wblid[wbi.label] = wbi.wb_id
        return wblid

    def _write_layer1nsm(self):
        """Write a natural system hpm."""
        ns = self._cur_hpm.ns
        atts = {
            'kw': f'{ns.layer1.kw}',
            'rd': f'{ns.layer1.rd}',
            'xd': f'{ns.layer1.xd}',
            'pd': f'{ns.layer1.pd}',
            'kveg': f'{ns.layer1.kveg}',
            'fld_cap': f'{ns.layer1.fld_cap}',
            'imax': f'{ns.layer1.imax}'
        }
        Et.SubElement(self._cur_hpm_xml, 'layer1nsm', atts)

    def _write_unsat(self):
        """Write a natural system hpm."""
        ns = self._cur_hpm.ns
        atts = {
            'ew': f'{ns.unsat.ew}',
            'kw': f'{ns.unsat.kw}',
            'rd': f'{ns.unsat.rd}',
            'xthresh': f'{ns.unsat.xthresh}',
            'pthresh': f'{ns.unsat.pthresh}',
            'pd': f'{ns.unsat.pd}',
            'wilt': f'{ns.unsat.wilt}',
            'kveg': f'{ns.unsat.kveg}'
        }
        Et.SubElement(self._cur_hpm_xml, 'unsat', atts)

    def _write_nam(self):
        """Write a natural system hpm."""
        ns = self._cur_hpm.ns
        atts = {
            'etcoef': f'{ns.nam.etcoef}',
            'k0inf': f'{ns.nam.k0inf}',
            'imax': f'{ns.nam.imax}',
            'umax': f'{ns.nam.umax}',
            'lmax': f'{ns.nam.lmax}',
            'tof': f'{ns.nam.tof}',
            'tif': f'{ns.nam.tif}',
            'tg': f'{ns.nam.tg}',
            'cqof': f'{ns.nam.cqof}',
            'ckol': f'{ns.nam.ckol}',
            'ckif': f'{ns.nam.ckif}',
            'ckbf': f'{ns.nam.ckbf}'
        }
        Et.SubElement(self._cur_hpm_xml, 'nam', atts)

    def _write_layer5(self):
        """Write a natural system hpm."""
        ns = self._cur_hpm.ns
        atts = {
            'ew': f'{ns.layer5.ew}',
            'kw': f'{ns.layer5.kw}',
            'rd': f'{ns.layer5.rd}',
            'xd': f'{ns.layer5.xd}',
            'pd': f'{ns.layer5.pd}',
            'kveg': f'{ns.layer5.kveg}'
        }
        Et.SubElement(self._cur_hpm_xml, 'layer5', atts)

    def _write_layerpc(self):
        Et.SubElement(self._cur_hpm_xml, 'layerpc')

    def _write_ramcc(self):
        """Write the ramcc agriculture hpm."""
        self._ramcc = True
        self._write_afsirs()

    def _write_afsirs(self):
        """Write an agriculture hpm."""
        ag = self._cur_hpm.ag

        tag = 'afsirs' if not self._ramcc else 'ramcc'
        xml_afsirs = Et.SubElement(self._cur_hpm_xml, tag)
        xml_dict = {}
        if ag.afsirs.crop_label:
            xml_dict['label'] = ag.afsirs.crop_label
        if ag.afsirs.crop_id > 0:
            xml_dict['id'] = f'{ag.afsirs.crop_id}'
        xml_dict['j1'] = f'{ag.afsirs.j1}'
        xml_dict['jn'] = f'{ag.afsirs.jn}'
        xml_dict['depth1'] = f'{ag.afsirs.depth1}'
        xml_dict['depth2'] = f'{ag.afsirs.depth2}'
        xml_afcrop = Et.SubElement(xml_afsirs, 'afcrop', xml_dict)

        xml_kctbl = Et.SubElement(xml_afcrop, 'kctbl')
        xml_kctbl.text = ' '.join(f'{v[0]}' for v in ag.afsirs.kctbl)

        xml_awtbl = Et.SubElement(xml_afcrop, 'awtbl')
        xml_awtbl.text = ' '.join(f'{v[0]}' for v in ag.afsirs.awtbl)

        xml_dict = {}
        if ag.afsirs.afirr_label:
            xml_dict['label'] = ag.afsirs.afirr_label
        xml_dict['wtd'] = f"{ag.afsirs.wtd}"
        xml_afirr = Et.SubElement(xml_afsirs, 'afirr', xml_dict)
        xml_dict = {}
        if ag.afsirs.irrmeth_id > 0:
            xml_dict['id'] = f"{ag.afsirs.irrmeth_id}"
        xml_dict['eff'] = f"{ag.afsirs.eff}"
        xml_dict['arzi'] = f"{ag.afsirs.arzi}"
        xml_dict['exir'] = f"{ag.afsirs.exir}"
        _ = Et.SubElement(xml_afirr, 'irrmeth', xml_dict)
        _ = Et.SubElement(xml_afirr, 'irrmgmt', {'trigcode': f'{ag.afsirs.trig_code}'})

        xml_dict = {}
        if ag.afsirs.soil_label:
            xml_dict['label'] = ag.afsirs.soil_label
        xml_dict['depth'] = f"{ag.afsirs.depth}"
        xml_dict['minwc'] = f"{ag.afsirs.minwc}"
        xml_dict['maxwc'] = f"{ag.afsirs.maxwc}"
        xml_dict['cond'] = f'{ag.afsirs.cond}'
        _ = Et.SubElement(xml_afsirs, 'afsoil', xml_dict)

    def _write_pumped_ditch(self):
        """Write a pumped ditch agriculture hpm."""
        pd = self._cur_hpm.ag.pumped_ditch
        atts = {
            'rks': f'{pd.rks}',
            'percentarea': f'{pd.percent_area}',
            'fcPump': f'{pd.fc_pump}',
            'wsPump': f'{pd.ws_pump}',
            'bottom': f'{pd.bottom}',
            'biDirSeep': "0" if not pd.bi_dir_seep else "1"
        }
        if pd.runoff:
            atts['runoff'] = pd.runoff
        elem = Et.SubElement(self._cur_hpm_xml, 'pumpedditch', atts)
        ml = Et.SubElement(elem, 'maxLevel', {'name': 'max Level'})
        Et.SubElement(ml, 'const', {'value': f'{pd.max_level}'})
        min_lvl = Et.SubElement(elem, 'minLevel', {'name': 'min Level'})
        Et.SubElement(min_lvl, 'const', {'value': f'{pd.min_level}'})
        fcp_on = Et.SubElement(elem, 'fcPumpOn', {'name': 'fc pump ON'})
        Et.SubElement(fcp_on, 'const', {'value': f'{pd.fc_pump_on}'})
        fcp_off = Et.SubElement(elem, 'fcPumpOff', {'name': 'fc pump OFF'})
        Et.SubElement(fcp_off, 'const', {'value': f'{pd.fc_pump_off}'})
        wsp_on = Et.SubElement(elem, 'wsPumpOn', {'name': 'ws pump ON'})
        Et.SubElement(wsp_on, 'const', {'value': f'{pd.ws_pump_on}'})
        wsp_off = Et.SubElement(elem, 'wsPumpOff', {'name': 'ws pump OFF'})
        Et.SubElement(wsp_off, 'const', {'value': f'{pd.ws_pump_off}'})

    def _write_agimp(self):
        """Write an agricultural impoundment hpm."""
        ag = self._cur_hpm.ag.agimp
        atts = {
            'rks': f'{ag.rks}',
            'field_area': f'{ag.field_area}',
            'height': f'{ag.height}',
        }
        if ag.runoff:
            atts['runoff'] = ag.runoff
        elem = Et.SubElement(self._cur_hpm_xml, 'agimp', atts)
        atts = {
            'r25y3d': f'{ag.r25y3d}',
            'allow': f'{ag.allow}',
            's': f'{ag.s}',
        }
        Et.SubElement(elem, 'stdtriorf', atts)

    def _write_eaa(self):
        """Write an EAA runoff demand agriculture hpm."""
        ag = self._cur_hpm.ag.eaa_runoff_demand
        atts = {}
        if ag.label:
            atts['label'] = ag.label
        atts.update({
            'kmax': f'{ag.kmax}',
            'owpond': f'{ag.owpond}',
            'drz': f'{ag.drz}',
            'srz': f'{ag.srz}',
        })
        self._add_rule_curve_id_to_atts(ag.kcoef_id, atts, 'kcoef_id')
        self._add_rule_curve_id_to_atts(ag.kcalib_id, atts, 'kcalib_id')
        Et.SubElement(self._cur_hpm_xml, 'eaaRunoffDemand', atts)

    def _add_rule_curve_id_to_atts(self, rc_label, atts, att_name):
        if rc_label in self._rc_lookup:
            atts[att_name] = f'{self._rc_lookup[rc_label]}'
        else:
            msg = f'HPM EAA Runoff attribute "{att_name}" skipped because rule curve ("{rc_label}") not found. ' \
                  'HPM property associated with the following polygon(s):\n' \
                  '(polygon id, coverage name)\n' \
                  f'{self._cur_id_str}'
            self._wd.logger.error(msg)

    def _write_imperv(self):
        """Write an urban impervious hpm."""
        ip = self._cur_hpm.urban.imperv
        atts = {
            'dirconn': f'{ip.dirconn}',
            'isto': f'{ip.isto}',
            'isto1': f'{ip.isto1}',
            'sdet': f'{ip.sdet}',
            'sdet1': f'{ip.sdet1}'
        }
        Et.SubElement(self._cur_hpm_xml, 'imperv', atts)

    def _write_mbrcell(self):
        mbr = self._cur_hpm.urban.mbrcell
        atts = {}
        if mbr.route in self._water_body_label_id:
            atts['route'] = f'{self._water_body_label_id[mbr.route]}'
        else:
            msg = f'HPM urban mbrcell "route" attribute skipped because water body "{mbr.route}" not found. ' \
                  'HPM property associated with the following polygon(s):\n' \
                  '(polygon id, coverage name)\n' \
                  f'{self._cur_id_str}'
            self._wd.logger.error(msg)
        atts.update(
            {
                'tc': f'{mbr.tc}',
                'kveg': f'{mbr.kveg}',
                'd_shal': f'{mbr.d_shal}',
                'd_deep': f'{mbr.d_deep}',
                'fld_cap': f'{mbr.fld_cap}'
            }
        )
        Et.SubElement(self._cur_hpm_xml, 'mbrcell', atts)

    def _write_cu(self):
        cu = self._cur_hpm.urban.cu
        atts = {'label': cu.label, 'percentarea': f'{cu.percentarea}'}
        if cu.wsupply:
            atts['wsupply'] = cu.wsupply
        if cu.sewer:
            atts['sewer'] = cu.sewer
        if cu.trigcat != hpmdd.CU_TRIG_NA:
            atts['trigcat'] = cu.trigcat
        elem = Et.SubElement(self._cur_hpm_xml, 'cu', atts)
        Et.SubElement(elem, 'const', {'value': f'{cu.const_value}'})
        if cu.fracloss > 0.0:
            Et.SubElement(elem, 'sewer', {'fracloss': f'{cu.fracloss}'})
        if cu.septic:
            Et.SubElement(elem, 'septic')

    def _write_urbandet(self):
        ud = self._cur_hpm.urban.urbandet
        if ud.runoff:
            self._cur_hpm_xml.set('runoff', ud.runoff)
        if ud.percentarea > 0.0:
            self._cur_hpm_xml.set('percentarea', f'{ud.percentarea}')
        elem = Et.SubElement(self._cur_hpm_xml, 'urbandet', {'rks': f'{ud.rks}'})
        atts = {'wlen': f'{ud.wlen}', 'angle': f'{ud.angle}', 'top': f'{ud.top}', 'apx': f'{ud.apx}'}
        Et.SubElement(elem, 'vnotchweir', atts)
