"""Class for writing mesh properties to model input files."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import xml.etree.cElementTree as Et

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.rsm.data import mesh_data_def as mdd


class RainETWriter:
    """Writer class for the rain and et control file."""
    def __init__(self, option, rain_et, xml_parent, csv_writer, csv_fname, poly_ids_cov):
        """Constructor.

        Args:
            option (str): the option to write see mesh_data_def OPT_* constants
            rain_et (RainEt): the rain/et data class
            xml_parent (xml.etree.cElementTree.SubElement): mesh item in the xml
            csv_writer (CsvWriter): csv writer
            csv_fname (str): csv file name
            poly_ids_cov (str): poly ids and coverage name used in error messages
        """
        self._option = option
        self._rain_et = rain_et
        self._xml_parent = xml_parent
        self._csv_writer = csv_writer
        self._csv_fname = csv_fname
        self._poly_ids_cov = poly_ids_cov

    def write(self):
        """Write the rain/et portion of the control file."""
        elem = self._xml_parent
        if self._option == mdd.OPT_CURVE:
            if not self._rain_et.curve:
                msg = (
                    'Error writing rain/et curve. The curve is not defined for the following feature(s):\n'
                    '(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            self._csv_writer.set_desired_filename(f'{self._csv_fname}.csv')
            atts = self._csv_writer.write_csv_file(self._rain_et.curve)
            Et.SubElement(elem, 'csv', atts)
        elif self._option == mdd.OPT_DSS:
            if not os.path.isfile(self._rain_et.dss_file):
                msg = (
                    f'Error writing rain/et DSS file. The file does not exist:\n'
                    f'{self._rain_et.dss_file}\n'
                    f'For the following feature(s):\n'
                    f'(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            if not self._rain_et.dss_path:
                msg = (
                    f'Error writing rain/et DSS path. The path is not defined for the following feature(s):\n'
                    f'(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            atts = {
                'file': f'{self._rain_et.dss_file}',
                'pn': f'{self._rain_et.dss_path}',
            }
            if self._rain_et.dss_mult != 1.0:
                atts['mult'] = f'{self._rain_et.dss_mult}'
            if self._rain_et.dss_units:
                atts['units'] = f'{self._rain_et.dss_units}'
            if self._rain_et.dss_type != mdd.OPT_NOT_SPECIFIED:
                atts['type'] = f'{self._rain_et.dss_type}'
            Et.SubElement(elem, 'dss', atts)
        elif self._option == mdd.OPT_GRID_FILE:
            if not os.path.isfile(self._rain_et.grid_file):
                msg = (
                    f'Error writing rain/et GRID file. The file does not exist:\n'
                    f'{self._rain_et.grid_file}\n'
                    f'For the following feature(s):\n'
                    f'(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            atts = {
                'file': f'{self._rain_et.grid_file}',
                'xorig': f'{self._rain_et.grid_file_x}',
                'yorig': f'{self._rain_et.grid_file_y}',
            }
            if self._rain_et.grid_file_mult != 1.0:
                atts['mult'] = f'{self._rain_et.grid_file_mult}'
            if self._rain_et.grid_file_dbintl > 0:
                atts['dbintl'] = f'{self._rain_et.grid_file_dbintl}'
            Et.SubElement(elem, 'gridio', atts)
        elif self._option == mdd.OPT_NETCDF_FILE:
            if not os.path.isfile(self._rain_et.netcdf_file):
                msg = (
                    f'Error writing rain/et NETCDF file. The file does not exist:\n'
                    f'{self._rain_et.grid_file}\n'
                    f'For the following feature(s):\n'
                    f'(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            if not self._rain_et.netcdf_variable:
                msg = (
                    f'Error writing rain/et NETCDF variable. The variable is not defined for the following '
                    f'feature(s):\n(polygon id, coverage name)\n'
                    f'{self._poly_ids_cov}'
                )
                raise ValueError(msg)
            atts = {
                'file': f'{self._rain_et.netcdf_file}',
                'variable': f'{self._rain_et.netcdf_variable}',
            }
            if self._rain_et.netcdf_mult != 1.0:
                atts['mult'] = f'{self._rain_et.netcdf_mult}'
            if self._rain_et.netcdf_dbintl > 0:
                atts['dbintl'] = f'{self._rain_et.netcdf_dbintl}'
            if self._rain_et.netcdf_units:
                atts['units'] = f'{self._rain_et.netcdf_units}'
            Et.SubElement(elem, 'netcdf', atts)
        else:
            raise RuntimeError('Unknown option for rain/et property.')
