"""Module for ReaderBase class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['ReaderBase']

# 1. Standard Python modules
from pathlib import Path
import re
from typing import Any, Optional, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.feedback_thread import ExpectedError

# 4. Local modules


class ReaderBase:
    """
    A base class for reading files.
    """
    def __init__(self, path: Path | str, comment_markers: list[str]):
        """
        Initialize the reader.

        Args:
            path: The file to be read.
            comment_markers: List of strings to consider as comment markers. When reading a line, the first comment
                marker and everything after it will be discarded.
        """
        self._line_number = 0
        self._path = Path(path)
        self._file: TextIO
        self._comment_markers = comment_markers
        self._pattern = re.compile('[dD]')

    def __enter__(self):
        """Open the file."""
        self._line_number = 0
        self._file = open(self._path, 'r')
        return self

    def __exit__(self, _exc_type, _exc_value, _exc_tb):
        """Close the file."""
        self._file.close()

    def _read_line(self, comment_markers: Optional[list[str] | str] = None) -> str:
        """
        Read a line out of the file.

        Args:
            comment_markers: List of strings to consider comment markers. Anything after the first comment marker will
                be discarded. If omitted, uses the comment markers provided to the constructor.


        Returns:
            The next line in the file.
        """
        self._line_number += 1
        line = self._file.readline()
        if line == '':
            raise self._error('Unexpected end-of-file.')

        if comment_markers is None:
            comment_markers = self._comment_markers

        for comment_marker in comment_markers:
            line = line.split(comment_marker)[0]

        return line

    def _parse_line(self, *types) -> Any:
        """
        Parse a line.

        The line is read using `self._read_line()`, then split on whitespace. Each piece is passed to one of the type
        constructors in `*args` to construct a parsed type. Then the parsed values are returned.

        Args:
            *types: One or more type constructors, e.g. `int` or `bool`.

        Returns:
            If `len(types)` == 1, returns a single value constructed by the constructor in `types`. Otherwise, returns
            a list of values, with types in the same order as `types`.
        """
        line = self._read_line()

        pieces = line.split()

        for i in range(min(len(pieces), len(types))):
            if types[i] is float:
                pieces[i] = self._pattern.sub('e', pieces[i], count=1)

        try:
            parsed_pieces = [t(p) for t, p in zip(types, pieces)]
        except ValueError:  # Something didn't parse
            expected_list = [t.__name__ for t in types]
            expected_message = ', '.join(expected_list)
            raise self._error(f'Expected {expected_message}.')

        if len(parsed_pieces) < len(types):
            # SCHISM's parser is lazy and ignores extra stuff at the end of a line, even if it's uncommented.
            # That isn't a great design, but the test files we received make use of that, so we support it too.
            expected_list = [t.__name__ for t in types]
            expected_message = ', '.join(expected_list)
            raise self._error(f'Expected {expected_message}.')

        if len(parsed_pieces) > 1:
            return parsed_pieces
        else:
            return parsed_pieces[0]

    def _error(self, reason: str, line_number: int = 0) -> ExpectedError:
        """
        Get an exception that reports an error.

        Args:
            reason: The reason for the error, e.g. "The thing was an invalid value".
            line_number: The line number to report the error on. If 0 (the default), uses the current line.

        Returns:
            An exception that can be raised to report an error.
        """
        line_number = line_number or self._line_number
        return ExpectedError(f'Error on line {line_number}: {reason}')
