"""Create pressure datasets from 3D Bridge coverages."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.extractor.ugrid_2d_data_extractor import UGrid2dDataExtractor

# 4. Local modules


class CeilingGenerator:
    """Create pressure datasets from 3D Bridge coverages."""
    def __init__(self, target_ugrid, bridge_data, ceiling_file_name, grid_units):
        """Initializes the class.

        Args:
            target_ugrid (:obj:`xms.grid.ugrid.UGrid`): Geometry to generate the dataset for
            bridge_data (:obj:`list[xms.grid.ugrid.UGrid]`): Bridge bottom UGrids.
            ceiling_file_name (:obj:`str`): file to write ceiling data
            grid_units (:obj:`str`): Units for the grid (METERS or FEET)
        """
        self._target_ugrid = target_ugrid
        self._grid_units = grid_units
        self._bridge_data = bridge_data
        self._ceiling_file_name = ceiling_file_name
        self._logger = logging.getLogger('xms.srh')
        self._target_ugrid_cell_centers = [target_ugrid.get_cell_centroid(x)[1] for x in range(target_ugrid.cell_count)]
        self._have_valid_bridge = False
        self._ceiling_list = []
        self._bridge_idx = 0

    def generate(self):
        """Generates the srh ceiling file from the inputs."""
        # Add the input 3D Bridge coverage data.
        for idx, bridge in enumerate(self._bridge_data):
            self._logger.info(
                f'Interpolating 3D Bridge ceiling elevations to mesh ({idx + 1} of {len(self._bridge_data)})...'
            )
            self._add_bridge(bridge)
        self._write_ceiling_file()

    def _add_bridge(self, bridge):
        """Add a 3D Bridge as an interpolation input.

        Args:
            bridge (:obj:`(UGrid, float)`): UGrid surface of bridge bottom and manning n value
        """
        self._bridge_idx += 1
        bridge_bottom = bridge[0]
        ug_elevations = [pt[2] for pt in bridge_bottom.locations]
        ug_activity = [1] * len(ug_elevations)
        extractor = UGrid2dDataExtractor(bridge_bottom)
        extractor.set_grid_point_scalars(ug_elevations, ug_activity, 'points')
        extractor.extract_locations = self._target_ugrid_cell_centers
        out_elev = extractor.extract_data()
        for idx, elev in enumerate(out_elev):
            if not math.isnan(elev):
                # if self._grid_units.endswith('FOOT"'):
                #     elev = elev * 0.3048
                self._ceiling_list.append((idx + 1, elev, self._bridge_idx, bridge[1]))

    def _write_ceiling_file(self):
        """Writes out the ceiling file."""
        grid_units = self._grid_units.replace('"', '')
        with open(self._ceiling_file_name, 'w') as f:
            f.write('SRHPRESSURE 41\n')
            f.write(f'{grid_units}\n')
            for item in self._ceiling_list:
                f.write(f'{item[0]} {item[1]} {item[2]} {item[3]}\n')
