"""ObstructionComponent class. Data for obstruction Coverage."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest
from xms.components.display.display_options_io import read_display_option_ids
from xms.core.filesystem import filesystem
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.srh.components.srh_cov_component import SrhCoverageComponent
from xms.srh.data.obstruction_data import ObstructionData
from xms.srh.gui.obstruction_dialog import run_dlg

INITIAL_OBS_PT_ATT_ID_FILE = 'initial_obs_pt_att.ids'
INITIAL_OBS_PT_COMP_ID_FILE = 'initial_obs_pt_comp.ids'
INITIAL_OBS_ARC_ATT_ID_FILE = 'initial_obs_arc_att.ids'
INITIAL_OBS_ARC_COMP_ID_FILE = 'initial_obs_arc_comp.ids'


class ObstructionComponent(SrhCoverageComponent):
    """A hidden Dynamic Model Interface (DMI) component for the SRH-2D model simulation."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.

        """
        super().__init__(main_file)
        self.data = ObstructionData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        # [(menu_text, menu_method)...]
        self.arc_commands = [
            ('Assign Arc Obstruction', 'open_assign_arc_obstruction'),
        ]
        self.point_commands = [
            ('Assign Point Obstruction', 'open_assign_point_obstruction'),
        ]

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            data = ObstructionData(new_main_file)
            data.load_all()
            data.info.attrs['cov_uuid'] = ''
            data.commit()

        return new_main_file, messages, action_requests

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (:obj:`bool`): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        self.data.load_all()
        self.data.commit()

        action = ActionRequest(
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            main_file=self.main_file,
            method_name='get_initial_display_options'
        )
        return [], [action]

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid

        # Initialize component ids if needed (created from a model native read).
        initial_pt_att_file = os.path.join(os.path.dirname(self.main_file), INITIAL_OBS_PT_ATT_ID_FILE)
        if os.path.isfile(initial_pt_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_pt_att_file)
            initial_pt_comp_file = os.path.join(os.path.dirname(self.main_file), INITIAL_OBS_PT_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_pt_comp_file)
            filesystem.removefile(initial_pt_att_file)
            filesystem.removefile(initial_pt_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.point, att_id, comp_id)
        initial_arc_att_file = os.path.join(os.path.dirname(self.main_file), INITIAL_OBS_ARC_ATT_ID_FILE)
        if os.path.isfile(initial_arc_att_file):  # Came from a model native read, initialize the component ids.
            att_ids = read_display_option_ids(initial_arc_att_file)
            initial_arc_comp_file = os.path.join(os.path.dirname(self.main_file), INITIAL_OBS_ARC_COMP_ID_FILE)
            comp_ids = read_display_option_ids(initial_arc_comp_file)
            filesystem.removefile(initial_arc_att_file)
            filesystem.removefile(initial_arc_comp_file)
            for att_id, comp_id in zip(att_ids, comp_ids):
                self.update_component_id(TargetType.arc, att_id, comp_id)

        self.data.commit()
        return [], []

    def open_assign_arc_obstruction(self, query, params, win_cont):
        """Opens the Assign Obstruction dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        _kwargs = params[0]
        arc_ids = _kwargs.get('selection', [])
        if len(arc_ids) == 0:
            return [('INFO', 'No arcs selected. Select one or more arcs to assign properties.')], []

        # get the component ids
        id_files = _kwargs.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'ARC': (id_files[0], id_files[1])})

        run_dlg(arc_ids, TargetType.arc, self, win_cont)

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)
        return [], []

    def open_assign_point_obstruction(self, query, params, win_cont):
        """Opens the Assign Obstruction dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.

        """
        check_uuid = self.ensure_cov_uuid(query)
        if check_uuid != ([], []):
            return check_uuid
        _kwargs = params[0]
        pt_ids = _kwargs.get('selection', [])
        if len(pt_ids) == 0:
            return [('INFO', 'No points selected. Select one or more points to assign properties.')], []

        # get the component ids
        id_files = _kwargs.get('id_files', [])
        if id_files is not None and len(id_files) > 1:
            self.load_coverage_component_id_map({'POINT': (id_files[0], id_files[1])})

        run_dlg(pt_ids, TargetType.point, self, win_cont)

        # Delete the id dumped by xms files.
        shutil.rmtree(os.path.join(os.path.dirname(self.main_file), 'temp'), ignore_errors=True)
        return [], []
