"""Helper functions for param.Parameterized classes."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO

# 2. Third party modules
import pandas as pd
import param

# 3. Aquaveo modules
from xms.guipy.param import param_util  # noqa: I202

# 4. Local modules


def orjson_dict_to_param_cls(pdict, cls, df_handler):
    """Changes param values from a json list.

    Args:
        pdict (:obj:`dict`): json dict of param data
        cls (:obj:`param.Parameterized`): class with param objects
        df_handler (:obj:`method`): method to handle DataFrame

    """
    for key, val in pdict.items():
        if not hasattr(cls, key):  # pragma: no cover
            continue  # pragma: no cover
        obj = getattr(cls, key)
        obj_type = type(obj) if obj is not None else None
        if obj_type is not None and issubclass(obj_type, param.Parameterized):
            orjson_dict_to_param_cls(val, obj, df_handler)
        else:
            if type(obj) is pd.DataFrame:
                if df_handler is not None:
                    df = df_handler(val)
                    if df is not None:
                        df.columns = obj.columns
                    else:
                        df = obj
                else:
                    df = pd.read_json(StringIO(val))
                    df.sort_index(inplace=True)
                    for col in obj.columns:
                        if col in df.columns:
                            df = df.astype({col: 'float'})
                setattr(cls, key, df)
            else:
                if obj_type is not None:
                    setattr(cls, key, obj_type(val))


def param_cls_to_orjson_dict(cls, df_handler, skip_negative_precedence=True):
    """List of data the will write to json.

    Args:
        cls (:obj:`param.Parameterized`): class with param objects
        df_handler (:obj:`method`): method to handle DataFrame
        skip_negative_precedence (:obj:`bool`): skip parameters with negative precedence

    Return:
        (:obj:`dict`): param name, value pairs
    """
    pnames, params = param_util.names_and_params_from_class(cls)
    class_obj_dict = cls.param.objects(instance=False)
    plist = []
    for i in range(len(pnames)):
        plist.append((class_obj_dict[pnames[i]].precedence, pnames[i], params[i]))
    plist.sort()

    output = {}
    for i in range(len(plist)):
        pname = plist[i][1]
        obj = plist[i][2]  # par
        if type(obj) is param.Action:
            continue
        if skip_negative_precedence and hasattr(obj, 'precedence') and obj.precedence < 0:
            continue
        val = getattr(cls, pname)
        if issubclass(type(val), param.Parameterized):
            output[pname] = param_cls_to_orjson_dict(val, df_handler, skip_negative_precedence)
        else:
            if type(obj) is param.DataFrame:
                if df_handler is not None:
                    df_out = df_handler(val)
                else:
                    df_out = val.to_json()
                output[pname] = df_out
            else:
                output[pname] = val
    return output
