"""Write the model control portion of the *.srhhydro file."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os.path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.file_io.hydro_sediment_writer import HydroSedimentWriter


class HydroModelControlWriter:
    """Writes SRH-2D model control data to the hydro file."""
    def __init__(self, file, model_control, file_list, grid_units='GridUnit "FOOT"', logger=None):
        """Constructor.

        Args:
            file: The output *.srhhydro file stream
            model_control (:obj:`ModelControl`): The simulation data
            file_list (:obj:`list[str]`): Additional files that need to be exported
            grid_units (:obj:`str`): units of the grid geometry
            num_ceiling_zones (:obj:`int`): number of ceiling zones
            logger (:obj:`logging.Logger`): Logger to use
        """
        self._file = file
        self._model_control = model_control
        self._file_list = file_list
        self._grid_units = grid_units
        self._run_name = None
        self._num_ceiling_zones = 1
        self.logger = logger

    def write(self, run_name=None):
        """Write the *.srhhydro file.

        Args:
            run_name (:obj:`str`): Name of the run if this is a parameter run model
        """
        self._run_name = run_name
        self._write_model_control(self._file)

    def _write_model_control(self, file):
        """Write the model control data to the hydro file.

        Args:
            file: The output hydro file stream
        """
        mc = self._model_control
        file.write(f'Case "{mc.hydro.case_name}"\n')
        file.write(f'Description "{mc.hydro.simulation_description}"\n')
        if not mc.enable_sediment:
            file.write('RunType FLOW\n')
        else:
            file.write('RunType MOBILE\n')
        file.write('ModelTemp OFF\n')
        if not mc.enable_sediment:
            if mc.advanced.unsteady_output:
                file.write('UnsteadyOutput UNSTEADY\n')
            else:
                file.write('UnsteadyOutput STEADY\n')
        file.write(f'SimTime {mc.hydro.start_time} {mc.hydro.time_step} {mc.hydro.end_time}\n')
        file.write(f'TurbulenceModel {mc.advanced.turbulence_model.upper()}\n')
        if mc.advanced.turbulence_model == 'Parabolic':
            file.write(f'ParabolicTurbulence {mc.advanced.parabolic_turbulence}\n')
        deprecated_items = ['Laminar', 'Constant']
        if mc.advanced.turbulence_model in deprecated_items:
            msg = f'Turbulence model using a deprecated option: {mc.advanced.turbulence_model}. ' \
                  f'Change to a non-deprecated option in the Advanced tab of the Model Control dialog.'
            self.logger.error(msg)

        sed_writer = HydroSedimentWriter(file, mc, self.logger)
        sed_writer.write()
        self._write_initial_condition(file)
        for item in self._file_list:
            items = item.split(maxsplit=1)
            card = items[0]
            filename = items[1]
            if filename.lower().endswith(' srhmat'):
                filename = filename[:-len(' srhmat')]
            filename = filename.strip('"')  # Make sure we don't add quotes twice
            card2 = ''
            if 'subsurfacebedfile' == card.lower():
                card2 = ' SRHMAT'
            elif 'pressuredatasetfile' == card.lower():
                self._calc_num_ceiling_zones(filename)
                card2 = f' {self._num_ceiling_zones} SRHCEILING'
            file.write(f'{card} "{filename}"{card2}\n')
        self._write_output(file)

    def _calc_num_ceiling_zones(self, filename):
        """Determine the number of ceiling zones in the SRHCEILING file.

        Args:
          filename (:obj:`str`): Ceiling filename
        """
        path = os.path.dirname(self._file.name)
        out_file = os.path.join(path, filename)
        with open(out_file, 'r') as f:
            lines = f.readlines()
            vals = lines[-2].split()
            if len(vals) == 4:
                self._num_ceiling_zones = int(vals[2])

    def _write_initial_condition(self, file):
        """Write the initial conditions to the hydro file.

        Args:
            file: The output hydro file stream
        """
        mc = self._model_control
        if mc.hydro.initial_condition == 'Dry':
            file.write('InitCondOption DRY\n')
        elif mc.hydro.initial_condition == 'Automatic':
            file.write('InitCondOption AUTO\n')
        elif mc.hydro.initial_condition == 'Initial Water Surface Elevation':
            file.write('InitCondOption WSE\n')
            file.write('NumInitZones 1\n')
            units_str = 'EN'
            if mc.hydro.initial_water_surface_elevation_units == 'Meters':
                units_str = 'SI'
            file.write(f'InitZoneParams 1 0 0 {mc.hydro.initial_water_surface_elevation} {units_str}\n')
        elif mc.hydro.initial_condition == 'Restart File':
            file.write('InitCondOption RST\n')
            file.write(f'RestartFile "{mc.hydro.case_name}_RST1.dat"\n')
        elif mc.hydro.initial_condition == 'Water Surface Elevation Dataset':
            unit_str = 'SI' if self._grid_units.endswith('"METER"') else 'EN'
            file.write('InitCondOption Vary_WE\n')
            file_name = 'wse.2dm'
            file.write(f'WSE_Init "{file_name}" {unit_str}\n')

    def _write_output(self, file):
        """Write the output control to the hydro file.

        Args:
            file: The output hydro file stream
        """
        mc = self._model_control
        out_unit = 'EN' if mc.output.output_units == 'English' else 'SI'
        format_str = mc.output.output_format if mc.output.output_format != 'XMDF' else 'XMDFC'
        file.write(f'OutputFormat {format_str} {out_unit}\n')
        output_option = 1
        if mc.output.output_method == 'Specified Times':
            output_option = 2
        if mc.output.output_method == 'Simulation End':
            output_option = 3
        file.write(f'OutputOption {output_option}\n')
        if output_option == 1:
            # output frequency must be in hours
            val = mc.output.output_frequency
            if mc.output.output_frequency_units == 'Days':
                val = val * 24
            elif mc.output.output_frequency_units == 'Minutes':
                val = val / 60
            elif mc.output.output_frequency_units == 'Seconds':
                val = val / (60 * 60)
            file.write(f'OutputInterval {val}\n')
        elif output_option == 2:
            times = mc.output.output_specified_times['Times'].tolist()
            file.write('OutputTimes')
            for t in times:
                file.write(f' {t}')
            file.write('\n')
        # elif output_option == 3:
        #     pass  # don't write anything
        if mc.output.output_maximum_datasets:
            file.write('MaxDatasetParams')
            if mc.output.specify_maximum_dataset_time_range:
                file.write(f' {mc.output.maximum_dataset_minimum_time} {mc.output.maximum_dataset_maximum_time}')
            else:
                file.write(f' 0.0 {mc.hydro.end_time}')
            file.write('\n')
