"""BcCoverageLooper class."""

# 1. Standard Python modules
import json

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa AQU103

# 4. Local modules
from xms.srh.components.bc_component import BcComponent
from xms.srh.data.bc_data import BcData
from xms.srh.file_io.report import plots, report_util
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase, label_from_bc_type
from xms.srh.floodway.xms_getter import XmsGetter
from xms.srh.manning_n.manning_n_calc import get_manning_n_calc_data
from xms.srh.manning_n.manning_n_data import ManningNData


class BcCoverageLooper(CoverageLooperBase):
    """Adds BC coverages to the report."""
    def __init__(self, notes_db, report_dir, query, mesh_boundaries, coverage_meshes, logger):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            mesh_boundaries (:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.
            coverage_meshes (:obj:`dict{bc coverage UUID: mesh UUID}`): BC coverages and their meshes.
            logger(:obj:`logger`): The logger.
        """
        super().__init__(
            notes_db, report_dir, query, 'Boundary Conditions', 'Bc_Component', 'SRH-2D', mesh_boundaries,
            coverage_meshes
        )
        self._logger = logger

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return BcComponent(main_file)

    def _load_component_ids(self, bc_component):  # pragma: no cover
        """Loads the feature arc/point IDs into the component.

        Args:
            bc_component (:obj:`BcComponent`): The bc component to load the IDs into.
        """
        self._query.load_component_ids(bc_component, arcs=True)

    @staticmethod
    def _get_constant(bc, option, constant, bc_jinja):  # pragma: no cover
        """Convenience function to fill the constant value jinja, if the option is constant.

        Args:
            bc: The BC class (e.g. BcDataInletQ).
            option (:obj:`str`): The param option (e.g. discharge_option) in the bc class that may or may
                not be constant.
            constant (:obj:`str`): The param constant value which will be the value in the jinja dict
            bc_jinja(:obj:`dict`): The jinja dict.
        """
        label = ''
        p = bc.param
        if p:
            c = getattr(p, constant)
            if c:
                label = c.label
        if label:
            if getattr(bc, option) == 'Constant':
                bc_jinja[label] = NumberCorrector.format_double(getattr(bc, constant))
            else:
                bc_jinja[label] = ''

    @staticmethod
    def _add_arc_ids_if_structure(bc_base, bc_id, structure_bc_ids, bc_component, bc_jinja):  # pragma: no cover
        """Adds the structure's arc ids to the jinja if the BC is a structure.

        Args:
            bc_base:
            bc_id (:obj:`int`): BC id.
            structure_bc_ids (:obj:`set{int}`): Set of BC ids so we don't repeat the BC for both arcs, just one.
            bc_component (:obj:`BcComponent`): The BC component.
            bc_jinja (:obj:`dict`): Dict with the BC jinja data.

        Returns:
            (:obj:`bool`): True if we should keep going, False if we've seen this structure before and should abort.
        """
        if bc_base.bc_type not in BcData.structures_list:
            return True  # Not a structure

        # Skip this arc if we've already seen it via it's companion arc
        if bc_id in structure_bc_ids:
            return False
        structure_bc_ids.add(bc_id)

        # Get the arc IDs
        bc_base.arcs = bc_component.data.structure_param_from_id(bc_id)  # arc ids are component ids
        if bc_base.arcs.arc_id_0 < 0 or bc_base.arcs.arc_id_1 < 0:
            return False
        arc_id_0 = bc_component.get_xms_ids(TargetType.arc, bc_base.arcs.arc_id_0)[0]
        arc_id_1 = bc_component.get_xms_ids(TargetType.arc, bc_base.arcs.arc_id_1)[0]

        # Add arc IDs to jinja
        if bc_base.arcs.arc_option_0 == 'Upstream':
            bc_jinja['arc_upstream'] = arc_id_0
            bc_jinja['arc_downstream'] = arc_id_1
        else:
            bc_jinja['arc_upstream'] = arc_id_1
            bc_jinja['arc_downstream'] = arc_id_0

        return True

    def _store_coverage_data(self, coverage, bc_component):  # pragma: no cover  # noqa: C901  (too complex)
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            bc_component (:obj:`BcComponent`): The bc component.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(
            coverage, bc_component, self._notes_db, self._report_dir, self._logger, self._coverage_meshes,
            self._mesh_boundaries, self._query
        )

    @staticmethod
    def get_jinja(
        coverage, bc_component, notes_db, report_dir, logger, coverage_meshes, mesh_boundaries, query
    ):  # pragma: no cover  # noqa: C901  (too complex)
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            bc_component (:obj:`BcComponent`): The bc component.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Filepath to directory where report files will be saved.
            logger(:obj:`logger`): The logger, or None.
            coverage_meshes (:obj:`dict{coverage UUID: mesh UUID}`): Coverages and their meshes.
            mesh_boundaries (:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS

        Returns:
            (:obj:`dict`):A dict of the coverage data for use with jinja.
        """
        if logger:
            logger.info(f'Getting data for BC coverage "{coverage.name}"')

        # Store all jinja stuff in a dict
        coverage_jinja = {}

        # Group things into lists by type
        # BCs
        inlet_q_jinja = []
        exit_h_jinja = []
        exit_q_jinja = []
        inlet_sc_jinja = []
        exit_ex_jinja = []
        wall_jinja = []
        symmetry_jinja = []
        internal_sink_jinja = []
        # Hydraulic structures
        culverts_hy8_jinja = []
        culverts_jinja = []
        weirs_jinja = []
        pressure_jinja = []
        gates_jinja = []
        links_jinja = []

        # Go through arcs storing bc data for the report
        coverage_jinja['name'] = coverage.name
        report_util.add_object_notes(notes_db, coverage.uuid, coverage_jinja)
        plot_bc_atts = {}  # Also store the data by arc id for convenience in creating the plot
        arcs = coverage.arcs
        structure_bc_ids = set()  # Set used to prevent repeating the 2nd arc of structures
        arc_ids_to_bc_ids = {}  # Dict of arc ids and the associated bc_id
        for arc in arcs:
            arc_id = arc.id
            comp_id = bc_component.get_comp_id(TargetType.arc, arc_id)
            bc_id = bc_component.data.bc_id_from_comp_id(comp_id)
            arc_ids_to_bc_ids[arc_id] = bc_id
            bc_base = bc_component.data.bc_data_param_from_id(bc_id)
            label = bc_base.label
            if not label:
                label = label_from_bc_type(bc_base.bc_type)
            bc_jinja = {'Arc ID': str(arc_id), 'Type': label}

            # If it's a structure, get the arc IDs
            go_on = BcCoverageLooper._add_arc_ids_if_structure(bc_base, bc_id, structure_bc_ids, bc_component, bc_jinja)
            if not go_on:
                continue  # We've already seen this structure BC's other arc. Don't repeat the BC.

            # Boundary conditions
            if bc_base.bc_type == 'Inlet-Q (subcritical inflow)':
                bc = bc_base.inlet_q
                bc_jinja[bc.param.discharge_option.label] = bc.discharge_option
                BcCoverageLooper._get_constant(bc, 'discharge_option', 'constant_q', bc_jinja)
                bc_jinja[bc.param.distribution_at_inlet.label] = bc.distribution_at_inlet
                inlet_q_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Exit-H (subcritical outflow)':
                bc = bc_base.exit_h
                bc_jinja[bc.param.water_surface_elevation_option.label] = bc.water_surface_elevation_option
                BcCoverageLooper._get_constant(bc, 'water_surface_elevation_option', 'constant_wse', bc_jinja)
                if bc.manning_n_calculator_inputs:
                    BcCoverageLooper._fill_channel_calculator_data(
                        arc_id, bc, bc_jinja, coverage, query, report_dir, logger
                    )
                exit_h_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Exit-Q (known-Q outflow)':
                bc = bc_base.exit_q
                bc_jinja[bc.param.discharge_option.label] = bc.discharge_option
                BcCoverageLooper._get_constant(bc, 'discharge_option', 'constant_q', bc_jinja)
                exit_q_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Inlet-SC (supercritical inflow)':
                bc = bc_base.inlet_sc
                bc_jinja[bc.param.discharge_q_option.label] = bc.discharge_q_option
                BcCoverageLooper._get_constant(bc, 'discharge_q_option', 'constant_q', bc_jinja)
                BcCoverageLooper._get_constant(bc, 'discharge_q_option', 'constant_wse', bc_jinja)
                bc_jinja[bc.param.distribution_at_inlet.label] = bc.distribution_at_inlet
                inlet_sc_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Exit-EX (supercritical outflow)':
                exit_ex_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Wall (no-slip boundary)':
                wall_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Symmetry (slip boundary)':
                symmetry_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Internal sink':
                bc = bc_base.internal_sink
                bc_jinja[bc.param.sink_flow_type.label] = bc.sink_flow_type
                BcCoverageLooper._get_constant(bc, 'sink_flow_type', 'constant_q', bc_jinja)
                internal_sink_jinja.append(bc_jinja)

            # Hydraulic structures
            elif bc_base.bc_type == 'Culvert HY-8':
                culverts_hy8_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Culvert':
                bc_jinja['culvert_invert_elevation'] = bc_base.culvert.invert_elevation
                bc_jinja['culvert_barrel_length'] = bc_base.culvert.barrel_length
                bc_jinja['culvert_barrel_count'] = bc_base.culvert.num_barrels
                culverts_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Link':
                bc_jinja['link_type'] = bc_base.link.inflow_type
                bc_jinja['link_length'] = bc_base.link.conduit_length
                bc_jinja['link_diameter'] = bc_base.link.conduit_diameter
                links_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Weir':
                bc_jinja['weir_type'] = bc_base.weir.type.type
                bc_jinja['weir_length'] = bc_base.weir.length
                bc_jinja['weir_elevation'] = bc_base.weir.crest_elevation
                weirs_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Pressure':
                bc_jinja['ceiling_type'] = bc_base.pressure.ceiling_type
                bc_jinja['upstream_elevation'] = bc_base.pressure.upstream_elevation
                bc_jinja['downstream_elevation'] = bc_base.pressure.downstream_elevation
                bc_jinja['roughness'] = bc_base.pressure.roughness
                pressure_jinja.append(bc_jinja)
            elif bc_base.bc_type == 'Gate':
                bc_jinja['gate_type'] = bc_base.gate.type.type
                bc_jinja['crest_elevation'] = bc_base.gate.crest_elevation
                bc_jinja['height'] = bc_base.gate.height
                bc_jinja['width'] = bc_base.gate.width
                bc_jinja['contract_coefficient'] = bc_base.gate.contract_coefficient
                gates_jinja.append(bc_jinja)

            else:
                pass

            plot_bc_atts[bc_id] = bc_jinja

        # BCs
        coverage_jinja['inlet_q'] = inlet_q_jinja
        coverage_jinja['exit_h'] = exit_h_jinja
        coverage_jinja['exit_q'] = exit_q_jinja
        coverage_jinja['inlet_sc'] = inlet_sc_jinja
        coverage_jinja['exit_ex'] = exit_ex_jinja
        coverage_jinja['wall'] = wall_jinja
        coverage_jinja['symmetry'] = symmetry_jinja
        coverage_jinja['internal_sink'] = internal_sink_jinja
        # Hydraulic structures
        coverage_jinja['culverts_hy8'] = culverts_hy8_jinja
        coverage_jinja['culverts'] = culverts_jinja
        coverage_jinja['weirs'] = weirs_jinja
        coverage_jinja['pressure'] = pressure_jinja
        coverage_jinja['gates'] = gates_jinja
        coverage_jinja['links'] = links_jinja

        coverage_jinja['bcs_arc_count'] = (
            len(inlet_q_jinja) + len(exit_h_jinja) + len(exit_q_jinja)  # noqa: W503
            + len(inlet_sc_jinja) + len(exit_ex_jinja) + len(wall_jinja)  # noqa: W503
            + len(symmetry_jinja) + len(internal_sink_jinja)  # noqa: W503
        )  # noqa: W503
        coverage_jinja['structures_arc_count'] = (
            len(culverts_hy8_jinja) + len(culverts_jinja)  # noqa: W503
            + len(weirs_jinja) + len(pressure_jinja) + len(gates_jinja)  # noqa: W503
        ) + len(links_jinja)  # noqa: W503

        # Draw the plot, including the mesh boundary
        mesh_boundary = CoverageLooperBase.mesh_boundary_from_coverage(coverage.uuid, coverage_meshes, mesh_boundaries)
        if logger:
            logger.info('Creating plot of BC coverage geometry.')
        coverage_jinja['plot'] = plots.plot_bc_coverage(
            coverage, plot_bc_atts, arc_ids_to_bc_ids, mesh_boundary, report_dir
        )

        return coverage_jinja

    @staticmethod
    def _fill_channel_calculator_data(arc_id, bc, bc_jinja, coverage, query, report_dir, logger):  # pragma: no cover
        """Fills in jinja info about the channel calculator.

        Args:
            arc_id (:obj:`int`): The arc id.
            bc (:obj:`BcDataExitH`): The BC.
            bc_jinja (:obj:`dict`): Dict getting filled with data used with jinja.
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            report_dir (:obj:`str`): Path to directory where report files are saved.
            logger: The logger.
        """
        # Get the inputs used in the Manning's N calculator to generate Exit-H WSE.
        input_dict = json.loads(bc.manning_n_calculator_inputs)
        xms_getter = XmsGetter(query, coverage.uuid, [arc_id])
        xms_getter.pe_tree = query.copy_project_tree()
        xms_getter.dataset_uuid = input_dict['elevation_dataset']
        geom_uuid = input_dict.get('geom_uuid', '')
        elevation_source = ''
        arc_data = {}
        if geom_uuid:
            arc_data = xms_getter.retrieve_xms_data(geom_uuid, True, False)
        if arc_id in arc_data:
            # Dataset used to generate WSE curve still exists.
            # Reconstruct the Exit-H cross section curve from the Manning's N calculator.
            elevs, stations, elevation_source = arc_data[arc_id]
            data_list, units = get_manning_n_calc_data(bc)
            data = ManningNData(bc_jinja[bc.param.water_surface_elevation_option.label], units, data_list, input_dict)
            data.set_geometry(elevs, stations)
            data.set_calc_data(
                input_dict['depth_type'], input_dict['manning_n'], input_dict['slope'], input_dict['flow'],
                input_dict['freeboard']
            )
            x_column, y_column = data.get_cross_section_curve_values()
            if logger:
                logger.info('Creating plot of Exit-H cross section.')
            bc_jinja['exit_h_xsect_plot'] = plots.plot_exit_h_cross_section(
                x_column, y_column, coverage.name, arc_id, report_dir
            )
        bc_jinja['channel_calculator'] = input_dict
        bc_jinja['channel_calculator']['elevation_source'] = elevation_source
