"""MaterialCoverageLooper class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.srh.components.material_component import MaterialComponent
from xms.srh.file_io.report import plots, report_util
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase


class MaterialCoverageLooper(CoverageLooperBase):
    """Adds material coverages to the report."""
    def __init__(self, notes_db, report_dir, query, logger):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            logger(:obj:`logger`): The logger.
        """
        super().__init__(notes_db, report_dir, query, 'Materials', 'Material_Component')
        self._logger = logger

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return MaterialComponent(main_file)

    def _load_component_ids(self, component):  # pragma: no cover
        """Loads the feature arc/point IDs into the component.

        Args:
            component (:obj:`MaterialComponent`): The material component to load the IDs into.
        """
        self._query.load_component_ids(component, polygons=True)

    def _store_coverage_data(self, coverage, material_component):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            material_component (:obj:`MaterialComponent`): The material component.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(coverage, material_component, self._notes_db, self._report_dir, self._logger)

    @staticmethod
    def get_jinja(coverage, material_component, notes_db, report_dir, logger):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            material_component (:obj:`MaterialComponent`): The material component.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are saved.
            logger: The logger.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        # Store all jinja stuff in a dict
        coverage_jinja = {'name': coverage.name}
        report_util.add_object_notes(notes_db, coverage.uuid, coverage_jinja)

        # Build materials table
        df = material_component.data.materials.to_dataframe()
        names = df['Name'].tolist()
        mannings = df["Manning's N"].tolist()
        colors = df['Color and Texture'].tolist()
        depth_varied_curve = df["Depth Varied Curve"].tolist()
        materials_jinja = []
        for i in range(len(names)):
            material = {'name': names[i]}
            if depth_varied_curve[i]:
                material['mannings_n'] = 'Depth varied curve'
            else:
                material['mannings_n'] = str(mannings[i])
            material['color'] = ', '.join(colors[i].split()[:3])
            materials_jinja.append(material)

        coverage_jinja['material_count'] = str(len(names))
        coverage_jinja['materials'] = materials_jinja

        # Go through the polygons storing data for the plot
        plot_polygon_atts = {}  # Also store the data by arc id for convenience in creating the plot
        df = material_component.data.materials.to_dataframe()
        polygons = coverage.polygons
        for polygon in polygons:
            comp_id = material_component.get_comp_id(TargetType.polygon, polygon.id)
            if comp_id and comp_id >= 0:
                row = df.loc[df['id'] == comp_id]
                if not row.empty:
                    plot_polygon_atts[polygon.id] = {
                        'name': row.iloc[0]['Name'],
                        'color': row.iloc[0]['Color and Texture'].split()
                    }

        if logger:
            logger.info(f'Creating plot for material coverage "{coverage.name}"')
        coverage_jinja['plot'] = plots.plot_material_coverage(coverage, plot_polygon_atts, report_dir)
        return coverage_jinja
