"""Monitor coverage looper class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.srh.components.monitor_component import MonitorComponent
from xms.srh.file_io.report import plots, report_util
from xms.srh.file_io.report.coverage_looper_base import CoverageLooperBase


class MonitorCoverageLooper(CoverageLooperBase):
    """Adds monitor coverages to the report."""
    def __init__(self, notes_db, report_dir, query, mesh_boundaries, coverage_meshes, logger):  # pragma: no cover
        """Initializes the class.

        Args:
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Path to directory where report files are created.
            query (:obj:`xms.api.dmi.Query`): Object for communicating with SMS
            mesh_boundaries (:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.
            coverage_meshes (:obj:`dict{coverage UUID: mesh UUID}`): Monitor coverages and their meshes.
            logger(:obj:`logger`): The logger
        """
        super().__init__(notes_db, report_dir, query, 'Monitor', 'Monitor_Component')
        self._logger = logger

    def _create_component(self, main_file):  # pragma: no cover
        """Constructs and returns the SRH component given its main file.

        Args:
            main_file: The main file associated with this component.

        Returns:
            See description.
        """
        return MonitorComponent(main_file)

    def _load_component_ids(self, monitor_component):  # pragma: no cover
        """Loads the feature arc/point IDs into the component.

        Args:
            monitor_component (:obj:`MonitorComponent`): The monitor component to load the IDs into.
        """
        self._query.load_component_ids(monitor_component, points=True, arcs=True)

    def _store_coverage_data(self, coverage, monitor_component):  # pragma: no cover
        """Main method to extract data from the coverage and store it in a dict for use with jinja.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            monitor_component (:obj:`MonitorComponent`): The monitors component.

        Returns:
            A dict of the coverage data for use with jinja.
        """
        return self.get_jinja(
            coverage, monitor_component, self._notes_db, self._report_dir, self._logger, self._coverage_meshes,
            self._mesh_boundaries
        )

    @staticmethod
    def get_jinja(
        coverage, monitor_component, notes_db, report_dir, logger, coverage_meshes, mesh_boundaries
    ):  # pragma: no cover
        """Return the jinja for this coverage.

        Args:
            coverage (:obj:`xms.data_objects.parameters.Coverage`): The coverage.
            monitor_component (:obj:`MonitorComponent`): The monitor component.
            notes_db (:obj:`Notes`): Notes object.
            report_dir (:obj:`str`): Filepath to directory where report files will be saved.
            logger(:obj:`logger`): The logger, or None.
            coverage_meshes (:obj:`dict{coverage UUID: mesh UUID}`): Coverages and their meshes.
            mesh_boundaries (:obj:`dict{mesh UUID: MeshBoundaryTuple}`): The mesh boundaries.

        Returns:
            (:obj:`dict`): A dict of the coverage data for use with jinja.
        """
        if logger:
            logger.info(f'Getting data for monitor coverage "{coverage.name}"')
        # Store all jinja stuff in a dict
        coverage_jinja = {'name': coverage.name}
        report_util.add_object_notes(notes_db, coverage.uuid, coverage_jinja)
        # Add arc monitors
        coverage_jinja['arc_count'] = str(len(coverage.arcs))
        arc_ids = [arc.id for arc in coverage.arcs]
        arc_labels = _arc_labels(arc_ids, monitor_component)
        # Add point monitors
        points = coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        pt_ids = [pt.id for pt in points]
        pt_labels = _point_labels(pt_ids, monitor_component)
        coverage_jinja['point_count'] = str(len(points))
        # Draw the plot, including the mesh boundary
        mesh_boundary = CoverageLooperBase.mesh_boundary_from_coverage(coverage.uuid, coverage_meshes, mesh_boundaries)
        if logger:
            logger.info('Creating plot of monitor coverage geometry.')
        coverage_jinja['plot'] = plots.plot_monitor_coverage(coverage, mesh_boundary, report_dir, arc_labels, pt_labels)
        return coverage_jinja


def _arc_labels(arc_ids, monitor_component):
    """Returns a list of the arc labels.

    Args:
        arc_ids (:obj:`list`): The arc IDs.
        monitor_component (:obj:`MonitorComponent`): The monitor component.

    Returns:
        (:obj:`list`): The arc labels.
    """
    fids, flabels = monitor_component.data.get_arc_comp_ids_labels()
    comp_id_to_label = dict(zip(fids, flabels))
    arc_comp_ids = [monitor_component.get_comp_id(TargetType.arc, arc_id) for arc_id in arc_ids]
    arc_comp_ids = [-1 if comp_id is None else comp_id for comp_id in arc_comp_ids]
    arc_labels = []
    for idx, arc_id in enumerate(arc_ids):
        comp_id = arc_comp_ids[idx]
        label = comp_id_to_label.get(comp_id, '')
        if not label:
            label = f'LN{arc_id}'
        arc_labels.append(label)
    return arc_labels


def _point_labels(pt_ids, monitor_component):
    """Returns a list of the arc labels.

    Args:
        pt_ids (:obj:`list`): The arc IDs.
        monitor_component (:obj:`MonitorComponent`): The monitor component.

    Returns:
        (:obj:`list`): The arc labels.
    """
    fids, flabels = monitor_component.data.get_point_comp_ids_labels(False)
    comp_id_to_label = dict(zip(fids, flabels))
    fids, flabels = monitor_component.data.get_point_comp_ids_labels(True)
    comp_id_to_label.update(dict(zip(fids, flabels)))
    pt_comp_ids = [monitor_component.get_comp_id(TargetType.point, pt_id) for pt_id in pt_ids]
    pt_comp_ids = [-1 if comp_id is None else comp_id for comp_id in pt_comp_ids]
    pt_labels = []
    for idx, pt_id in enumerate(pt_ids):
        comp_id = pt_comp_ids[idx]
        label = comp_id_to_label.get(comp_id, '')
        if not label:
            label = f'PT{pt_id}'
        pt_labels.append(label)
    return pt_labels
