"""The Generate Floodway Evaluation Lines dialog."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import math
import uuid

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.api.tree import tree_util
import xms.constraint as xc
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
import xms.data_objects.parameters as dp

# 4. Local modules
from xms.srh.components.sim_query_helper import SimQueryHelper


class GenerateFloodwayEvaluationLines:
    """The Generate Floodway Evaluation Lines Class that will hold data and generate the floodway lines."""
    def __init__(self, query):
        """Initializes the dialog.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
        """
        self.wse_reader = None
        self.interval = 1.0
        self.threshold = 100.0
        self.output_name = ''
        self.intersect_contours = False
        self.contour_values = None
        self.new_xs_cov = None

        self.co_grid = None
        self.pe_tree = None
        self.grid_tree = None
        if query:
            helper = SimQueryHelper(query)
            helper.get_geometry_data()
            self.co_grid = helper.co_grid
            self.pe_tree = query.copy_project_tree()
            self.grid_tree = tree_util.trim_project_explorer(query.copy_project_tree(), helper.grid_uuid)
        self.query = query
        self.cl_uuid = ''
        self.txt_wse_selected = ''
        self.txt_centerline_selected = ''
        self.wse_uuid = None

        # Dataset Info
        self.txt_wse_selected = "(none selected)"
        self.selected_dataset = False
        self.wse_reader = None
        self.wse_ts = None
        self.min_val = None
        self.max_val = None

        self.selected_centerline_cov = False
        self.selected_dataset = False
        self.max_num_contours = 0
        self._logger = logging.getLogger('xms.srh')

    def select_cl_coverage(self, cl_uuid):
        """Launches the select dataset dialog and sets the cross section values.

        Args:
            cl_uuid (:obj:`UUID`): The UUID of the centerline coverage
        """
        self.cl_uuid = cl_uuid
        if self.cl_uuid:
            cover_node = tree_util.find_tree_node_by_uuid(self.pe_tree, self.cl_uuid)
            cl_name = cover_node.name
            self.txt_centerline_selected = cl_name
            self.selected_centerline_cov = True
        else:
            self.txt_centerline_selected = "(none selected)"
            self.selected_centerline_cov = False

    def select_dataset(self, wse_uuid):
        """Launches the select dataset dialog and sets the cross section values.

        Args:
            wse_uuid (:obj:`UUID`): The UUID of the wse dataset
        """
        if wse_uuid:
            # self.wse_ts = dialog.get_selected_time_step_index()
            self.wse_reader = self.query.item_with_uuid(wse_uuid)
            wse_name = tree_util.build_tree_path(self.pe_tree, wse_uuid)
            self.selected_dataset = True
            self.txt_wse_selected = f'{wse_name}:{self.wse_ts}'

            # Get Min and Max values from the dataset
            self.min_val = math.ceil(self.wse_reader.mins[self.wse_ts].item())
            self.max_val = math.floor(self.wse_reader.maxs[self.wse_ts].item())
        else:
            self.txt_wse_selected = "(none selected)"
            self.selected_dataset = False
            self.wse_reader = None

    def generate_floodway_evaluation_lines(self):
        """Generates the floodway elevation lines as a coverage in a member variable."""
        self._logger.info('Creating hydraulic evaluation lines.')
        self._logger.info('Setting up UGrid Contours')
        ugrid = self.co_grid.ugrid
        contours = xc.contour.UGrid2dContour(ugrid)
        self._logger.info('Read scalar datasets to memory.')
        # Get the grid cell/point scalars from self.wse_reader
        if self.wse_reader:
            wse_ts_data = self.wse_reader.values[self.wse_ts]
            activity = []
            if self.wse_reader.null_value is not None:
                contours.set_no_data_value(self.wse_reader.null_value)
            else:
                if self.wse_reader.num_activity_values is not None and self.wse_reader.num_activity_values != \
                        self.wse_reader.num_values:
                    self.wse_reader.activity_calculator = CellToPointActivityCalculator(ugrid)
                wse_ts_data, ts_activity = self.wse_reader.timestep_with_activity(self.wse_ts, nan_activity=True)
                self.wse_reader.null_value = -999.0
                wse_ts_data[np.isnan(wse_ts_data)] = self.wse_reader.null_value
                contours.set_no_data_value(self.wse_reader.null_value)
            # Determine the location at which the data will be stored.
            if self.wse_reader.location == 'cells' or wse_ts_data.size == ugrid.cell_count:
                contours.set_grid_cell_scalars(wse_ts_data, activity, 1)  # DataLocationEnum::LOC_CELLS == 1
            else:
                contours.set_grid_point_scalars(wse_ts_data, activity, 0)  # DataLocationEnum::LOC_POINTS == 0
        self._logger.info('Extracting contours.')
        contours.set_extract_scalars(self.contour_values)
        contours.set_contour_length_threshold(self.threshold)
        new_xs_arcs = []
        cur_pt_id = 1
        cur_arc_id = 1
        if self.intersect_contours:
            self._logger.info('Intersecting centerline with contours.')
            # Get the polyline geometry from self.cl_uuid
            coverage = self.query.item_with_uuid(self.cl_uuid)
            cl_arcs = coverage.arcs
            polyline = []
            if cl_arcs:
                pts = cl_arcs[0].get_points(dp.FilterLocation.PT_LOC_ALL)
                for pt in pts:
                    polyline.append([pt.x, pt.y, pt.z])
            intersecting_lines = contours.get_intersecting_contour_lines(polyline)
            # Create a coverage from the intersecting_lines
            self._logger.info('Create coverage Data')
            for z_val, contour_lines in intersecting_lines.items():
                for contour in contour_lines:
                    points = []
                    for idx, pt in enumerate(contour):
                        point = dp.Point(pt[0], pt[1], z_val)
                        if (idx == 0) or (idx == len(contour) - 1):
                            point.id = cur_pt_id
                            cur_pt_id += 1
                        points.append(point)
                    if len(points) > 2:
                        arc = dp.Arc(
                            start_node=points[0], end_node=points[-1], vertices=points[1:-1], feature_id=cur_arc_id
                        )
                        cur_arc_id += 1
                        new_xs_arcs.append(arc)
        else:
            all_lines = contours.get_connected_contour_lines()
            # Create a coverage from the contour lines
            self._logger.info('Create coverage Data')
            for z_val, contour in all_lines.items():
                points = []
                for idx, pt in enumerate(contour):
                    point = dp.Point(pt[0], pt[1], z_val)
                    if (idx == 0) or (idx == len(contour) - 1):
                        point.id = cur_pt_id
                        cur_pt_id += 1
                    points.append(point)
                if len(points) > 2:
                    arc = dp.Arc(
                        start_node=points[0], end_node=points[-1], vertices=points[1:-1], feature_id=cur_arc_id
                    )
                    cur_arc_id += 1
                    new_xs_arcs.append(arc)

        self._logger.info('Creating the coverage with hydraulic evaluation lines.')
        self.new_xs_cov = dp.Coverage(name=f'{self.output_name}', uuid=str(uuid.uuid4()))
        self.new_xs_cov.arcs = new_xs_arcs
        self.new_xs_cov.complete()

    def get_contour_values(self):
        """Compute a list of contour values from the dialog input.

        Returns:
            (:obj:`list[float]`): See description
        """
        min_val = self.min_val
        max_val = self.max_val
        if min_val is None or max_val is None:
            return []
        # Get the contour values from the dialog parameters
        contour_values = [self.min_val]
        interval = self.interval
        # Prevent infinite loop with interval of 0.0
        if interval == 0:
            interval = 1.0
        value = min_val + interval
        while value <= max_val:
            contour_values.append(value)
            value += interval
        self.max_num_contours = len(contour_values)
        self.contour_values = contour_values
        return contour_values
