"""Qt delegate for the curve editor button widget."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSize, Qt
from PySide2.QtGui import QBrush, QPixmap
from PySide2.QtWidgets import QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules
from xms.srh.data.material_data import MaterialData


class SrhCurveButtonDelegate(QStyledItemDelegate):
    """Qt delegate for the curve editor button widget."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (:obj:`QObject`): The parent Qt object
        """
        super().__init__(parent)
        self.parent_dlg = parent

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (:obj:`QWidget`): The editor widget.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (:obj:`QPainter`): The painter.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            btn = QPushButton('Edit Curve...')
            height = option.rect.height()
            width = option.rect.width()
            min_dim = min(height, width)
            icon_size = QSize(min_dim, min_dim)
            btn.setIconSize(icon_size)
            # TODO: Get a curve preview image on the button
            # btn.setIcon(DisplayOptionIconFactory.get_icon(self.str_to_poly_opts(
            #                                               index.data(Qt.EditRole)), min_dim))
            btn.setFixedWidth(width)
            btn.setFixedHeight(height)

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Called when the depth varied Manning's N XY series editor button is clicked.

        Args:
            event (:obj:`QEvent`): The editor event that was triggered.
            model (:obj:`QAbstractItemModel`): The data model.
            option (:obj:`QStyleOptionViewItem`): The style options.
            index (:obj:`QModelIndex`): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease and index.row() != MaterialData.UNASSIGNED_MAT:
                parent = self.parent_dlg
                id_index = parent.material_model.index(index.row(), MaterialData.COL_ID)
                mat_id = int(float(parent.material_model.data(id_index, Qt.DisplayRole)))
                df = parent.material_data.depth_curve_from_mat_id(mat_id).to_dataframe()
                if not df.index.empty and df.index[0] == 0:
                    df.index += 1  # Start index at 1, not 0

                col_str = 'Depth (m)'
                if 'FEET' in self.parent_dlg.horiz_units:
                    col_str = 'Depth (ft)'
                df.rename(columns={'Depth': col_str}, inplace=True)

                # xy_series dialog
                dlg = XySeriesEditor(
                    data_frame=df,
                    series_name='Depth vs Roughness',
                    dialog_title="Depth Varied Manning's Roughness",
                    parent=self.parent_dlg
                )
                if dlg.exec():
                    df = dlg.model.data_frame
                    df.rename(columns={col_str: 'Depth'}, inplace=True)
                    parent.material_data.set_depth_curve(mat_id, df.to_xarray())
                return True

        return super().editorEvent(event, model, option, index)
