"""A simple Qt dialog for displaying a floating point edit field with an optional units combobox."""

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.srh.gui.double_message_box_ui import Ui_DoubleMessageBox


class DoubleMessageBox(QDialog):
    """Displays a dialog with a floating point edit field and an optional units combobox."""
    def __init__(
        self,
        prompt_string,
        initial_value,
        minimum_value,
        maximum_value,
        precision=2,
        units=None,
        help_url=None,
        parent=None
    ):
        """Initializes the dialog.

        Args:
            prompt_string (:obj:`str`): Prompt string for name of variable.
            initial_value (:obj:`float`): Initial float value.
            minimum_value (:obj:`float`): Minimum float value allowed.
            maximum_value (:obj:`float`): Maximum float value allowed.
            precision (:obj:`int`): The number of digits after decimal to display
            units (:obj:`list`): List of option strings for the unit combobox. No
                combobox shown if not provided.
            help_url (:obj:`str`): URL for help page. Help button hidden if not
                provided
            parent (:obj:`QObject`): The Qt parent.
        """
        super().__init__(parent)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.ui = Ui_DoubleMessageBox()
        self.ui.setupUi(self)
        self.setMinimumSize(self.size())
        if parent:
            self.setWindowIcon(parent.windowIcon())

        if units:
            self.ui.cbx_units.addItems(units)
            self.units = units
        else:
            self.ui.cbx_units.hide()

        # example of help URL
        # help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"
        if help_url:
            self.help_url = help_url
        else:
            self.ui.btn_help.hide()

        dbl_valid = QxDoubleValidator(bottom=minimum_value, top=maximum_value, decimals=precision, parent=self)
        self.ui.double_value.setValidator(dbl_valid)
        initial_string = str(initial_value)
        self.ui.double_value.setText(initial_string)
        self.ui.txt_double.setText(prompt_string)

    def get_double(self):
        """Get the floating point value entered in the dialog.

        Returns:
            (:obj:`float`): See description
        """
        double_text = self.ui.double_value.text()
        return float(double_text)

    def get_units(self):
        """Get the units option selected in the dialog.

        Returns:
            (:obj:`str`): See description
        """
        units = self.ui.cbx_units.currentText()
        return units

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)
