"""Dialog for assigning Material and Sediment Material coverage properties."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import tempfile
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel, Qt
from PySide2.QtWidgets import (QDialogButtonBox, QHBoxLayout, QHeaderView, QLabel, QStyle, QToolBar, QVBoxLayout)

# 3. Aquaveo modules
from xms.core.filesystem import filesystem
from xms.guipy.data.polygon_texture import PolygonOptions, PolygonTexture
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.qx_cbx_delegate import QxCbxDelegate
from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory
from xms.guipy.widgets.qx_table_view import QxTableView
from xms.guipy.widgets.widget_builder import setup_toolbar

# 4. Local modules
from xms.srh.data.material_data import DEFAULT_MATERIAL_COLORS, MaterialData
from xms.srh.gui.curve_btn_delegate import SrhCurveButtonDelegate
from xms.srh.gui.material_filter_model import MaterialFilterModel
from xms.srh.gui.sed_curve_btn_delegate import SrhSedCurveButtonDelegate
from xms.srh.gui.sediment_properties_edit_model import SedAttDisableEditModel
from xms.srh.gui.texture_btn_delegate import SrhDisplayOptionDelegate

COL_DEPTH_CURVE_WIDTH = 80
COL_PROP_GRADATION_CURVE_WIDTH = 100


class MaterialDialog(XmsDlg):
    """A dialog to define materials and their properties."""
    def __init__(self, title, win_cont, filename, edit_sediment, horiz_units, selected_material=None):
        """Initializes the material list and properties dialog.

        Args:
            title (:obj:`str`): Title of the dialog
            win_cont (:obj:`QWidget`): Parent Qt dialog
            filename (:obj:`str`): File path of the file containing material data
            edit_sediment (:obj:`bool`): flag to allow editing sediment properties
            horiz_units (:obj:`str`): horizontal units
            selected_material (:obj:`int`): 0-based row index of the material that should be selected.
                If provided, implies the dialog is coming from the Assign Material command.
        """
        super().__init__(win_cont, 'xms.srh.gui.material_dialog')
        self.filename = filename
        self.edit_sediment = edit_sediment
        self.horiz_units = horiz_units
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:SRH-2D_Material_Properties'
        self.selected_material = selected_material
        self.widgets = {}
        self.material_model = None  # QxPandasTableModel for materials spreadsheet
        self.actions = None  # Disable delete of unassigned material

        self.material_data = None
        self.mat_att_material_id = -1  # init to -1 so it will be set to UNASSIGNED when the dialog starts
        self.mat_att_model = None  # QxPandasTableModel for materials sediment properties spreadsheet
        self.mat_att_models = dict()  # dict of QxPandasTableModel by material id for the sediment properties
        self.mat_att_filter_model = SedAttDisableEditModel()

        self.load_material_data()

        self.deleted_material_ids = set()

        self.change_all_textures = PolygonOptions()

        self.delegates = dict()
        self.delegates['check_delegate'] = CheckBoxNoTextDelegate(self)
        self.delegates['check_delegate'].state_changed.connect(self.depth_varied_state_changed)
        self.delegates['material_display'] = SrhDisplayOptionDelegate(self)
        self.delegates['material_depth_curve_delegate'] = SrhCurveButtonDelegate(self)
        self.delegates['sed_curve_delegate'] = SrhSedCurveButtonDelegate(self)
        self.delegates['sed_cbx'] = QxCbxDelegate(self)
        self.delegates['sed_cbx'].set_strings(['SI', 'English'])
        self.add_icon = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'add.svg')
        self.delete_icon = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'delete.svg')
        self.change_all_icon = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), 'resources', 'icons', 'change-all-mat.svg'
        )

        # Setup the dialog
        self.setWindowTitle(title)
        self.setup_ui()
        self.adjustSize()

        # Calculate a reasonable width for the dialog.
        self.dlg_width = self.widgets['table_view'].horizontalHeader().length()
        self.dlg_width += self.widgets['table_view'].style().pixelMetric(QStyle.PM_ScrollBarExtent)
        self.dlg_width += self.widgets['table_view'].frameWidth() * 2
        if self.edit_sediment:
            self.dlg_width += self.widgets['att_table_view'].horizontalHeader().length()
            self.dlg_width += self.widgets['att_table_view'].style().pixelMetric(QStyle.PM_ScrollBarExtent)
            self.dlg_width += self.widgets['att_table_view'].frameWidth() * 2
        self.dlg_width += 20
        self.resize(self.dlg_width, self.size().height())

    def load_material_data(self):
        """Reads all the material properties from the file."""
        self.material_data = MaterialData(self.filename)
        df = self.material_data.materials.to_dataframe()
        # load all depth curves for materials
        mat_ids = df['id'].tolist()
        for mat_id in mat_ids:
            _ = self.material_data.depth_curve_from_mat_id(mat_id)
        # load all sediment properties
        if self.edit_sediment:
            for mat_id in mat_ids:
                _ = self.material_data.sediment_properties_from_mat_id(mat_id)

    def add_multi_polygon_select_warning(self):
        """Add a label if coming from the Assign Material command with multiple polygons selected."""
        self.widgets['lbl_multi_select'] = QLabel('Material assignment will be applied to all selected polygons.')
        self.widgets['lbl_multi_select'].setStyleSheet('font-weight: bold; color: red')
        self.widgets['main_vert_layout'].insertWidget(0, self.widgets['lbl_multi_select'])

    def setup_ui(self):
        """Setup dialog widgets."""
        # Add a main vertical layout.
        self.widgets['main_vert_layout'] = QVBoxLayout()
        self.setLayout(self.widgets['main_vert_layout'])

        # set up the materials table view
        self._setup_ui_materials_view()

        # add buttons to add, delete, change all textures for materials
        self._setup_ui_material_buttons()

        # Connect a slots to the material list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        selection_model.selectionChanged.connect(self.on_material_changed)
        if self.selected_material is not None:  # Assign Material command
            current_row = self.selected_material
        else:  # Material List and Properties command
            current_row = MaterialData.UNASSIGNED_MAT
        current_index = self.widgets['table_view'].model().index(current_row, MaterialData.COL_NAME)
        selection_model.setCurrentIndex(current_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)

        # add the ok, cancel, help... buttons at the bottom of the dialog
        self._setup_ui_bottom_button_box()

    def _setup_ui_materials_view(self):
        """Sets up the table veiw for the materials."""
        self.widgets['table_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['table_horiz_layout'])

        # Add the material list table view
        self.widgets['table_view'] = QxTableView()
        df = self.material_data.materials.to_dataframe()
        df.index = list(range(1, len(df['id']) + 1))
        self.material_model = QxPandasTableModel(df)
        self.widgets['table_horiz_layout'].addWidget(self.widgets['table_view'])

        # Override filter model to enable editing of description.
        self.widgets['table_view'].filter_model = MaterialFilterModel(self)
        self.widgets['table_view'].filter_model.setSourceModel(self.material_model)
        self.widgets['table_view'].setModel(self.widgets['table_view'].filter_model)

        # Set up the depth varied curve column to be a checkbox
        self.material_model.set_checkbox_columns([MaterialData.COL_DEPTH_TOG])
        self.widgets['table_view'].setItemDelegateForColumn(
            MaterialData.COL_DEPTH_TOG, self.delegates['check_delegate']
        )

        # Set up the color and texture column to be a button
        self.widgets['table_view'].setItemDelegateForColumn(MaterialData.COL_COLOR, self.delegates['material_display'])

        # Set up the depth varied Manning's N column to be an XY series editor button
        self.widgets['table_view'].setItemDelegateForColumn(
            MaterialData.COL_DEPTH_CURVE, self.delegates['material_depth_curve_delegate']
        )

        # Hide the label options and checkbox.
        self.widgets['table_view'].setColumnHidden(MaterialData.COL_ID, True)
        if self.edit_sediment:
            self.widgets['table_view'].setColumnHidden(MaterialData.COL_N, True)
            self.widgets['table_view'].setColumnHidden(MaterialData.COL_DEPTH_TOG, True)
            self.widgets['table_view'].setColumnHidden(MaterialData.COL_DEPTH_CURVE, True)

        self.widgets['table_view'].horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
        self.widgets['table_view'].horizontalHeader().setSectionResizeMode(MaterialData.COL_NAME, QHeaderView.Stretch)
        self.widgets['table_view'].horizontalHeader().setSectionResizeMode(
            MaterialData.COL_DEPTH_CURVE, QHeaderView.Fixed
        )
        self.widgets['table_view'].resizeColumnsToContents()
        self.widgets['table_view'].horizontalHeader().resizeSection(MaterialData.COL_DEPTH_CURVE, COL_DEPTH_CURVE_WIDTH)

        # set the name for the unassigned material to be read only
        self.material_model.read_only_cells.add((0, MaterialData.COL_NAME))

        if self.edit_sediment:
            self._setup_ui_sediment_materials_view()

    def _setup_ui_sediment_materials_view(self):
        """Sets up the table view for the sediment properties."""
        # Add a table view for the material attributes.
        self.widgets['att_table_vert_layout'] = QVBoxLayout()
        self.widgets['table_horiz_layout'].addLayout(self.widgets['att_table_vert_layout'])

        self.widgets['att_table_multi_select_text'] = QLabel('Select one material to edit sediment properties')
        self.widgets['att_table_multi_select_text'].setStyleSheet('font-weight: bold; color: red')
        self.widgets['att_table_vert_layout'].addWidget(self.widgets['att_table_multi_select_text'])

        self.widgets['att_table_multi_select_text'].hide()
        self.widgets['att_table_view'] = QxTableView()
        self._set_attribute_model(MaterialData.UNASSIGNED_MAT)
        self.widgets['att_table_view'].setModel(self.mat_att_model)
        self.widgets['att_table_vert_layout'].addWidget(self.widgets['att_table_view'])

        self.widgets['att_table_view'].setColumnHidden(MaterialData.COL_PROP_ID, True)
        self.widgets['att_table_view'].horizontalHeader().setSectionResizeMode(
            MaterialData.COL_PROP_THICKNESS, QHeaderView.Stretch
        )
        self.widgets['att_table_view'].horizontalHeader().setSectionResizeMode(
            MaterialData.COL_PROP_DENSITY, QHeaderView.Stretch
        )
        self.widgets['att_table_view'].horizontalHeader().setSectionResizeMode(
            MaterialData.COL_PROP_GRADATION_CURVE, QHeaderView.Fixed
        )
        self.widgets['att_table_view'].resizeColumnsToContents()
        self.widgets['att_table_view'].horizontalHeader().resizeSection(
            MaterialData.COL_PROP_GRADATION_CURVE, COL_PROP_GRADATION_CURVE_WIDTH
        )
        # Setup delegate for curve button column.
        self.widgets['att_table_view'].setItemDelegateForColumn(
            MaterialData.COL_PROP_GRADATION_CURVE, self.delegates['sed_curve_delegate']
        )
        # Setup delegate for combobox column.
        self.widgets['att_table_view'].setItemDelegateForColumn(MaterialData.COL_PROP_UNITS, self.delegates['sed_cbx'])

    def _set_attribute_model(self, current_row):
        """Called to update the model linked to the material attribute table view.

        Args:
            current_row (:obj:`int`): The currently selected row.
        """
        if current_row < MaterialData.UNASSIGNED_MAT:  # No material selected
            current_row = MaterialData.UNASSIGNED_MAT

        new_id_index = self.material_model.index(current_row, MaterialData.COL_ID)
        mat_id = int(self.material_model.data(new_id_index, Qt.DisplayRole))
        if mat_id != self.mat_att_material_id:
            if mat_id not in self.mat_att_models:  # Model not created yet
                df = self.material_data.sediment_properties_from_mat_id(mat_id).to_dataframe()
                self._fix_df_column_names_for_gui(df)
                self.mat_att_models[mat_id] = QxPandasTableModel(df)
                self.mat_att_models[mat_id].set_combobox_column(
                    MaterialData.COL_PROP_UNITS, self.delegates['sed_cbx'].get_strings()
                )
            # Disable attribute table if unassigned material selected.
            self.mat_att_model = self.mat_att_models[mat_id]
            enable_add = current_row != MaterialData.UNASSIGNED_MAT
            enable_delete = enable_add and not self.mat_att_models[mat_id].data_frame.empty
            self.widgets['att_table_view'].filter_model = self.mat_att_filter_model
            self.widgets['att_table_view'].filter_model.setSourceModel(self.mat_att_model)
            self.widgets['att_table_view'].setModel(self.widgets['att_table_view'].filter_model)
            self.widgets['att_table_view'].setEnabled(enable_add)
            self.widgets['att_table_view'].setColumnHidden(MaterialData.COL_PROP_DENSITY, True)
            if 'att_toolbar' in self.widgets:
                tb = self.widgets['att_toolbar']
                tb.widgetForAction(self.att_actions[self.add_icon]).setEnabled(enable_add)
                tb.widgetForAction(self.att_actions[self.delete_icon]).setEnabled(enable_delete)
            self.widgets['att_table_view'].selectionModel().selectionChanged.connect(self.on_material_att_changed)
            self.mat_att_material_id = mat_id

    def _setup_ui_material_buttons(self):
        """Adds toolbar below the materials table view to add, delete, change all textures."""
        self.widgets['tool_horiz_layout'] = QHBoxLayout()
        self.widgets['main_vert_layout'].addLayout(self.widgets['tool_horiz_layout'])
        # Add the "+", "-", and change all buttons.
        self.widgets['toolbar'] = QToolBar()
        button_list = [
            [self.add_icon, 'Add Material', self.on_btn_add_material],
            [self.delete_icon, 'Delete Material', self.on_btn_delete_material],
            [self.change_all_icon, 'Change all material textures', self.on_btn_change_all]
        ]
        self.actions = setup_toolbar(self.widgets['toolbar'], button_list)
        change_all_btn = self.widgets['toolbar'].widgetForAction(self.actions[self.change_all_icon])
        texture_icon = DisplayOptionIconFactory.get_icon(self.change_all_textures, change_all_btn.size().width())
        change_all_btn.setIcon(texture_icon)
        self.widgets['tool_horiz_layout'].addWidget(self.widgets['toolbar'])

        if self.edit_sediment:
            # Add the "+", "-" buttons for the material att list.
            self.widgets['att_toolbar'] = QToolBar()
            att_button_list = [
                [self.add_icon, 'Add Material Attribute Row', self.on_btn_add_material_att],
                [self.delete_icon, 'Delete Material Attribute Row', self.on_btn_delete_material_att]
            ]
            self.att_actions = setup_toolbar(self.widgets['att_toolbar'], att_button_list)
            self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.add_icon]).setEnabled(False)
            self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.delete_icon]).setEnabled(False)
            self.widgets['tool_horiz_layout'].addWidget(self.widgets['att_toolbar'])

    def _setup_ui_bottom_button_box(self):
        """Add buttons to the bottom of the dialog."""
        # Add Import and Export buttons
        self.widgets['btn_horiz_layout'] = QHBoxLayout()
        self.widgets['btn_box'] = QDialogButtonBox()
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(self.reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)
        self.widgets['btn_horiz_layout'].addWidget(self.widgets['btn_box'])

        self.widgets['main_vert_layout'].addLayout(self.widgets['btn_horiz_layout'])

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def depth_varied_state_changed(self, index):
        """Called when the depth varied Manning's N checkbox is clicked.

        Args:
            index (:obj:`QModelIndex`): Model index of the clicked item
        """
        # Disable/enable the constant Manning's N column and the depth varied column.
        self.widgets['table_view'].update(index.model().index(index.row(), MaterialData.COL_N))
        self.widgets['table_view'].update(index.model().index(index.row(), MaterialData.COL_DEPTH_CURVE))

    def on_btn_change_all(self):
        """Called when the change all material texture/style button is clicked."""
        dlg = PolygonDisplayOptionsDialog(self.change_all_textures, self)
        dlg.ui.lab_line_color.setVisible(False)
        dlg.ui.color_btn.setVisible(False)
        if dlg and dlg.exec():
            self.change_all_textures = dlg.get_options()
            new_opts = self.delegates['material_display'].poly_opts_to_str(self.change_all_textures)
            texture = new_opts.split()[-1]
            for row in range(self.material_model.rowCount()):
                data = self.material_model.data(self.material_model.index(row, MaterialData.COL_COLOR), Qt.EditRole)
                lst = data.split()
                data = f'{lst[0]} {lst[1]} {lst[2]} {texture}'
                self.material_model.setData(self.material_model.index(row, MaterialData.COL_COLOR), data, Qt.EditRole)

    def on_btn_add_material(self):
        """Called when the add material button is clicked."""
        mat_id = max(self.material_model.data_frame['id']) + 1
        description = self._get_new_material_name()
        row_idx = self.material_model.rowCount()
        mat_color = DEFAULT_MATERIAL_COLORS[row_idx % len(DEFAULT_MATERIAL_COLORS)]
        self.material_model.set_default_values(
            {
                'id': mat_id,
                'Color and Texture': f'{mat_color[0]} {mat_color[1]} {mat_color[2]} {int(PolygonTexture.null_pattern)}',
                'Name': description,
                "Manning's N": 0.02,
                'Depth Varied Curve': 0,
                'Curve': '',
            }
        )
        row_idx = self.material_model.rowCount()
        self.material_model.insertRows(row_idx, 1)

        # Connect a slots to the material list view for when the selection changes.
        selection_model = self.widgets['table_view'].selectionModel()
        new_mat_index = self.widgets['table_view'].model().index(row_idx, MaterialData.COL_NAME)
        selection_model.setCurrentIndex(new_mat_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear)

    def _get_new_material_name(self):
        """Get a unique name for a new material.

        Returns:
            (:obj:`str`): See description
        """
        mat_name = 'new material'
        unique = False
        i = 1
        while not unique:
            unique = True
            for row in self.material_model.data_frame['Name']:
                if str(row) == mat_name:
                    unique = False
                    mat_name = f'new material ({i})'
                    i += 1
                    break
        return mat_name

    def on_btn_delete_material(self):
        """Called when the delete material button is clicked."""
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        if len(indices) <= 0:
            return
        rows = dict()
        for index in indices:
            if index.row() != MaterialData.UNASSIGNED_MAT:
                rows[index.row()] = index

        for row in sorted(rows.keys(), reverse=True):
            index = rows[row]
            # Keep track of deleted material ids so we can reassign polygons in XMS.
            delete_id_index = self.material_model.index(index.row(), MaterialData.COL_ID)
            mat_id = int(float(self.material_model.data(delete_id_index, Qt.DisplayRole)))
            self.deleted_material_ids.add(mat_id)
            self.material_model.removeRows(index.row(), 1)

        # select the row above the first currently selected row
        row = sorted(rows.keys())[0] - 1
        select_index = self.widgets['table_view'].model().index(row, MaterialData.COL_NAME)
        self.widgets['table_view'].selectionModel().setCurrentIndex(
            select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
        )

    def on_material_changed(self):
        """Called when the material row selection changes."""
        # Disable the delete button if the unassigned material currently selected.
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        enable_delete = True
        # if the 'unassigned' material is select then the user can not delete
        for index in indices:
            if index.row() == MaterialData.UNASSIGNED_MAT:
                enable_delete = False
        self.widgets['toolbar'].widgetForAction(self.actions[self.delete_icon]).setEnabled(enable_delete)

        if self.edit_sediment:
            enable_add = False
            enable_delete = False
            if len(indices) > 1 or len(indices) <= 0:
                self.widgets['att_table_multi_select_text'].show()
                self.widgets['att_table_view'].setEnabled(False)
            else:
                if indices[0].row() != MaterialData.UNASSIGNED_MAT:
                    enable_add = True
                    enable_delete = True
                self._set_attribute_model(indices[0].row())
                if len(self.mat_att_model.data_frame.index) > 0:
                    att_index = self.mat_att_model.index(0, MaterialData.COL_PROP_THICKNESS)
                    self.widgets['att_table_view'].selectionModel().setCurrentIndex(
                        att_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
                    )
                else:
                    enable_delete = False
                self.widgets['att_table_multi_select_text'].hide()
                self.widgets['att_table_view'].setEnabled(True)
            self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.add_icon]).setEnabled(enable_add)
            self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.delete_icon]).setEnabled(enable_delete)

    def on_material_att_changed(self):  # pragma: no cover
        """Implement to handle selection changes in the material attribute table."""
        pass

    def on_btn_add_material_att(self):
        """Called when the add material attribute row button is clicked."""
        # only 1 material should be selected
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        if not len(indices) > 0 or not indices[-1].isValid():
            return
        # get next material att id
        att_id = 1
        if len(self.mat_att_model.data_frame.index) > 0:
            att_id = max(self.mat_att_model.data_frame['att_id']) + 1
        cols = self.mat_att_model.data_frame.columns
        vals = [att_id, 0.0, 1, 0.0, '']
        self.mat_att_model.set_default_values(dict(zip(cols, vals)))
        row_idx = self.mat_att_model.rowCount()
        self.mat_att_model.insertRows(row_idx, 1)
        new_index = self.mat_att_model.index(row_idx, MaterialData.COL_PROP_THICKNESS)
        self.widgets['att_table_view'].selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
        )
        self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.delete_icon]).setEnabled(True)

    def on_btn_delete_material_att(self):
        """Called when the delete material attribute row button is clicked."""
        # only 1 material should be selected
        indices = self.widgets['table_view'].selectionModel().selectedIndexes()
        if not len(indices) > 0 or not indices[-1].isValid():
            return

        indices = self.widgets['att_table_view'].selectionModel().selectedIndexes()
        rows = dict()
        for index in indices:
            rows[index.row()] = index

        for row in sorted(rows.keys(), reverse=True):
            index = rows[row]
            self.mat_att_model.removeRows(index.row(), 1)

        if len(self.mat_att_model.data_frame) < 1:
            self.widgets['att_toolbar'].widgetForAction(self.att_actions[self.delete_icon]).setEnabled(False)

        # select the row above the first currently selected row
        row = sorted(rows.keys())[0] - 1
        if row < 0:
            row = 0
        select_index = self.widgets['att_table_view'].model().index(row, MaterialData.COL_PROP_THICKNESS)
        self.widgets['att_table_view'].selectionModel().setCurrentIndex(
            select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear
        )

    def _fill_material_data_from_dialog_data(self, mat_data):
        """Puts the dialog data into a MaterialData class.

        Args:
            mat_data (:obj:`MaterialData`): material data class
        """
        mat_data.set_materials(self.material_model.data_frame.to_xarray())
        mat_ids = self.material_model.data_frame['id'].tolist()
        dlg_data = self.material_data
        for mat_id in mat_ids:
            # copy depth curves
            mat_data.set_depth_curve(mat_id, dlg_data.depth_curve_from_mat_id(mat_id))
            # copy sediment properties
            df = None
            if mat_id in self.mat_att_models.keys():
                df = self.mat_att_models[mat_id].data_frame
                self._fix_df_column_names_for_file_io(df)
            else:
                df = dlg_data.sediment_properties_from_mat_id(mat_id).to_dataframe()
            mat_data.set_sediment_properties(mat_id, df.to_xarray())
            # copy sediment gradation curves
            prop_ids = df['att_id'].tolist()
            for prop_id in prop_ids:
                curve = dlg_data.gradation_curve_from_mat_id_prop_id(mat_id, prop_id)
                mat_data.set_gradation_curve(mat_id, prop_id, curve)

    def _fix_df_column_names_for_gui(self, df):
        """We want multiline headers for some data frames for the gui but we don't want newline characters in the files.

        So this method replaces special characters according to the dict.

        Args:
            df (:obj:`pandas.DataFrame`): the data frame
        """
        df.columns = [
            'att_id', 'Thickness\n(ft or m)', 'Units            ', 'Density\n(lb/ft^3 or kg/m^3)', 'Gradation Curve'
        ]

    def _fix_df_column_names_for_file_io(self, df):
        """We want multiline headers for some data frames for the gui but we don't want newline characters in the files.

        So this method replaces special characters according to the dict.

        Args:
            df (:obj:`pandas.DataFrame`): the data frame
        """
        with tempfile.NamedTemporaryFile(mode='wt') as temp:
            filename = temp.name
        mat = MaterialData(filename)
        sed_df = mat.sediment_properties_from_mat_id(0).to_dataframe()
        df.columns = sed_df.columns
        filesystem.removefile(filename)

    def accept(self):
        """Save material properties and block accepted signal if material names are no longer unique."""
        app_name = os.environ.get('XMS_PYTHON_APP_NAME')
        mat_names = self.material_model.data_frame['Name'].to_list()
        set_names = set()
        for name in mat_names:
            num_names = len(set_names)
            set_names.add(name)
            if num_names == len(set_names):
                msg = f'Material: "{name}" is not a unique name.\n' \
                      f'Material names must be unique. Edit the names to ensure that they are unique.'
                message_box.message_with_ok(parent=self, message=msg, app_name=app_name, win_icon=self.windowIcon())
                return

        # If we came from the Assign Material command, make sure we have a single material row selected.
        if self.selected_material is not None:
            indices = self.widgets['table_view'].selectionModel().selectedIndexes()
            if len(indices) != 1 or not indices[0].isValid():
                msg = 'Please select a single material row to update the assignment of the selected polygon(s).'
                message_box.message_with_ok(
                    parent=self, message=msg, app_name=app_name, icon='Critical', win_icon=self.windowIcon()
                )
                return
            self.selected_material = indices[0].row()

        self._fill_material_data_from_dialog_data(self.material_data)
        self.material_data.commit()
        super().accept()

    def reject(self):
        """Call reject slot on parent if we have one."""
        super().reject()
