"""Module for data used in the Manning's N calculator."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas

# 3. Aquaveo modules

# 4. Local modules
from xms.srh.manning_n.manning_n_calc import ManningNCalc


class ManningNData:
    """Class to store data for the Manning's N calculator."""
    def __init__(self, option=None, units=None, in_values=None, inputs=None):
        """Constructor.

        Args:
            option (:obj:`str`): The BC type (Constant, Time series, or Rating curve)
            units (:obj:`str`): The type of units for units labels (U.S. Customary Units or SI Units (Metric))
            in_values (:obj:`list`): The input values (hours for time series, flows for rating curve)
            inputs (:obj:`dict`): Dict of input options to use
        """
        self.manning_n_calc = ManningNCalc()
        self.depth_type = "Normal depth"
        self.freeboard = 0.0

        self.units = "U.S. Customary Units"
        if units:
            self.units = units
        self.feet_to_meter_conversion = 1.0
        self.cfs_to_cms_conversion = 1.0

        self.hours = []
        self.flows = []
        self.option = option
        self.use_multiple_flows = option != "Constant"
        if in_values and option == "Time series":
            self.hours = in_values
            self.flows = [0.0] * len(self.hours)
        elif in_values and option == "Rating curve":
            self.flows = in_values
            self.hours = [0.0] * len(self.flows)

        # The final result
        self.exit_h_results = None

        # Declare remaining variables
        self.stations = []
        self.elevations = []
        self.num_rows = 0
        self.no_data = -99999.999

        # Set input options if provided
        if inputs:
            self.depth_type = inputs['depth_type']
            self.manning_n_calc.composite_n = inputs['manning_n']
            self.manning_n_calc.slope = inputs['slope']
            self.manning_n_calc.flows = [inputs['flow']]
            self.freeboard = inputs['freeboard']

    def get_cross_section_curve_values(self):
        """Returns a tuple of the cross section curve x and y values."""
        x_column = [station / self.feet_to_meter_conversion for station in self.manning_n_calc.original_stations]
        y_column = [elevation / self.feet_to_meter_conversion for elevation in self.manning_n_calc.original_elevations]
        return x_column, y_column

    def set_geometry(self, elevations, stations):
        """Set cross-section elevations and stations.

        Args:
            elevations (:obj:`list[float]`): Elevations of the cross-section
            stations (:obj:`[float]`): Stations of the cross-section
        """
        self.elevations = elevations
        self.stations = stations

    def add_flows(self, min_flow, max_flow, inc_flow):
        """Populate the flows table.

        Args:
            min_flow (:obj:`float`): Starting flow to add
            max_flow (:obj:`float`): Ending flow to add
            inc_flow (:obj:`float`): Flow size increment to use between the starting and ending flow
        """
        cur_flow = min_flow
        while cur_flow < max_flow:
            self.hours.append(0.0)
            self.flows.append(cur_flow)
            cur_flow += inc_flow
        self.hours.append(0.0)
        self.flows.append(max_flow)

    def add_flow(self):
        """Add an default row to the flows table."""
        self.hours.append(0.0)
        self.flows.append(0.0)

    def remove_flow(self, selected_row):
        """Remove a row from the flows table.

        Args:
            selected_row (:obj:`int`): Index of the row to remove
        """
        if len(self.hours) > 0:
            self.hours.pop(selected_row)
            self.flows.pop(selected_row)

    def set_flow(self, row, col, value):
        """Update either the time or flow rate of a row in the flows table.

        Args:
            row (:obj:`int`): Index of the row to update
            col (:obj:`int`): Index of the column to update
            value (:obj:`float`): The new time or flow rate for the row
        """
        if row < len(self.flows):
            if self.option == 'Time series' and col == 0:
                self.hours[row] = value
            else:
                self.flows[row] = value
        self.compute_data()

    def set_flows(self, flows, hours=None):
        """Set the times and flow rates of the flows table.

        Args:
            flows (:obj:`list`): The flow rates to use
            hours (:obj:`list`): Parallel list of float time values for each flow rate. If not provided,
                parallel list of 0.0 times is used.
        """
        self.flows = flows
        if hours:
            self.hours = hours
        else:
            self.hours = [0.0] * len(self.flows)

    def set_calc_data(self, depth_type, manning_n, slope, flow, freeboard):
        """Set data in the Manning's N calculator from user input.

        Args:
            depth_type (:obj:`str`): 'Normal depth' or 'Critical depth'
            manning_n (:obj:`float`): Manning's N value to be used
            slope (:obj:`float`): Slope value to be used
            flow (:obj:`float`): Flow value to be used
            freeboard (:obj:`float`): Freeboard value to be used
        """
        self.depth_type = depth_type

        if self.units == "SI Units (Metric)":
            self.feet_to_meter_conversion = 3.28084
            self.cfs_to_cms_conversion = 35.31466

        # Set the data in U.S. Units only!!
        self.manning_n_calc.original_stations = []
        self.manning_n_calc.original_elevations = []
        for station in self.stations:
            self.manning_n_calc.original_stations.append(station * self.feet_to_meter_conversion)
        for elevation in self.elevations:
            self.manning_n_calc.original_elevations.append(elevation * self.feet_to_meter_conversion)
        self.manning_n_calc.composite_n = manning_n
        self.manning_n_calc.slope = slope
        self.freeboard = freeboard * self.feet_to_meter_conversion

        if self.use_multiple_flows:
            # handle multiple flows
            self.manning_n_calc.flows = []
            for flow in self.flows:
                self.manning_n_calc.flows.append(flow * self.cfs_to_cms_conversion)
            self.num_rows = len(self.flows)
        else:
            flow = flow * self.cfs_to_cms_conversion
            self.manning_n_calc.flows = [flow]
            self.num_rows += 1

    def compute_data(self):
        """Compute the Manning's N curves.

        Returns:
            (:obj:`tuple(bool,list)`): True if no errors, list of warning messages if any
        """
        self.manning_n_calc.compute_data()
        warnings = self.manning_n_calc.check_warnings()
        success = self.manning_n_calc.get_can_compute()
        success = success or self.depth_type == "Critical depth" and self.manning_n_calc.get_can_compute_critical()
        if success:
            # Print results to table
            if self.depth_type == "Normal depth":
                self._build_exit_h_results(self.manning_n_calc.normal_wses)
            else:
                self._build_exit_h_results(self.manning_n_calc.critical_wses)
        else:
            # Not enough data to compute (or data is bad)
            if not self.use_multiple_flows:
                self.num_rows = 0
            warnings.append("Results will be shown when all the input is entered")
            blank_wse_list = []
            if self.use_multiple_flows:
                blank_wse_list = [self.no_data] * len(self.manning_n_calc.flows)

            self._build_exit_h_results(blank_wse_list)
        return success, warnings

    def _build_exit_h_results(self, wse_list):
        """Build a pandas.DataFrame of WSE values for an Exit-H boundary.

        Args:
            wse_list (:obj:`list`): The water surface elevation values
        """
        index = 0
        # put final answer in form easy to give calling function
        hours = []
        flows = []
        wses = []
        wses_freeboard = []
        for wse in wse_list:
            # Time
            if self.option == 'Time series':
                hour = self.hours[index]
                hours.append(hour)

            # Flow
            converted_flow_value = self.manning_n_calc.flows[index] / self.cfs_to_cms_conversion
            flows.append(converted_flow_value)

            # WSE
            if wse == self.no_data:
                converted_wse_value = self.no_data
            else:
                converted_wse_value = wse / self.feet_to_meter_conversion
            wses.append(converted_wse_value)

            # WSE + Freeboard
            if self.freeboard > 0.0:
                if wse == self.no_data:
                    converted_wse_value = self.no_data
                else:
                    converted_wse_value = (wse + self.freeboard) / self.feet_to_meter_conversion
                wses_freeboard.append(converted_wse_value)
            index += 1

        frame_data = {}
        if len(hours) > 0:
            frame_data['hrs'] = hours
        frame_data['flow'] = flows
        frame_data['wse'] = wses
        if len(wses_freeboard) > 0:
            frame_data['wse_freeboard'] = wses_freeboard
        self.exit_h_results = pandas.DataFrame(frame_data)

    def add_series(self, axes, selected_row, station_header, elevation_header):
        """Adds the XY line series to the plot.

        Args:
            axes (:obj:`AxesSubplot`): The plot axes.
            selected_row (:obj:`int`): Currently selected row of the Manning's N calculator flows table
            station_header (:obj:`str`): The x-axis label
            elevation_header (:obj:`str`): The y-axis label
        """
        if selected_row < 0:
            selected_row = len(self.manning_n_calc.flows) - 1
            if selected_row < 0:
                selected_row = 0
        plot_name = "Cross Section"
        if len(self.manning_n_calc.flows) > selected_row:
            plot_name = "Cross Section with flow of "
            plot_name += str(self.manning_n_calc.flows[selected_row] / self.cfs_to_cms_conversion)
            if self.units == "U.S. Customary Units":
                plot_name += " cfs"
            else:
                plot_name += " cms"

        # If we add plot fill, look up fill_between and convert the stations & elevations of the cross section and
        # normal wse to the same station, cross section elevation, and wse elevation (the same stations list for
        # both elevation lists) to pass to that function.

        self._add_cross_section_series(axes, plot_name, station_header, elevation_header, True)
        self._add_wse_series(axes, selected_row)
        self._add_critical_series(axes, selected_row)

        # Add legend, but check which side will have less data
        if len(self.elevations) > 0:
            legend_location = "upper left"
            if self.elevations[0] > self.elevations[-1]:
                legend_location = "upper right"
            axes.legend(loc=legend_location)

    def _add_cross_section_series(self, axes, name, station_header, elevation_header, show_axis_titles):
        """Adds an XY line series to the plot.

        Args:
            axes (:obj:`AxesSubplot`): The plot axes.
            name (:obj:`str`): The series name.
            station_header (:obj:`str`): The x-axis label
            elevation_header (:obj:`str`): The y-axis label
            show_axis_titles (:obj:`bool`): If True, axis titles displayed.
        """
        axes.clear()
        axes.set_title(name)
        axes.grid(True)
        axes.ticklabel_format(useOffset=False)

        # Add data to plot
        x_column, y_column = self.get_cross_section_curve_values()
        axes.plot(x_column, y_column, 'k', label="Cross section")

        # Axis titles
        if show_axis_titles:
            axes.set_xlabel(station_header)
        if show_axis_titles:
            axes.set_ylabel(elevation_header)

    def _add_wse_series(self, axes, index):
        """Adds an XY line series for water surface elevations to the plot.

        Args:
            axes (:obj:`AxesSubplot`): The plot axes.
            index (:obj:`int`): The index of station and water surface elevations to add.
        """
        if not self.manning_n_calc.get_can_compute():
            return

        # Add data to plot
        selected_flow = "Normal depth"
        stations = self.manning_n_calc.normal_wses_stations[index]
        x_column = [station / self.feet_to_meter_conversion for station in stations]
        y_column = [self.manning_n_calc.normal_wses[index] / self.feet_to_meter_conversion] * len(x_column)
        if len(x_column) > 0:
            axes.plot(x_column, y_column, 'b', label=selected_flow)

    def _add_critical_series(self, axes, index):
        """Adds an XY line series for critical water surface elevations to the plot.

        Args:
            axes (:obj:`AxesSubplot`): The plot axes.
            index (:obj:`int`): The index of station and water surface elevations to add.
        """
        if not self.manning_n_calc.get_can_compute_critical():
            return

        # Add data to plot
        selected_flow = "Critical depth"
        x0 = self.manning_n_calc.stations[0] / self.feet_to_meter_conversion
        x1 = self.manning_n_calc.stations[-1] / self.feet_to_meter_conversion
        x_column = x0, x1
        y_column = [self.manning_n_calc.critical_wses[index] / self.feet_to_meter_conversion] * len(x_column)
        if len(x_column) > 0:
            axes.plot(x_column, y_column, 'r--', label=selected_flow)
