"""Exports SRH simulation."""
# 1. Standard python modules
from datetime import datetime, timedelta
import logging
import math
import os
import sys
import time

# 2. Third party modules
from osgeo import osr
import pandas
from PySide2.QtCore import QThread, Signal

# 3. Aquaveo modules
from xms.api.dmi import Query  # noqa: I202
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.api.tree import tree_util
from xms.data_objects.parameters import FilterLocation
from xms.gdal.rasters import RasterInput
from xms.gdal.utilities import gdal_utils as gu
from xms.grid.geometry import MultiPolyIntersector
from xms.guipy.dialogs.treeitem_selector_datasets import uuid_and_time_step_index_from_string
from xms.snap.snap_polygon import SnapPolygon
from xms.srh.file_io.geom_writer import GeomWriter
from xms.srh.file_io.material_writer import MaterialWriter

# 4. Local modules
from xms.srhw.components.rainfall_component import MET_AIR_PRESSURE_HEADER, MET_LONGWAVE_RADIATION_HEADER, \
    MET_RAINFALL_TOTAL_HEADER, MET_REL_HUMIDITY_HEADER, MET_SOLAR_RADIATION_HEADER, MET_SURFACE_TEMP_HEADER, \
    MET_SURFACE_WIND_SPEED_HEADER, MET_TIME_HEADER, rainfall_defaults, RAINFALL_PT_XY_SERIES
from xms.srhw.components.sim_query_helper import SimQueryHelper
from xms.srhw.file_io.hydro_writer import HydroWriter
from xms.srhw.file_io.shapefile_converter import ShapefileConverter
from xms.srhw.gui.model_control_dialog_w import land_use_defaults, soil_type_defaults
from xms.srhw.mapping.coverage_mapper import CoverageMapper

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"


def _get_closest_pt(center, pts):
    """Gets the closest point to center in the list of pts."""
    min_dist = sys.float_info.max
    min_id = -9999
    for pt in pts:
        dist = math.dist([pt.x, pt.y, pt.z], center)
        if dist < min_dist:
            min_dist = dist
            min_id = pt.id
    return min_id


def _get_cell_polygons(ugrid) -> tuple[list[()], list[list[int]]]:
    """Returns the ugrid's cells as polygons: list of points, list of polygons defined by point indices."""
    cell_points = ugrid.locations
    cell_polys = []
    for cell_idx in range(ugrid.cell_count):
        cell_polys.append(ugrid.get_cell_points(cell_idx))
    return cell_points, cell_polys


class ExportSimulationRunner(QThread):
    """Class for exporting SRH-W."""
    processing_finished = Signal()

    def __init__(self, out_dir):
        """Constructor.

        Args:
            out_dir (:obj:`str`): output directory
        """
        super().__init__()
        self.out_dir = os.path.normpath(out_dir)
        self.query = None
        self.sim_query_helper = None
        self.coverage_mapper = None
        self.project_name = None
        self.sim_component = None
        self._polygon_snapper = None
        self._logger = logging.getLogger('xms.srhw')
        self.files_exported = []
        self.cells_to_land_use = {}
        self.cells_to_soil_type = {}
        self.land_use_data = None
        self.soil_type_data = None
        self._sediment = False
        self._stream_cell_ids = set()
        self._adj_stream_cell_ids = set()

    def run(self):
        """Creates the snap preview of coverages onto the mesh."""
        try:
            self._setup_query()
            self.coverage_mapper.do_map()
            if self.coverage_mapper.bc_coverage is None or not self.coverage_mapper.bc_arc_id_to_grid_ids:
                self._logger.error("The BC Coverage has not been defined correctly.  Please define an exit boundary "
                                   "condition arc at the watershed outlet location in an SRH-2D Boundary Conditions "
                                   "coverage and assign the coverage to your model.")
            self.do_export()
        except Exception as error:
            err_str = str(error)
            if err_str:
                self._logger.exception(f'Error exporting simulation: {err_str}')
                # raise error
        finally:
            self.processing_finished.emit()

    def _setup_query(self):
        self._logger.info('Establishing communication with SMS.')
        time.sleep(0.1)
        self.query = Query()
        self.project_name = os.path.splitext(os.path.basename(self.query.xms_project_path))[0]
        self.sim_query_helper = SimQueryHelper(self.query, at_sim=True)
        self.sim_query_helper.get_sim_data()
        # Get the tree simulation name
        self._simulation_name = self.sim_query_helper.sim_tree_item.name
        # Get the sim component
        self.sim_component = self.sim_query_helper.sim_component
        # Get the coverage mapper
        self.coverage_mapper = CoverageMapper(self.sim_query_helper)
        # Get the cogrid
        if self.sim_query_helper.co_grid is None:
            self._logger.error('No mesh or ugrid in the SRH simulation.')
            raise RuntimeError()

    def do_export(self):
        """Export the simulation."""
        if XmEnv.xms_environ_running_tests() != 'TRUE':  # can't reach during testing
            self._logger.info(f'Writing SRH files to this directory: \n\n{self.out_dir}\n')  # pragma no cover
        self.files_exported = []
        self.export_geom()
        self.read_land_use_table()
        self.read_soil_type_table()
        self.get_land_use()
        self.get_soil_type()
        self.get_streams()
        self.export_materials()
        self.export_spatial_land_soil()
        self.export_land_use_table()
        self.export_soil_type_table()
        self.export_rainfall()
        self.export_rainfall_cell_mapping()
        self.export_erosion()
        self.export_groundwater()
        self.export_sif_file()
        self.export_hydro()

    def export_geom(self):
        """Exports the srhgeom file."""
        self._logger.info('Writing SRH-W geometry file.')
        co_grid = self.coverage_mapper.co_grid
        base_name = f'{self.project_name}.srhgeom'
        self.files_exported.append(f'Grid "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        ugrid = co_grid.ugrid
        grid_name = self.coverage_mapper.grid_name
        grid_units = self.coverage_mapper.grid_units
        node_strings = []
        if self.coverage_mapper.bc_arc_id_to_grid_ids:
            node_strings = sorted(self.coverage_mapper.bc_arc_id_to_grid_ids.items())

        if len(node_strings) > 100:
            msg = 'SRH-W supports a maximum of 100 node strings (monitor lines, BC lines). ' \
                  f'This simulation has {len(node_strings)} node strings defined.'
            self._logger.error(msg)

        writer = GeomWriter(file_name=file_name, ugrid=ugrid, grid_name=grid_name, grid_units=grid_units,
                            node_strings=node_strings)
        writer.write()
        self._logger.info('Success writing SRH-W geometry file.')

    def export_materials(self):
        """Exports the SRH material file."""
        mat_names = ['unassigned', 'channel', 'overland']
        my_dict = {0: [], 1: [], 2: []}
        self._logger.info('Writing SRH-W material file.')
        ugrid = self.sim_query_helper.co_grid.ugrid
        for cell_idx in range(ugrid.cell_count):
            if cell_idx + 1 in self._stream_cell_ids:
                my_dict[1].append(cell_idx)
            else:
                my_dict[2].append(cell_idx)
        base_name = f'{self.project_name}.srhmat'
        file_name = os.path.join(self.out_dir, base_name)
        self.files_exported.append(f'HydroMat "{base_name}"')
        writer = MaterialWriter(file_name=file_name, mat_names=mat_names, mat_grid_cells=my_dict)
        writer.write()
        self._logger.info('Success writing SRH-W material file.')

    def export_spatial_land_soil(self):
        """Exports the spatial land soil file."""
        self._logger.info('Writing SRH-W spatial land soil file.')
        base_name = f'{self.project_name}_land_soil.dat'
        self.files_exported.append(f'LandSoil "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        null_land_use_cells = []
        null_soil_type_cells = []
        with open(file_name, 'w') as f:
            f.write('Cell\tSoil\tLC\tChannel\n')
            ugrid = self.sim_query_helper.co_grid.ugrid
            for cell_idx in range(ugrid.cell_count):
                f.write(f'{cell_idx + 1}')
                if cell_idx in self.cells_to_soil_type:
                    f.write(f'\t{self.cells_to_soil_type[cell_idx]}')
                else:
                    f.write('\t-9999')
                    null_soil_type_cells.append(cell_idx)
                land_use_index = -9999
                if cell_idx in self.cells_to_land_use:
                    land_use = self.cells_to_land_use[cell_idx]
                    land_use_indices = self.land_use_data.index[self.land_use_data["ID"] == land_use].tolist()
                    if land_use_indices:
                        land_use_index = land_use_indices[0] + 1
                f.write(f'\t{land_use_index}')
                if land_use_index == -9999:
                    null_land_use_cells.append(cell_idx)
                channel_identifier = 1
                if cell_idx + 1 in self._stream_cell_ids:
                    channel_identifier = 2
                elif cell_idx + 1 in self._adj_stream_cell_ids:
                    channel_identifier = 3
                f.write(f'\t{channel_identifier}\n')
        if null_land_use_cells:
            self._logger.warning(f'The following cells do not have coverage for land use data: {null_land_use_cells}')
        elif null_soil_type_cells:
            self._logger.warning(f'The following cells do not have coverage for soil type data: {null_soil_type_cells}')
        else:
            self._logger.info('Success writing SRH-W spatial land soil file.')

    def read_land_use_table(self):
        """Reads the land use table information."""
        if self.land_use_data is None:
            land_defaults = land_use_defaults()
            num_columns = len(land_defaults)
            land_use_file = os.path.join(os.path.dirname(self.sim_component.main_file), 'land_use_table.json')
            if os.path.isfile(land_use_file):
                self.land_use_data = pandas.read_json(land_use_file)
            if self.land_use_data is None:
                self.land_use_data = pandas.DataFrame(land_defaults)
                self._logger.info('No land use table has been set. Using default table.')
            elif len(self.land_use_data.columns) != num_columns:
                self.land_use_data = pandas.DataFrame(land_defaults)
                self._logger.info("The number of fields in the land use table is not correct. The land use table "
                                  "was reinitialized to use the default table.")
            # Always start dataframe index at 1
            self.land_use_data.index = range(1, len(self.land_use_data) + 1)

    def read_soil_type_table(self):
        """Reads the land use table information."""
        if self.soil_type_data is None:
            soil_defaults = soil_type_defaults()
            num_columns = len(soil_defaults)
            soil_type_file = os.path.join(os.path.dirname(self.sim_component.main_file), 'soil_type_table.json')
            if os.path.isfile(soil_type_file):
                self.soil_type_data = pandas.read_json(soil_type_file)
            if self.soil_type_data is None:
                self.soil_type_data = pandas.DataFrame(soil_defaults)
                self._logger.info('No soil type table has been set. Using default table.')
            elif len(self.soil_type_data.columns) != num_columns:
                self.soil_type_data = pandas.DataFrame(soil_defaults)
                self._logger.info("The number of fields in the soil type table is not correct. The soil type table "
                                  "was reinitialized to use the default table.")
            # Always start dataframe index at 1
            self.soil_type_data.index = range(1, len(self.soil_type_data) + 1)

    def export_land_use_table(self):
        """Exports the land use table to a file."""
        self._logger.info('Writing SRH-W land use table file.')
        base_name = f'{self.project_name}_land_use.dat'
        self.files_exported.append(f'LandUseTable "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        self.read_land_use_table()
        num_land_use = len(self.land_use_data['ID'])
        with open(file_name, 'w') as f:
            f.write(f'NUMLC\t{num_land_use}\n')
            f.write('INDEX\tSHDFAC\tDROOT\tALBMIN\tALBMAX\tROUGH\tcint\tDs\n')
            for _, row in self.land_use_data.iterrows():
                f.write(f'{row["ID"]}\t')
                f.write(f'{row["Shade Factor"]}\t')
                f.write(f'{row["Root Depth"]}\t')
                f.write(f'{row["Min Albedo"]}\t')
                f.write(f'{row["Max Albedo"]}\t')
                mannings_n = row["Manning's Roughness"]
                f.write(f'{mannings_n}\t')
                f.write(f'{row["Canopy Cover Coefficient"]}\t')
                f.write(f'{row["Surface Depression Storage Volume"]}\t')
                f.write(f'#\t{row["Name"]}\n')
        self._logger.info('Success writing SRH-W land use table file.')

    def export_soil_type_table(self):
        """Exports the soil type table to a file."""
        self._logger.info('Writing SRH-W soil type table file.')
        base_name = f'{self.project_name}_soil_type.dat'
        self.files_exported.append(f'SoilTypeTable "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        self.read_soil_type_table()
        num_soil_type = len(self.soil_type_data['ID'])
        with open(file_name, 'w') as f:
            f.write(f'NUMSOIL\t{num_soil_type}\n')
            f.write('INDEX\tKINF\tKSATH\tDINF\tPOROSITY\tFC\tRM\tALPHA\tBETA\tSIGX\tSIGN\tPSIMIN\tMACPORE_Y\t'
                    'MACPORE_K\tMACPORE_FRAC\tCs\tC_ice\tKt\tFs\tTs_Depth\n')
            for _, row in self.soil_type_data.iterrows():
                f.write(f'{row["ID"]}\t')
                f.write(f'{row["Vertical Hydraulic Conductivity"]}\t')
                f.write(f'{row["Lateral Hydraulic Conductivity"]}\t')
                f.write(f'{row["Vertical Depth"]}\t')
                f.write(f'{row["Porosity"]}\t')
                f.write(f'{row["Field Capacity"]}\t')
                f.write(f'{row["Residual Soil Moisture"]}\t')
                f.write(f'{row["Alpha"]}\t')
                f.write(f'{row["Beta"]}\t')
                f.write(f'{row["Sigmoid X"]}\t')
                f.write(f'{row["Sigmoid N"]}\t')
                f.write(f'{row["Psi Min"]}\t')
                f.write(f'{row["Macro Pore Depth"]}\t')
                f.write(f'{row["Macro Pore Hydraulic Conductivity"]}\t')
                f.write(f'{row["Macro Pore Volume Fraction"]}\t')
                f.write(f'{row["Volume Specific Heat"]}\t')
                f.write(f'{row["Ice Latent Heat"]}\t')
                f.write(f'{row["Soil Thermal Conductivity"]}\t')
                f.write(f'{row["Damping Coefficient"]}\t')
                f.write(f'{row["Ts Depth"]}\t')
                f.write(f'#\t{row["Name"]}\n')
        self._logger.info('Success writing SRH-W soil type table file.')

    def export_erosion(self):
        """Exports the erosion tables to a file."""
        self._logger.info('Writing SRH-W erosion table file.')
        base_name = f'{self.project_name}_erosion.dat'
        self.files_exported.append(f'ErosionTable "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        self.read_land_use_table()
        with open(file_name, 'w') as f:
            num_soil_type = len(self.soil_type_data['ID'])
            f.write(f'NUMSOIL\t{num_soil_type}\n')
            f.write('SOIL_ID KUSLE Krain	Fraction_Size_1 Max_Allow_EroD(m) Max_Thickness(m)\n')
            for _, row in self.soil_type_data.iterrows():
                f.write(f'{row["ID"]}\t')
                f.write('0.30  508345	1.0             0.54		100.0\n')
            f.write('\n')
            num_land_use = len(self.land_use_data['ID'])
            f.write(f'NUMLC\t{num_land_use}\n')
            f.write('LULC_ID	CUSLE PUSLE Frac_Grnd_Cov Canopy_Cov_Factor(last_2_not_used)\n')
            for _, row in self.land_use_data.iterrows():
                f.write(f'{row["ID"]}\t')
                f.write('0.0005	1    0.05		0.98\n')
        self._logger.info('Success writing SRH-W erosion table file.')

    def export_rainfall(self):
        """Exports the rainfall data to a file."""
        self._logger.info('Writing SRH-W rainfall file.')
        dfs = self._get_rainfall_dataframes()
        if dfs:
            base_name = f'{self.project_name}_rainfall.dat'
            self.files_exported.append(f'Rainfall "{base_name}"')
            file_name = os.path.join(self.out_dir, base_name)
            with open(file_name, 'w') as f:
                for index, df in dfs.items():
                    f.write(f'RAIN_TS {index}\n')
                    f.write('TIME\tPRCP\tSFCTMP\tRH\tSFCSPD\tSOLAR\tLONGWV\tPRES\n')
                    f.write('TS\tkg/m2/s\tK\t%\tm/s\tW/m2\tW/m2\tPa\n')
                    time_vals = df[MET_TIME_HEADER]
                    rainfall_vals = df[MET_RAINFALL_TOTAL_HEADER]
                    temp_vals = df[MET_SURFACE_TEMP_HEADER]
                    humid_vals = df[MET_REL_HUMIDITY_HEADER]
                    wind_vals = df[MET_SURFACE_WIND_SPEED_HEADER]
                    radiation_vals = df[MET_SOLAR_RADIATION_HEADER]
                    longwave_vals = df[MET_LONGWAVE_RADIATION_HEADER]
                    pressure_vals = df[MET_AIR_PRESSURE_HEADER]
                    mc = self.sim_component.data
                    begin_time = datetime.strptime(mc.watershed.start_date_time, '%Y-%m-%d %H:%M')
                    for idx, cur_minutes in enumerate(time_vals):
                        cur_time = begin_time + timedelta(minutes=cur_minutes)
                        # rainfall in mm/s (kg/m2/s)
                        if idx == 0:
                            rainfall = 0.0
                        else:
                            delta_time_sec = (time_vals.values[idx] - time_vals.values[idx - 1]) * 60.0
                            rainfall = rainfall_vals.values[idx] / delta_time_sec
                        rainfall_string = "%11.9lf" % rainfall
                        f.write(f"{cur_time.strftime('%Y-%m-%d %H:%M')}\t{rainfall_string}\t"
                                f"{float(temp_vals.values[idx])}\t{float(humid_vals.values[idx])}\t"
                                f"{float(wind_vals.values[idx])}\t{float(radiation_vals.values[idx])}\t"
                                f"{float(longwave_vals.values[idx])}\t{float(pressure_vals.values[idx])}\n")
                    f.write('\n')
            self._logger.info('Success writing SRH-W rainfall file.')
        else:
            self._logger.warning('No SRH-W rainfall data has been defined.')

    def export_rainfall_cell_mapping(self):
        """Exports the mapping of the rainfall points to cells to a file."""
        self._logger.info('Writing SRH-W rainfall mapping file.')
        if 'Rainfall' in self.sim_query_helper.coverages:
            cov = self.sim_query_helper.coverages['Rainfall'][0]
            if cov is not None:
                base_name = f'{self.project_name}_radar_att.dat'
                self.files_exported.append(f'RadarAtt "{base_name}"')
                file_name = os.path.join(self.out_dir, base_name)
                with open(file_name, 'w') as f:
                    f.write('CELL\tRain_METEO\n')
                    ugrid = self.sim_query_helper.co_grid.ugrid
                    for cell_idx in range(ugrid.cell_count):
                        _, center = ugrid.get_cell_centroid(cell_idx)
                        pt_id = _get_closest_pt(center, cov.get_points(FilterLocation.PT_LOC_DISJOINT))
                        f.write(f'{cell_idx + 1}\t{pt_id}\n')
        self._logger.info('Success writing SRH-W rainfall mapping file.')

    def _get_rainfall_dataframes(self):
        """Gets the rainfall data as a dict of dataframes."""
        dfs = {}
        if self.sim_query_helper.rainfall_component is not None:
            defaults = rainfall_defaults()
            directory = os.path.dirname(self.sim_query_helper.rainfall_component.main_file)
            files = [f for f in os.listdir(directory) if os.path.isfile(os.path.join(directory, f))]
            for file in files:
                if RAINFALL_PT_XY_SERIES in file:
                    index = int(file[len(RAINFALL_PT_XY_SERIES):-len('.json')])
                    df = self.sim_query_helper.rainfall_component.read_xy_series(index)
                    if df is not None:
                        dfs[index] = df
                    else:
                        self._logger.warning(f"The meteorologic data for point with index {index} was not defined "
                                             "correctly. This point has been reset and has not been written.")
                        dfs[index] = defaults
        return dfs

    def get_land_use(self):
        """Gets the Land Use indices for each cell."""
        raster_paths = self.sim_component.data.watershed.landuse_raster_string.split('|')
        raster_items = []
        for name in raster_paths:
            item = tree_util.item_from_path(self.query.project_tree, name)
            if item:
                raster_items.append(item)
        if not raster_items:
            self._logger.error(f'Cannot get the landuse rasters: {raster_paths}.')
            return
        land_use_to_cells = {}
        for item in raster_items:
            raster_filename = self.query.item_with_uuid(item.uuid)
            self._logger.info(f'Determining land use type for cells from {raster_filename}...')
            raster = RasterInput(raster_filename)
            ugrid = self.sim_query_helper.co_grid.ugrid
            project_wkt = self.sim_query_helper.grid_wkt
            project_sr = osr.SpatialReference()
            h_wkt = gu.strip_vertical(project_wkt)
            project_sr.ImportFromWkt(h_wkt)
            raster_wkt = raster.wkt
            raster_sr = osr.SpatialReference()
            h_wkt = gu.strip_vertical(raster_wkt)
            raster_sr.ImportFromWkt(h_wkt)
            coord_trans = osr.CreateCoordinateTransformation(project_sr, raster_sr)
            not_found = []
            for cell_idx in range(ugrid.cell_count):
                _, center = ugrid.get_cell_centroid(cell_idx)
                if coord_trans:
                    center = coord_trans.TransformPoint(center[0], center[1], center[2])
                land_use = raster.get_raster_value_at_loc(center[0], center[1], False)
                if land_use != raster.nodata_value:
                    # Default the land use ID to -9999 if the land use is not found in the landuse table.
                    int_land_use = int(land_use)
                    landuse_id = -9999
                    if self.land_use_data is None:
                        landuse_id = int_land_use
                    else:
                        for land_index in self.land_use_data["ID"]:
                            if land_index == int_land_use:
                                landuse_id = land_index
                                break
                    if landuse_id == -9999 and int_land_use not in not_found:
                        self._logger.warning(f'Land use with ID {int_land_use} not found in the land use table. A '
                                             f'NODATA ID ({landuse_id}) will be written for this land use.')
                        not_found.append(int_land_use)
                    if landuse_id in land_use_to_cells.keys():
                        land_use_to_cells[landuse_id].append(cell_idx)
                    else:
                        land_use_to_cells[landuse_id] = [cell_idx]
        land_use_to_cells = dict(sorted(land_use_to_cells.items()))
        for land_use_id, cell_ids in land_use_to_cells.items():
            for cell_id in cell_ids:
                self.cells_to_land_use[cell_id] = land_use_id
        self.cells_to_land_use = dict(sorted(self.cells_to_land_use.items()))

    def get_soil_type(self):
        """Gets the Soil Type indices for each cell."""
        # Create dict with soil type as key and feature polygon ID list as value
        shapefile_paths = self.sim_component.data.watershed.soil_shapefile_string.split('|')
        shapefile_items = []
        for name in shapefile_paths:
            item = tree_util.item_from_path(self.query.project_tree, name)
            if item:
                shapefile_items.append(item)
        if not shapefile_items:
            self._logger.error(f'Cannot get the shapefiles: {shapefile_paths}.')
            return
        soil_type_to_cells = {}
        for item in shapefile_items:
            shapefile_filename = self.query.item_with_uuid(item.uuid)
            self._logger.info(f'Converting soil shapefile {shapefile_filename} to features...')
            converter = ShapefileConverter(shapefile_filename, self.sim_query_helper.grid_wkt,
                                           self.sim_query_helper.co_grid.ugrid.extents)
            do_cov = converter.convert_polygons()
            if not do_cov:
                self._logger.error("Unable to get the spatial soil information.")
                return
            feature_id_to_att_id = converter.feature_id_to_att_id

            # Use xms.snap to create dict with polygon IDs as key and cell ID list as value
            cov_polys = do_cov.polygons
            self._logger.info('Assigning soil polygons to mesh nodes...')
            self.polygon_snapper.add_polygons(polygons=cov_polys)
            polygon_ids_to_cell_ids = {}
            self._logger.info('Building soil polygon cell lists...')
            for poly in do_cov.polygons:
                polygon_ids_to_cell_ids[poly.id] = self.polygon_snapper.get_cells_in_polygon(poly.id)
            self._logger.info('Assigning soil types to cells...')
            not_found = []
            for poly_id, cell_ids in polygon_ids_to_cell_ids.items():
                if cell_ids:
                    # Default the soil ID to -9999 if the soil texture is not found in the soil table.
                    soil_id = -9999
                    if self.soil_type_data is not None:
                        for _, row in self.soil_type_data.iterrows():
                            if row['Name'].lower() == feature_id_to_att_id[poly_id].lower():
                                soil_id = row["ID"]
                                break
                    if soil_id == -9999 and feature_id_to_att_id[poly_id] not in not_found:
                        self._logger.warning(f'Soil with name {feature_id_to_att_id[poly_id]} not found in the soil '
                                             f'type table. A NODATA ID ({soil_id}) will be written for this soil.')
                        not_found.append(feature_id_to_att_id[poly_id])
                    if soil_id in soil_type_to_cells.keys():
                        soil_type_to_cells[soil_id].extend(list(cell_ids))
                    else:
                        soil_type_to_cells[soil_id] = list(cell_ids)
                    soil_type_to_cells[soil_id].sort()
        soil_type_to_cells = dict(sorted(soil_type_to_cells.items()))
        for soil_type, cell_ids in soil_type_to_cells.items():
            for cell_id in cell_ids:
                self.cells_to_soil_type[cell_id] = soil_type
        self.cells_to_soil_type = dict(sorted(self.cells_to_soil_type.items()))

    def get_streams(self):
        """Gets the stream cell identifiers for each cell."""
        stream_path = self.sim_component.data.watershed.stream_coverage_string
        stream_coverage_item = tree_util.item_from_path(self.query.project_tree, stream_path)
        if not stream_coverage_item:
            self._logger.error(f'Cannot get the stream coverage: {stream_path}.')
            return
        self._logger.info('Assigning streams to cells...')
        stream_cov = self.query.item_with_uuid(stream_coverage_item.uuid)
        arcs = stream_cov.arcs
        ugrid = self.sim_query_helper.co_grid.ugrid
        points, polys = _get_cell_polygons(ugrid)
        mpi = MultiPolyIntersector(points, polys)
        for arc in arcs:
            arc_pts = arc.get_points(FilterLocation.PT_LOC_ALL)
            for i in range(len(arc_pts) - 1):
                poly_ids, _, _ = mpi.traverse_line_segment((arc_pts[i].x, arc_pts[i].y, arc_pts[i].z),
                                                           (arc_pts[i + 1].x, arc_pts[i + 1].y, arc_pts[i + 1].z))
                for poly_id in poly_ids:
                    if poly_id == -1:
                        continue
                    self._stream_cell_ids.add(poly_id)
        for cell in self._stream_cell_ids:
            adj_cells = ugrid.get_cell_adjacent_cells(cell - 1)
            for adj_cell in adj_cells:
                if adj_cell + 1 not in self._stream_cell_ids:
                    self._adj_stream_cell_ids.add(adj_cell + 1)

    @property
    def polygon_snapper(self):
        """Lazy initialization getter so we only construct the polygon snapper if we need it."""
        if self._polygon_snapper is None:
            self._logger.info('Building polygon snapper for mesh...')
            self._polygon_snapper = SnapPolygon()
            self._polygon_snapper.set_grid(grid=self.sim_query_helper.co_grid, target_cells=True)
        return self._polygon_snapper

    def export_hydro(self, run_name=None):
        """Exports the SRH hydro file.

        Args:
            run_name (:obj:`str`): name used with scenario runs
        """
        message = 'Writing SRH-W hydro file'
        message = f'{message}.'
        self._logger.info(message)
        extension = ''
        file_name = os.path.join(self.out_dir, f'{self.project_name}.srhwhydro{extension}')
        writer = HydroWriter(file_name=file_name, model_control=self.sim_component.data,
                             file_list=self.files_exported, logger=self._logger, main_file=self.sim_component.main_file,
                             grid_units=self.coverage_mapper.grid_units)
        writer.materials_manning = [0.02, self.sim_component.data.watershed.channel_manning,
                                    self.sim_component.data.watershed.overland_manning]
        writer.write(run_name)
        message = 'Success writing SRH-W hydro file'
        message = f'{message}.'
        self._logger.info(message)

    def _get_dataset_info(self, data):
        dset_uuid, ts = uuid_and_time_step_index_from_string(data)
        dataset = self.query.item_with_uuid(dset_uuid)
        null_value = -999.0
        if not dataset:
            return None, null_value
        if ts > 0:
            ts -= 1
        ts_data = dataset.values[ts]
        if dataset.null_value is not None:
            null_value = dataset.null_value
        return ts_data, null_value

    def export_groundwater(self):
        """Exports the groundwater file."""
        self._logger.info('Writing SRH-W groundwater file.')
        bedrock_ts_data, null_value = self._get_dataset_info(self.sim_component.data.watershed.bedrock_dataset)
        unsat_ts_data, null_value = self._get_dataset_info(self.sim_component.data.watershed.unsat_dataset)
        sat_ts_data, null_value = self._get_dataset_info(self.sim_component.data.watershed.sat_dataset)
        init_soil_temp_ts_data, null_value = self._get_dataset_info(
            self.sim_component.data.watershed.init_soil_temp_dataset)
        if bedrock_ts_data is None:
            self._logger.error('Error writing groundwater data. No bedrock dataset has been selected.')
            return
        if unsat_ts_data is None:
            self._logger.error('Error writing groundwater data. No unsaturated depth dataset has been selected.')
            return
        if sat_ts_data is None:
            self._logger.error('Error writing groundwater data. No saturated depth dataset has been selected.')
            return
        if init_soil_temp_ts_data is None:
            self._logger.error('Error writing groundwater data. No initial soil temp dataset has been selected.')
            return
        ugrid = self.sim_query_helper.co_grid.ugrid
        bedrock_points = len(bedrock_ts_data) != ugrid.cell_count
        unsat_points = len(unsat_ts_data) != ugrid.cell_count
        sat_points = len(sat_ts_data) != ugrid.cell_count
        init_soil_temp_points = len(init_soil_temp_ts_data) != ugrid.cell_count
        if bedrock_points or unsat_points or sat_points or init_soil_temp_points:
            self._logger.error('Error writing groundwater data. All selected datasets must be cell datasets.')
            return
        base_name = f'{self.project_name}_spatial_groundwater.dat'
        self.files_exported.append(f'GroundwaterFile "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        null_bedrock_cells = []
        null_unsat_cells = []
        null_sat_cells = []
        null_init_soil_temp_cells = []
        with open(file_name, 'w') as f:
            f.write('Cell\tBedrock\tYunsat\tYsat\tLAI\tSoilTemp\n')
            for cell_idx in range(ugrid.cell_count):
                f.write(f'{cell_idx + 1}')
                val = bedrock_ts_data[cell_idx]
                if val != null_value:
                    f.write(f'\t{val}')
                else:
                    f.write('\t-9999')
                    null_bedrock_cells.append(cell_idx)
                val = unsat_ts_data[cell_idx]
                if val != null_value:
                    f.write(f'\t{val}')
                else:
                    f.write('\t-9999')
                    null_unsat_cells.append(cell_idx)
                val = sat_ts_data[cell_idx]
                if val != null_value:
                    f.write(f'\t{val}')
                else:
                    f.write('\t-9999')
                    null_sat_cells.append(cell_idx)
                f.write('\t1')
                val = init_soil_temp_ts_data[cell_idx]
                if val != null_value:
                    f.write(f'\t{val}\n')
                else:
                    f.write('\t-9999\n')
                    null_init_soil_temp_cells.append(cell_idx)
        if null_bedrock_cells:
            self._logger.warning(f'The following cells have null values for bedrock: {null_bedrock_cells}')
        else:
            self._logger.info('Success writing SRH-W bedrock groundwater data.')
        if null_unsat_cells:
            self._logger.warning(f'The following cells have null values for unsaturated depth: {null_unsat_cells}')
        else:
            self._logger.info('Success writing SRH-W unsaturated depth groundwater data.')
        if null_sat_cells:
            self._logger.warning(f'The following cells have null values for saturated depth: {null_sat_cells}')
        else:
            self._logger.info('Success writing SRH-W saturated depth groundwater data.')
        if null_init_soil_temp_cells:
            self._logger.warning('The following cells have null values for initial soil temperature: '
                                 f'{null_init_soil_temp_cells}')
        else:
            self._logger.info('Success writing SRH-W initial soil temp groundwater data.')

    def export_sif_file(self):
        """Exports the SIF file (SRHPre Input File)."""
        self._logger.info('Writing SRH-W SIF file.')
        base_name = f'{self.project_name}_SIF.dat'
        self.files_exported.append(f'SifFile "{base_name}"')
        file_name = os.path.join(self.out_dir, base_name)
        with open(file_name, 'w') as f:
            mc = self.sim_component.data
            f.write('// Simulation Description (not used by SRH):\n')
            f.write(f'{mc.hydro.simulation_description}\n')
            f.write('// Solver Module Selected (RIVER WATERSHED COAST or 3D)\n')
            f.write('WATERSHED\n')
            f.write('// Solver Selection (FLOW MOBile WQ TCUR TEM DIFF_EX DIFF_IM SED_DIFF_IM DYNAMIC ...)\n')
            if self._sediment:
                f.write('SED_DIFF_IM\n')
            else:
                f.write('DIFF_IM\n')
            f.write('// Monitor-Point-Info: NPOINT\n')
            f.write('0\n')
            f.write('// Time Parameters: T_Start(hr) T_end(hr) Dt(s) [Dt_max(s) CFL]\n')
            f.write(f'{mc.hydro.start_time} {mc.hydro.end_time} {mc.hydro.time_step} 60.0 0.5\n')
            f.write('// Mesh-Unit (FOOT METER INCH MM MILE KM GSCALE)\n')
            units = gu.get_horiz_unit_from_wkt(self.sim_query_helper.grid_wkt)
            if units == gu.UNITS_METERS:
                f.write('METER\n')
            else:
                f.write('FOOT\n')
            f.write('// Mesh FILE_NAME and FORMAT(SMS...)\n')
            f.write(f'{self.project_name}.srhgeom SRHGEOM\n')
            f.write('// Precipitation Time Period: Rain_Start Rain_end in hours\n')
            # Get the ending time of the precipitation in hours
            dfs = self._get_rainfall_dataframes()
            end_time_hours = 0
            for key in dfs:
                df = dfs[key]
                time_vals = df[MET_TIME_HEADER]
                if not time_vals.empty:
                    end_time_hours = time_vals.values[-1] / 60
                break
            if end_time_hours == 0:
                self._logger.warning("An end time of 0 was defined for the precipitation.  Make sure rainfall values "
                                     "have been defined in the SRH-W rainfall coverage.")
            f.write(f'0 {end_time_hours}\n')
            f.write('// Rainfall Data Options: Constant Gage Radar\n')
            f.write('RADAR\n')
            f.write('// Rainfall Radar data file input: DATE Time FNAME_RADAR LinkType FNAME_LOC [CellSize]\n')
            f.write(f'{mc.watershed.start_date_time} {self.project_name}_rainfall.dat 2 '
                    f'{self.project_name}_radar_att.dat\n')
            if self._sediment:
                f.write('// General Sediment Parameters: spec_grav sed_nclass\n')
                f.write('2.65 1\n')
                f.write('// Size-Class Diameter & Dry_Bulk_Density: D_Lower(mm) D_Upper(mm) [Den_Bulk UNIT]\n')
                f.write('0.00035  0.00035\n')
                f.write('// Sediment Capacity Eqn and Method\n')
                f.write('KILINC\n')
                f.write('// Water Temperature (Celsius):\n')
                f.write('\n')
                f.write('// Adaptation Coefs for Suspended Load: A_DEP A_ERO (0.25 1.0 are defaults)\n')
                f.write('1.0 0.25\n')
            f.write('// Initial Flow Condition Setup Option (DRY RST AUTO ZONAL Vary_WSE/Vary_WD)\n')
            if mc.hydro.initial_condition == 'Dry':
                f.write('DRY\n')
            elif mc.hydro.initial_condition == 'Automatic':
                f.write('AUTO\n')
            elif mc.hydro.initial_condition == 'Initial Water Surface Elevation':
                f.write('WSE\n')
                f.write('1\n')
                units_str = 'EN'
                if mc.hydro.initial_water_surface_elevation_units == 'Meters':
                    units_str = 'SI'
                f.write(f'1 0 0 {mc.hydro.initial_water_surface_elevation} {units_str}\n')
            elif mc.hydro.initial_condition == 'Restart File':
                f.write('RST\n')
                f.write('restart.rst\n')
            elif mc.hydro.initial_condition == 'Water Surface Elevation Dataset':
                unit_str = 'SI' if self._grid_units.endswith('"METER"') else 'EN'
                f.write('Vary_WSE\n')
                f.write(f'"wse.2dm" {unit_str}\n')
            f.write('// Infiltration Model: ZERO GREEN COUPLE\n')
            f.write('COUPLE\n')
            f.write('//  LAND-USE-&-SOIL-TYPE-DISTRIBUTION-FILE\n')
            f.write(f'{self.project_name}_land_soil.dat\n')
            f.write('// Landuse input parameters file\n')
            f.write('//    7 parameters for each landuse class; the 7 include the following:\n')
            f.write('//         SHDFAC=shade factor; DROOT=root depth; ALBMIN=Minimum Albedo; ALBMAX=Maximum Albedo '
                    '<-- Used by ET Module?\n')
            f.write('//         ROUGH=Manning coefficient; cint=interception (0 means intercept is not computed even '
                    'if ET is activated)\n')
            f.write('//         Ds=maximum depression storage <-- rainfall fill this depth first before it is '
                    'available for runoff (different from my Storage which is an initial ponded area)\n')
            f.write('//         Note: Last string is the Land Type Name (2nd last is ignored)\n')
            f.write(f'{self.project_name}_land_use.dat\n')
            f.write('// Soil Type input parameters file\n')
            f.write('//   18 parameters for each of 19 soil types; 18 include the following:\n')
            f.write('//         KINF = vertical hydraulic conductivity; KSATH=Horizontal hydraulic conductivity\n')
            f.write('//         POROSITY; FC=soil-moisture-content filed capacity; RM=soil-moisture-content residual\n')
            f.write('//         ALPHA; BETA; SIGX=sigmoid_X; SIGN=sigmoid_N; PSIMIN;\n')
            f.write('//         MACPORE_Y=Macro Pore Depth; MACPORE_K=Macro Pore hydraulic conductivity; '
                    'MACPORE_FRAC=Macro Pore volume fraction\n')
            f.write('//         Cs=volume speciifc heat; C_ice=ice latent heat; Kt=soil thermal conductivity; '
                    'Fs=damping coefficient; Ts_Depth\n')
            f.write('//       Last line gives the "surface-subsurface exchange depth" - one constant for entire '
                    'watershed\n')
            f.write(f'{self.project_name}_soil_type.dat\n')
            f.write('//  Groundwater Spatial Distrbution File\n')
            f.write('//   each cell is assigned with: Bedrock elevation(m); Unsaturated Zone Depth (m); Saturated '
                    'Zone Depth (m)\n')
            f.write('//         METEO=Meteorology data ID (needed only if Radar rainfall is used);\n')
            f.write('//         LAI=LAI Type (for ET ...);\n')
            f.write('//         BEDROCK=Bedrock elevation\n')
            f.write('//         Yu_Init=unsaturated zone Depth/Elev; Ya_init=Saturated Depth/Elev; Soil_Temp=Soil '
                    'temperature in Kelvin;\n')
            f.write(f'{self.project_name}_spatial_groundwater.dat\n')
            f.write('// SPECIAL-MODULE-SELECTION(0/1): ET SNOW FREEZE\n')
            f.write('0 0 0\n')
            if self._sediment:
                f.write('// Soil Erosion Parameters Input File: Fname_Ero\n')
                f.write(f'{self.project_name}_erosion.dat\n')
            f.write('// Special-OUTPUT(Abban): Infiltration_Rate_Output(0/1) N_Capillary_Output [CELL_ID_List]\n')
            f.write('1 1 1\n')
            f.write('// Channel Network Solver: NoUSE Explicit SRH1DS 2D\n')
            f.write('NOUSE\n')
            node_strings = []
            if self.coverage_mapper.bc_arc_id_to_grid_ids:
                node_strings = sorted(self.coverage_mapper.bc_arc_id_to_grid_ids.items())
            for count in range(len(node_strings)):
                f.write(f'// Boundary Type (INLET-Q EXIT-H etc) for Nodestring {count + 1}\n')
                f.write('EXIT-EX\n')
            f.write('// Results-Output-Format-and-Unit(SRHC/TEC/SRHN/XMDF/XMDFC/PARA;SI/EN) + Optional STL FACE\n')
            out_unit = 'EN' if mc.output.output_units == 'English' else 'SI'
            format_str = mc.output.output_format if mc.output.output_format != 'XMDF' else 'XMDFC'
            f.write(f'{format_str} {out_unit}\n')
            f.write('// Intermediate Result Output Control: INTERVAL(hour) OR List of T1 T2 ...  EMPTY means the end\n')
            if mc.output.output_method == 'Specified Frequency':
                # output frequency must be in hours
                val = mc.output.output_frequency
                if mc.output.output_frequency_units == 'Days':
                    val = val * 24
                elif mc.output.output_frequency_units == 'Minutes':
                    val = val / 60
                elif mc.output.output_frequency_units == 'Seconds':
                    val = val / (60 * 60)
                f.write(f'{val}\n')
            elif mc.output.output_method == 'Specified Times':
                times = mc.output.output_specified_times['Times'].tolist()
                for t in times:
                    f.write(f' {t}')
                f.write('\n')
            else:
                f.write('\n')
        self._logger.info('Success writing SRH-W SIF file.')
