"""Handles coverage data with updates."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData, VALUE_KEY
from xms.guipy.data.target_type import TargetType


# 4. Local modules


class CoverageData(CoverageBaseData):
    """Manages data file for a feature coverage component with updates."""

    def update_feature(self, target: TargetType, value: str, component_id: int):
        """
        Updates features.

        Args:
            target: The type of the features.
            value: value for the feature.
            component_id: The component ID.
        """
        # The length of the strings in the existing dataset might not be long enough.  If it is not, we need to lengthen
        # the dataset strings.
        existing_features = self._get_dataset(target)
        bytes_per_char = np.dtype('U1').itemsize
        bytes_per_feature = existing_features[VALUE_KEY].dtype.itemsize
        chars_per_feature = bytes_per_feature / bytes_per_char
        required_length = len(value)
        if chars_per_feature < required_length:
            new_dtype = np.dtype(f'<U{required_length}')
            existing_features[VALUE_KEY] = existing_features[VALUE_KEY].drop_encoding().astype(new_dtype)
            self._set_dataset(existing_features, target)
        existing_features[VALUE_KEY].loc[dict(comp_id=component_id)] = value
