"""Parent Qt widget for the individual tidal constituent checkboxes."""
# 1. Standard python modules

# 2. Third party modules
from harmonica.resource import MAX_NUM_CONS
import numpy as np
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QWidget
import xarray as xr

# 3. Aquaveo modules

# 4. Local modules
from xms.tides.gui.constituent_list_ui import Ui_ConstituentListWidget


NUM_COLS = 4


MINOR_CONS = [
    'K1',
    'M2',
    'N2',
    'O1',
    'S2',
    'K2',
    'M4',
    'M6',
    'Q1',
]


MAJOR_CONS = [
    'K1',
    'M2',
    'N2',
    'O1',
    'S2',
]


class ConstituentListWidget(QWidget):
    """The User Defined tidal consituents selector table widget."""
    def __init__(self, data_frame, parent=None):
        """Construct the widget.

        Args:
            data_frame (pandas.DataFrame): The constituent dataset
            parent (Something derived from :obj:`QObject`): The parent object.
        """
        super().__init__(parent)
        self.con_data = data_frame.sort_values(by=['name'])
        self.ui = Ui_ConstituentListWidget()
        self.ui.setupUi(self)
        self._setup_ui()

    def _setup_ui(self):
        """Setup the widgets in the dialog."""
        self._setup_constituent_checkboxes()
        self.ui.btn_all.clicked.connect(self.on_btn_all)
        self.ui.btn_none.clicked.connect(self.on_btn_none)
        self.ui.btn_minor.clicked.connect(self.on_btn_minor)
        self.ui.btn_major.clicked.connect(self.on_btn_major)

    def on_btn_all(self):
        """Turn on all the constituents."""
        for row in range(self.con_data.shape[0]):
            chk = self._get_checkbox(row)
            chk.setCheckState(Qt.Checked)

    def on_btn_none(self):
        """Turn off all the constituents."""
        for row in range(self.con_data.shape[0]):
            chk = self._get_checkbox(row)
            chk.setCheckState(Qt.Unchecked)

    def on_btn_minor(self):
        """Turn on all minor the constituents."""
        for row in range(self.con_data.shape[0]):
            chk = self._get_checkbox(row)
            if chk.text() in MINOR_CONS:
                chk.setCheckState(Qt.Checked)

    def on_btn_major(self):
        """Turn on all major the constituents."""
        for row in range(self.con_data.shape[0]):
            chk = self._get_checkbox(row)
            if chk.text() in MAJOR_CONS:
                chk.setCheckState(Qt.Checked)

    def change_model(self, df):
        """Change the tidal database model and thus the list of available constituents.

        Args:
            df (pandas.DataFrame): The constituent dataset
        """
        self.con_data = df.sort_values(by=['name'])
        self._setup_constituent_checkboxes()

    def get_dataset(self):
        """Get a dataset representing the current constituent list."""
        names = []
        states = []
        for row in range(self.con_data.shape[0]):
            chk = self._get_checkbox(row)
            names.append(chk.text())
            states.append(1 if chk.checkState() == Qt.Checked else 0)
        con_data = {
            'name': xr.DataArray(data=np.array(names, dtype=np.unicode_)),
            'enabled': xr.DataArray(data=np.array(states, dtype=np.int32)),
        }
        return xr.Dataset(data_vars=con_data)

    def _get_checkbox(self, idx):
        """Given an index, get the checkbox widget.

        Args:
            idx: Index of the constituent as defined harmonica

        Returns:
            QCheckbox: The checkbox associated with the given index
        """
        col = int(idx / NUM_COLS)
        row = idx % NUM_COLS
        checkbox_name = f'chk_{row}_{col}'
        return getattr(self.ui, checkbox_name)

    def _setup_constituent_checkboxes(self):
        """Set up the list of constituents and On/Off toggles.

        I realize this is a hokey way to implement this. I tried various models/views and built-in widgets, but nothing
        looked as good as manually adding all the checkboxes to a grid layout in the UI designer and programmatically
        setting the text and enabled state based on the current tidal harmonic model.
        """
        # Hide/show constituents, update text and values
        num_cons = self.con_data.shape[0]
        for row in range(num_cons):
            chk = self._get_checkbox(row)
            chk.setVisible(True)
            chk.setText(self.con_data.iloc[row, 0])
            chk.setCheckState(Qt.Checked if int(self.con_data.iloc[row, 1]) == 1 else Qt.Unchecked)
        for hidden_con in range(num_cons, MAX_NUM_CONS):
            chk = self._get_checkbox(hidden_con)
            chk.setVisible(False)
        # Unused checkboxes - just to keep the layout looking good.
        self.ui.chk_1_9.setVisible(False)
        self.ui.chk_2_9.setVisible(False)
        self.ui.chk_3_9.setVisible(False)
