"""Qt data model for the user defined constituent table."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtCore import Qt

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel

# 4. Local modules
from xms.tides.gui import gui_util

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


DSET_TREE_PATH_ROLE = Qt.UserRole + 1


class QxTidesPandasTableModel(QxPandasTableModel):
    """Class derived from QxPandasTableModel to handle a pandas DataFrame for tidal constituents."""

    def __init__(self, data_frame, parent=None):
        """Initializes the class.

        Args:
            data_frame (DataFrame): The pandas DataFrame.
            parent (Something derived from :obj:`QWidget`): The parent window.
        """
        super().__init__(data_frame, parent)
        self.tree_paths = {}

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Override to get the selected dataset tree path outside the model. Only want this string for display,
        don't want it stored in the DataFrame.

        Args:
            index (QModelIndex): The index.
            role (int): The role.

        Returns:
            object: The data at index, or None.
        """
        if not index.isValid():
            return None

        if role == DSET_TREE_PATH_ROLE:
            uuid = index.data()
            if uuid in self.tree_paths:
                return self.tree_paths[uuid]
            else:
                return gui_util.NULL_SELECTION

        return super().data(index, role)

    def setData(self, index, value, role=Qt.EditRole):  # noqa N802
        """Adjust the data (set it to <value>) depending on index and role.

        Override to store the selected dataset tree path outside the model. Only want this string for display,
        don't want it stored in the DataFrame.

        Args:
            index (QModelIndex): The index.
            value (): The value.
            role (int): The role.

        Returns:
            bool: True if successful; otherwise False.
        """
        if not index.isValid():
            return False

        if role == DSET_TREE_PATH_ROLE:
            self.tree_paths[index.data()] = value
            return True

        return super().setData(index, value, role)
