"""TrimCoverage class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from shapely import wkt
from shapely.geometry import Polygon as SHPolygon

# 3. Aquaveo modules
from xms.gdal.rasters import raster_utils as ru
from xms.gdal.utilities import gdal_utils as gu

# 4. Local modules
from xms.tool.utilities.coverage_conversion import convert_polygons_to_coverage, shapely_polygon_list_to_polygons


class PolygonFromRasterBounds:
    """Class to create a polygon in the coverage cov_name from the given raster's bounds."""

    def __init__(self, input_raster, cov_name, display_wkt, vertical_datum, vertical_units, logger):
        """Initializes the class.

        Args:
            input_raster (xms.gdal.rasters.raster_input.RasterInput): The input raster.
            cov_name (str): The name of the coverage to create.
            display_wkt (str): The WKT of the display projection
            vertical_datum (str): The vertical datum
            vertical_units (str): The vertical units
            logger (logging.logger): logger
        """
        self._logger = logger
        self._input_raster = input_raster
        self._cov_name = cov_name
        self._display_wkt = display_wkt
        self._vertical_datum = vertical_datum
        self._vertical_units = vertical_units

    def create_boundary_coverage(self):
        """Grabs the active region of a raster and sets it to a new raster."""
        vec_ds = ru.convert_raster_data_to_polygons(self._input_raster)
        vec_lyr = vec_ds.GetLayerByName('bounds')

        # Convert vec_lyr from the raster's projection to the display projection
        coord_trans = None
        display_wkt = None
        if self._display_wkt is not None:
            display_wkt = gu.add_vertical_to_wkt(self._display_wkt, self._vertical_datum, self._vertical_units, False)
            coord_trans = gu.get_coordinate_transformation(self._input_raster.wkt, display_wkt)

        # Loop on the polygon features, removing interior rings
        polygon_list = []
        vec_lyr.SetAttributeFilter('VALUE = 1')  # Filter only for polygons with val == 1
        for feature in vec_lyr:
            # Read in as a shapely polygon, and remove the interior polygons
            geom = feature.GetGeometryRef()
            if coord_trans is not None:
                geom.Transform(coord_trans)
            geom_wkt = geom.ExportToWkt()
            polygon_obj = wkt.loads(geom_wkt)
            gtype = polygon_obj.geom_type
            polygon_objs = [polygon_obj] if gtype == 'Polygon' else polygon_obj.geoms if gtype == 'MultiPolygon' else []
            for polygon in polygon_objs:
                polygon_no_holes = SHPolygon(polygon.exterior.coords, holes=None)
                polygon_list.append(polygon_no_holes)

        polygons = shapely_polygon_list_to_polygons(polygon_list)
        _display_wkt = display_wkt if display_wkt is not None and coord_trans is not None else self._input_raster.wkt
        new_cov = convert_polygons_to_coverage(polygons, self._cov_name, _display_wkt)
        vec_ds = None
        return new_cov
