"""MapActivity Algorithm."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from logging import Logger
import math

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules


def map_activity(output_dataset_name: str, value_reader: DatasetReader, activity_reader: DatasetReader,
                 activity_g: Grid, logger: Logger) -> DatasetWriter:
    """Tool to convert scalar datasets to a vector dataset.

    Args:
        output_dataset_name: The name for the output dataset
        value_reader: Dataset reader for the values
        activity_reader: Dataset reader for the activity
        activity_g: The grid for the activity dataset
        logger: Logger for user output

    Returns:
        The DatasetWriter for the output dataset
    """
    # Setup activity_reader's activity_calculator
    if activity_reader.num_activity_values is not None \
            and activity_reader.num_activity_values != activity_reader.num_values:
        activity_reader.activity_calculator = CellToPointActivityCalculator(activity_g.ugrid)

    # Setup Output Dataset Builder
    builder = DatasetWriter()
    builder.name = output_dataset_name
    builder.geom_uuid = value_reader.geom_uuid
    builder.num_components = value_reader.num_components
    builder.ref_time = value_reader.ref_time
    builder.time_units = value_reader.time_units
    builder.null_value = activity_reader.null_value
    builder.activity_calculator = activity_reader.activity_calculator

    time_count = len(value_reader.times)

    for tsidx in range(time_count):
        logger.info(f'Processing time step {tsidx + 1} of {time_count}...')
        activity_array = None
        # read values dataset
        values = value_reader.values[tsidx]
        # read activity dataset
        activity_values, activity_array = activity_reader.timestep_with_activity(tsidx)
        # direction if doing mag/dir conversion
        y_data, y_activity = activity_reader.timestep_with_activity(tsidx)
        if activity_array is not None:
            builder.use_activity_as_null = True
        elif activity_reader.null_value is not None:
            value_components = value_reader.num_components
            active_is_vector = activity_reader.num_components > 1
            for i, activity_value in enumerate(activity_values):
                is_active = True
                if active_is_vector:
                    if math.isnan(activity_value[0]):
                        is_active = False
                elif math.isnan(activity_value):
                    is_active = False
                if not is_active:
                    if value_components == 1:
                        values[i] = float("nan")
                    else:
                        values[i] = (float("nan"),) * value_components
        builder.append_timestep(value_reader.times[tsidx], values, activity_array)

    logger.info('Writing output dataset to XMDF file...')
    builder.appending_finished()

    return builder
