"""scalars_from_vectors Function."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from typing import Tuple

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.constraint.ugrid_activity import CellToPointActivityCalculator
from xms.datasets import vectors as xmdv
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter

# 4. Local modules
from xms.tool.utilities.dataset_tool import set_builder_activity_flags


def scalars_from_vectors(input_dataset: DatasetReader, input_dataset_grid: Grid, dataset_prefix: str, mag_name: str,
                         dir_name: str, vx_name: str, vy_name: str, logger: logging.Logger) \
        -> Tuple[DatasetWriter | None, DatasetWriter | None, DatasetWriter | None, DatasetWriter | None]:
    """
    Creates scalar datasets from a vector dataset.

    Args:
        input_dataset (DatasetReader): The dataset reader for the input data.
        input_dataset_grid (Grid): The Grid of the input data (or None).
        dataset_prefix (str): The prefix to add to each output dataset.
        mag_name (str): The name for the output magnitude dataset. The empty string skips creation of magnitude output.
        dir_name (str): The name for the output direction dataset. The empty string skips creation of direction output.
        vx_name (str): The name for the output vx dataset. Use the empty string to skip creation of vx output.
        vy_name (str): The name for the output vy dataset. Use the empty string to skip creation of vy output.
        logger (logging.Logger): The logger used to output progress info to the user.

    Returns:
        Returns a tuple of four DatasetWriter instances. Each entry in the tuple can be None.
    """
    # Extract the component, time step at a time
    # import os
    # self.logger.info(f'PID {os.getpid()}')

    # Local variables to hold DatasetWriter instances
    mag_builder = None
    dir_builder = None
    vx_builder = None
    vy_builder = None

    # Create a place for the output dataset file
    loc = 'points' if input_dataset.location not in ['cells', ('cells', 'cells')] else 'cells'

    if mag_name:
        dataset_name = f'{dataset_prefix} {mag_name}'
        mag_builder = DatasetWriter(
            name=dataset_name,
            geom_uuid=input_dataset.geom_uuid,
            num_components=1,
            ref_time=input_dataset.ref_time,
            time_units=input_dataset.time_units,
            null_value=input_dataset.null_value,
            location=loc,
        )
    if dir_name:
        dataset_name = f'{dataset_prefix} {dir_name}'
        dir_builder = DatasetWriter(
            name=dataset_name,
            geom_uuid=input_dataset.geom_uuid,
            num_components=1,
            ref_time=input_dataset.ref_time,
            time_units=input_dataset.time_units,
            null_value=input_dataset.null_value,
            location=loc,
        )
    if vx_name:
        dataset_name = f'{dataset_prefix} {vx_name}'
        vx_builder = DatasetWriter(
            name=dataset_name,
            geom_uuid=input_dataset.geom_uuid,
            num_components=1,
            ref_time=input_dataset.ref_time,
            time_units=input_dataset.time_units,
            null_value=input_dataset.null_value,
            location=loc,
        )
    if vy_name:
        dataset_name = f'{dataset_prefix} {vy_name}'
        vy_builder = DatasetWriter(
            name=dataset_name,
            geom_uuid=input_dataset.geom_uuid,
            num_components=1,
            ref_time=input_dataset.ref_time,
            time_units=input_dataset.time_units,
            null_value=input_dataset.null_value,
            location=loc,
        )

    dset_ugrid = None
    if input_dataset_grid:
        dset_ugrid = input_dataset_grid.ugrid
    if input_dataset.activity and input_dataset.values and dset_ugrid:
        if input_dataset.num_values != input_dataset.num_activity_values:
            # Nodal dataset values with cell activity
            input_dataset.activity_calculator = CellToPointActivityCalculator(dset_ugrid)
            if mag_builder is not None:
                mag_builder.activity_calculator = CellToPointActivityCalculator(dset_ugrid)
            if dir_builder is not None:
                dir_builder.activity_calculator = CellToPointActivityCalculator(dset_ugrid)
            if vy_builder is not None:
                vy_builder.activity_calculator = CellToPointActivityCalculator(dset_ugrid)
            if vx_builder is not None:
                vx_builder.activity_calculator = CellToPointActivityCalculator(dset_ugrid)

    time_count = len(input_dataset.times)
    for tsidx in range(time_count):
        logger.info(f'Processing time step {tsidx + 1} of {time_count}...')
        data, activity = input_dataset.timestep_with_activity(tsidx)
        set_builder_activity_flags(activity, [mag_builder, dir_builder, vx_builder, vy_builder])
        # if self._dataset_reader.activity is not None:
        #     # point data, cell activity
        #     if self._dataset_reader.num_activity_values != self._dataset_reader.num_values:
        #         activity = self._dataset_reader.activity[tsidx]
        #         # get point activity from cell activity for the ugrid
        #         from xms.constraint.ugrid_activity import active_points_from_cells
        #         point_activity = active_points_from_cells(self._ugrid.ugrid, activity)
        #         point_data = data.copy()
        #         point_data[point_activity == 0] = np.nan
        #         point_data_min = np.nanmin(point_data)
        #         point_data_max = np.nanmax(point_data)
        #         self.log.error('This tool does not work with cell based activity on point based data.')
        if mag_builder is not None:
            # Append the stuff for this time step
            values = xmdv.vx_vy_to_magnitude(vectors=data)
            mag_builder.append_timestep(input_dataset.times[tsidx], values, activity)
        if dir_builder is not None:
            # Append the stuff for this time step
            angles = xmdv.vx_vy_to_direction(data[:, 0], data[:, -1])
            dir_builder.append_timestep(input_dataset.times[tsidx], angles, activity)
        if vx_builder is not None:
            vx_builder.append_timestep(input_dataset.times[tsidx], data[:, 0], activity)
        if vy_builder is not None:
            vy_builder.append_timestep(input_dataset.times[tsidx], data[:, -1], activity)

    if mag_builder is not None:
        logger.info('Writing output magnitude dataset to XMDF file...')
        mag_builder.appending_finished()
    if dir_builder is not None:
        logger.info('Writing output direction dataset to XMDF file...')
        dir_builder.appending_finished()
    if vx_builder is not None:
        logger.info('Writing output Vx dataset to XMDF file...')
        vx_builder.appending_finished()
    if vy_builder is not None:
        logger.info('Writing output Vy dataset to XMDF file...')
        vy_builder.appending_finished()

    return mag_builder, dir_builder, vx_builder, vy_builder
