"""GridArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, contains_invalid_characters, IoDirection
from .data_handler import DataHandler


class GridArgument(Argument):
    """Grid tool argument."""
    def __init__(
        self,
        data_handler: DataHandler,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[str] = None,
        hide: Optional[bool] = False
    ):
        """Construct a dataset argument.

        Args:
            data_handler (DataHandler): Data handler to give list of potential datasets.
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[str]): Default value.
            hide (Optional[bool]): Should the argument be hidden (True) or visible?
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.data_handler = data_handler

    def _get_type(self) -> str:
        """Get a string representing the argument type (grid).

        Returns:
            (str): The argument type.
        """
        return 'grid'

    def _set_value(self, value: Optional[str]) -> None:
        """Set the argument value.

        Args:
            value (str): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {
            'type': 'StringSelector' if self.io_direction == IoDirection.INPUT else 'String'
        }
        interface_info = interface_info | super().get_interface_info()
        if self.io_direction == IoDirection.INPUT:
            grid_list = self.data_handler.get_available_grids().copy()
            grid_list.insert(0, Argument.NONE_SELECTED)
            if self.value not in grid_list:
                interface_info['value'] = Argument.NONE_SELECTED
            interface_info['choices'] = grid_list
        return interface_info

    def to_dict(self) -> dict:
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'GridArgument'}
        super_values = super(GridArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('uuid', self.data_handler.get_uuid_from_grid_name(self.value), values)
        return values

    def adjust_value_from_results(self) -> None:
        """Adjust the value obtained when running from previous results."""
        if self.io_direction == IoDirection.INPUT and \
                self.value not in self.data_handler.get_available_grids():
            self.value = None

    def validate(self):
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result  # Did not specify when required.

        return contains_invalid_characters(self)
