"""GridArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, contains_invalid_characters, IoDirection
from .data_handler import DataHandler


class RasterArgument(Argument):
    """Raster tool argument."""
    def __init__(
        self,
        data_handler: DataHandler,
        name: str,
        description: str = '',
        io_direction: Optional[IoDirection] = IoDirection.INPUT,
        optional: Optional[bool] = False,
        value: Optional[str] = None,
        hide: Optional[bool] = False
    ):
        """Construct a raster argument.

        Args:
            data_handler (DataHandler): Data handler to give list of potential rasters.
            name (Optional[str]): Python friendly argument name.
            description (Optional[str]): User friendly description of the argument.
            io_direction (Optional[IoDirection]): IO Direction of the argument (input or output).
            optional (Optional[bool]): Is the argument optional?
            value (Optional[str]): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
        """
        super().__init__(name, description, io_direction, optional, value, hide)
        self.data_handler = data_handler

    def _get_type(self):
        """Get a string representing the argument type (raster).

        Returns:
            (str): The argument type.
        """
        return 'raster'

    def _set_value(self, value):
        """Set the argument value.

        Args:
            value (int): The new argument value.
        """
        self._value = None if value is None else str(value)

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        interface_info = {
            'type': 'StringSelector' if self.io_direction == IoDirection.INPUT else 'String'
        }
        interface_info = interface_info | super().get_interface_info()
        if self.io_direction == IoDirection.INPUT:
            raster_list = self.data_handler.get_available_rasters().copy()
            raster_list.insert(0, Argument.NONE_SELECTED)
            if self.value not in raster_list:
                interface_info['value'] = Argument.NONE_SELECTED
            interface_info['choices'] = raster_list
        return interface_info

    def to_dict(self):
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'RasterArgument'}
        super_values = super(RasterArgument, self).to_dict()
        values.update(super_values)
        self._add_key_value('uuid', self.data_handler.get_uuid_from_raster_name(self.value), values)
        return values

    def validate(self):
        """Validate the argument.

        Returns:
            (str): An error string if invalid or None.
        """
        result = super().validate()
        if result is not None:
            return result

        if not self.optional:
            if self.io_direction == IoDirection.INPUT:
                raster_input = self.data_handler.get_input_raster(self.value)
                if raster_input is None:
                    return 'Could not read raster.'
            if self.io_direction == IoDirection.OUTPUT:
                err = contains_invalid_characters(self, only_basename=True)
                if err:
                    return f'Raster "{self.value}" contains an invalid character. Please enter a valid filename.'
                # Check if raster already exists in the list of possible input rasters.
                # If it does exist, put up an error message.
                filename = os.path.basename(self.value)
                out_raster_basename = os.path.splitext(filename)[0]
                rasters = self.data_handler.get_available_rasters()
                for raster_name in rasters:
                    raster_file = self.data_handler.get_input_raster_file(raster_name)
                    if raster_file:
                        raster_base, extension = os.path.splitext(os.path.basename(raster_file))
                        if extension.casefold() == '.tif'.casefold():
                            if out_raster_basename.casefold() == raster_base.casefold():
                                return f'Raster {out_raster_basename} already exists. Please enter another filename.'
        return None

    def adjust_value_from_results(self) -> None:
        """Adjust the value obtained when running from previous results."""
        if self.io_direction == IoDirection.INPUT and self.value not in self.data_handler.get_available_rasters():
            self.value = None
