"""TableArgument class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
from typing import Optional

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument, IoDirection
from .table_definition import TableDefinition


class TableArgument(Argument):
    """Table tool argument."""
    def __init__(
        self,
        name: str,
        description: str = '',
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: pd.DataFrame | str | None = None,
        hide: bool = False,
        table_definition: TableDefinition | None = None
    ):
        """Construct a table argument.

        Args:
            name (str): Python friendly argument name.
            description (str): User friendly description of the argument.
            io_direction (IoDirection): IO Direction of the argument (input or output).
            optional (bool): Is the argument optional?
            value (pd.DataFrame|str|None): Default value.
            hide (bool): Should the argument be hidden (True) or visible?
            table_definition (TableDefinition): Defines the table column types.

        Returns:
            (TableArgument): A table argument.
        """
        if value is not None and not isinstance(value, pd.DataFrame):
            value = pd.read_json(StringIO(value), orient='table')
        super().__init__(name, description, io_direction, optional, value, hide)
        self.table_definition = table_definition
        if self.io_direction == IoDirection.OUTPUT:
            self.optional = True  # output doesn't need to be specified

    def value_equals(self, value: object) -> bool:
        """Check whether the argument value is equal to the given value.

        Args:
            value: The value to check.

        Returns:
            If the argument value is equal to the given value.
        """
        if isinstance(value, pd.DataFrame):
            return value.equals(self.value)

    def _get_type(self):
        """Get a string representing the argument type (table).

        Returns:
            (str): The argument type.
        """
        return 'table'

    def _set_value(self, value):
        """Set the argument value.

        Args:
            value (pd.DataFrame): The new argument value.
        """
        self._value = value

    def get_interface_info(self) -> Optional[dict[str, object]]:
        """Get interface info for argument to be used in settings dialog.

        Returns:
            Dictionary of interface info.
        """
        if self.io_direction == IoDirection.OUTPUT or self.hide:
            return None
        interface_info = {
            'type': 'Table',
            'table_definition': self.table_definition
        }
        interface_info = interface_info | super().get_interface_info()
        return interface_info

    def _is_required_satisfied(self) -> bool | None:
        """Is the "required" value satisfied?

        For a table argument, returns True if the value is not None.

        Returns:
              True/False if the required value is satisfied. None if not required.
        """
        return self.value is not None

    def to_dict(self):
        """Convert an argument to a dictionary.

        Returns:
            (dict): The object values as a dictionary.
        """
        values = {'__class__': 'TableArgument'}
        super_values = super(TableArgument, self).to_dict()
        values.update(super_values)
        if self._value is not None:
            values['value'] = self._value.to_json(orient='table')
        if self.table_definition is not None:
            table_def_dict = self.table_definition
            if isinstance(self.table_definition, TableDefinition):
                table_def_dict = self.table_definition.to_dict()
            self._add_key_value('table_definition', table_def_dict, values)
        return values
