"""Tool class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from abc import ABC, abstractmethod
import datetime
import logging
import os
from pathlib import Path
import traceback
from typing import Optional, Union
import warnings

# 2. Third party modules
from geopandas import GeoDataFrame
import pandas as pd

# 3. Aquaveo modules
from xms.constraint import Grid
from xms.datasets.dataset_reader import DatasetReader
from xms.datasets.dataset_writer import DatasetWriter
from xms.gdal.rasters import RasterInput

# 4. Local modules
from .argument import Argument, IoDirection
from .bool_argument import BoolArgument
from .color_argument import ColorArgument
from .coverage_argument import CoverageArgument
from .data_handler import DataHandler
from .dataset_argument import DatasetArgument
from .exceptions import ToolError, ValidateError
from .file_argument import FileArgument
from .float_argument import FloatArgument
from .grid_argument import GridArgument
from .integer_argument import IntegerArgument
from .raster_argument import RasterArgument
from .string_argument import StringArgument
from .table_argument import TableArgument
from .table_definition import TableDefinition
from .timestep_argument import TimestepArgument
from .tool_log_controller import ToolLogController


def equivalent_arguments(arguments: list[Argument], default_arguments: list[Argument]) -> bool:
    """Determine if two argument lists the same except for values.

    Args:
        arguments: The tool arguments.
        default_arguments: The default tool arguments.

    Returns:
        (bool): True if no errors, False otherwise.
    """
    if len(arguments) != len(default_arguments):
        return False
    for i in range(len(default_arguments)):
        if not arguments[i].equivalent_to(default_arguments[i]):
            return False
    return True


class Tool(ABC):
    """Abstract Tool class to be overridden to make a tool."""
    def __init__(self, name: str = ''):
        """Construct a Tool.

        Args:
            name: A user-friendly short name of the tool.
        """
        self.name = name
        self.results = None
        # Define the module and class name of QDialog to view tool results. Constructor must take a single argument
        # that is the Qt parent.
        self.results_dialog_module = ''
        self.results_dialog_class = ''
        self._data_handler = DataHandler()
        self._logger_controller = ToolLogController(self.__class__.__module__)
        warnings.filterwarnings('ignore', message='All-NaN axis encountered', module='xms.datasets')

    @abstractmethod
    def initial_arguments(self) -> list[Argument]:
        """Get initial arguments for tool.

        Must override.

        Returns:
            A list of the initial tool arguments.
        """

    def enable_arguments(self, arguments: list[Argument]) -> None:
        """Called to show/hide arguments, change argument values and add new arguments.

        Args:
            arguments: The tool arguments.
        """
        return

    def validate_arguments(self, arguments: list[Argument]) -> dict[str]:
        """Called to determine if arguments are valid.

        Args:
            arguments: The tool arguments.

        Returns:
            Dictionary of errors for arguments.
        """
        return {}

    def validate_from_history(self, arguments: list[Argument]) -> bool:
        """Called to determine if arguments are valid from history.

        Args:
            arguments: The tool arguments.

        Returns:
            True if no errors, False otherwise.
        """
        init_args = self.initial_arguments()
        return equivalent_arguments(arguments, init_args)

    def internal_validate_arguments(self, arguments: list[Argument]) -> dict[str]:
        """Called to determine if arguments are valid (within bounds etc.).

        Args:
            arguments (list): The tool arguments.

        Returns:
            (dict): Dictionary of errors for arguments.
        """
        errors = {}
        for argument in arguments:
            error = argument.validate()
            if error is not None:
                errors[argument.name] = error
        return errors

    def _validation_error_str(self, errors, arguments):
        """Creates an error message from validating the arguments.

        Args:
            errors (dict): Dict of errors associated with arguments
            arguments (list): The tool arguments.

        Returns:
            (str): A description of errors or None.
        """
        arguments_by_name = {}
        for argument in arguments:
            arguments_by_name[argument.name] = argument
        # Build a list of argument errors
        message = 'Invalid arguments:\n'
        for name, error in errors.items():
            message += f'{arguments_by_name[name].description}: {error}\n'
        message = message.rstrip('\n')
        return message

    def validate(self, arguments):
        """Check the tool arguments to determine if they are valid.

        Args:
            arguments (list): The tool arguments.

        Returns:
            (str): A description of errors or None.
        """
        internal_errors = self.internal_validate_arguments(arguments)
        if internal_errors:
            return self._validation_error_str(internal_errors, arguments)

        errors = self.validate_arguments(arguments)
        if errors:
            return self._validation_error_str(errors, arguments)

        return None

    @abstractmethod
    def run(self, arguments: list[Argument]) -> None:
        """Override to run the tool.

        Args:
            arguments (list): The tool arguments.
        """

    def build_results(self, arguments: list[Argument], call_stack: str = None) -> None:
        """Build the results dict.

        Args:
            arguments (list): The tool arguments.
            call_stack (string): A string with the entire exception call stack, if it exists.
        """
        self.results = {}
        if call_stack is None:
            self.results['status'] = 'success'
        else:
            self.results['status'] = 'failure'
        saved_arguments = []
        for argument in arguments:
            saved_arguments.append(argument.to_dict())
        self.results['arguments'] = saved_arguments
        output = self.get_output()
        if call_stack is not None:
            output += '\n\n' + call_stack
        self.results['output'] = output

    def run_tool(self, arguments: list[Argument], validate_arguments: bool = True) -> None:
        """Run the tool.

        Args:
            arguments (list): The tool arguments.
            validate_arguments (bool): Should arguments be validated?
        """
        if validate_arguments:
            messages = self.validate(arguments)
            if messages is not None:
                raise ValidateError(messages)

        with self._logger_controller.record_output():
            # Need use this with statement for tests to work. This is necessary because the same logger gets used
            # across multiple tests and the handler to grab the output needs to be temporary.
            if self.echo_output:
                self.logger.info(f'Running tool "{self.name}"...')
            else:
                self.logger.info(f'$XMS_BOLD$Running tool "{self.name}"...')

            self._log_arguments(arguments)
            try:
                self.run(arguments)
                self.logger.info(f'Completed tool "{self.name}"')
                self.build_results(arguments)
            except ToolError:
                # exception description should have already been logged
                call_stack = traceback.format_exc()
                self.build_results(arguments, call_stack)
                raise
            except Exception as e:
                # log exception thrown from tool
                with self._logger_controller.allow_errors():
                    self.logger.error(f'Problem running tool "{self.name}".  More information:\n{str(e)}')
                call_stack = traceback.format_exc()
                self.build_results(arguments, traceback.format_exc())
                raise ToolError('Exception thrown from tool!') from e

    def float_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[float] = None,
        hide: bool = False,
        min_value: Optional[float] = None,
        max_value: Optional[float] = None
    ) -> FloatArgument:
        """Get a new floating point argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            min_value: The minimum value of the argument.
            max_value: The maximum value of the argument.

        Returns:
            A floating point argument.
        """
        return FloatArgument(name, description, io_direction, optional, value, hide, min_value, max_value)

    def integer_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[int] = None,
        hide: bool = False,
        min_value: Optional[int] = None,
        max_value: Optional[int] = None
    ) -> IntegerArgument:
        """Get a new integer argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            min_value: The minimum value of the argument.
            max_value: The maximum value of the argument.

        Returns:
            An integer argument.
        """
        return IntegerArgument(name, description, io_direction, optional, value, hide, min_value, max_value)

    def string_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False,
        choices: Optional[list[str]] = None,
        file: bool = False
    ):
        """Get a new string argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            choices: List of potential argument values.
            file:  False if not using a file, else use a file selector.

        Returns:
            A string argument.
        """
        return StringArgument(name, description, io_direction, optional, value, hide, choices, file)

    def table_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: pd.DataFrame = None,
        hide: bool = False,
        table_definition: TableDefinition | None = None
    ):
        """Get a new string argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            table_definition (TableDefinition): Defines the table column types.

        Returns:
            A string argument.
        """
        return TableArgument(name, description, io_direction, optional, value, hide, table_definition)

    def bool_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[bool] = None,
        hide: bool = False
    ) -> BoolArgument:
        """Get a new bool (yes/no) argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?

        Returns:
            A bolean argument.
        """
        return BoolArgument(name, description, io_direction, optional, value, hide)

    def file_argument(self, name: str = None, description: str = None, io_direction: IoDirection = IoDirection.INPUT,
                      optional: bool = False, value: Optional[str] = None, hide: bool = False,
                      file_filter: str = 'All files (*.*)', default_suffix: str = '', select_folder: bool = False)\
            -> FileArgument:
        """Get a new file argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            file_filter: Allowed file extensions
            default_suffix (str): Default file extension added if none is provided
            select_folder (bool): Should the argument select a folder instead of a file?
        Returns:
            A file argument.
        """
        return FileArgument(
            name, description, io_direction, optional, value, hide, file_filter, default_suffix, select_folder
        )

    def grid_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False
    ) -> GridArgument:
        """Get a new grid argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?

            A grid argument.
        """
        return GridArgument(self._data_handler, name, description, io_direction, optional, value, hide)

    def dataset_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False,
        filters: Optional[Union[list[str], str]] = None
    ) -> DatasetArgument:
        """Get a new dataset argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            filters: List of filters for allowed dataset types.

            A dataset argument.
        """
        return DatasetArgument(self._data_handler, name, description, io_direction, optional, value, hide, filters)

    def coverage_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False,
        filters: Optional[Union[dict[str, str], tuple[str, str]]] = None
    ) -> CoverageArgument:
        """Get a new coverage argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?
            filters: List of filters for allowed coverage types.

        Returns:
            A coverage argument.
        """
        return CoverageArgument(self._data_handler, name, description, io_direction, optional, value, hide, filters)

    def raster_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False
    ) -> RasterArgument:
        """Get a new raster argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?

        Returns:
            A raster argument.
        """
        return RasterArgument(self._data_handler, name, description, io_direction, optional, value, hide)

    def timestep_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[str] = None,
        hide: bool = False
    ) -> TimestepArgument:
        """Get a new timestep argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?

        Returns:
            A timestep argument.
        """
        return TimestepArgument(self._data_handler, name, description, io_direction, optional, value, hide)

    def color_argument(
        self,
        name: str = None,
        description: str = None,
        io_direction: IoDirection = IoDirection.INPUT,
        optional: bool = False,
        value: Optional[tuple[int, int, int]] = None,
        hide: bool = False
    ) -> Argument:
        """Get a new color argument.

        Args:
            name: Python friendly argument name.
            description: User friendly description of the argument.
            io_direction: IO Direction of the argument (input or output).
            optional: Is the argument optional?
            value: Default value.
            hide: Is the argument visible?

        Returns:
            A color argument.
        """
        return ColorArgument(name, description, io_direction, optional, value, hide)

    def get_input_grid(self, grid_name: str) -> Grid:
        """Get a grid chosen from the list of available grids (from DataHandler.available_grids).

        Args:
            grid_name: The grid to retrieve.

        Returns:
            The constrained grid object.
        """
        return self._data_handler.get_input_grid(grid_name)

    def get_grid_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input grid from a UUID.

        Args:
            uuid: The grid UUID.

        Returns:
            The grid name or None.
        """
        return self._data_handler.get_grid_name_from_uuid(uuid)

    def get_grid_datasets(self, grid: str, filters: Optional[list] = None) -> list[str]:
        """Returns a list of datasets for an input grid.

        Args:
            grid (str): The grid.
            filters (Optional[list]: The filters to apply to the returned list.

        Returns:
            (List[str]): A list of the grid's datasets.
        """
        return self._data_handler.get_grid_datasets(grid, filters)

    def set_output_grid(self, grid: Grid, argument: GridArgument, projection: str = None,
                        force_ugrid: bool = True) -> None:
        """Set output grid for argument.

        Args:
            grid: The output grid.
            argument: The grid argument which has the output grid path with name.
            projection: The WKT of the grid.
            force_ugrid (bool): Whether to save the output as an UGrid in SMS.
        """
        self._data_handler.set_output_grid(grid, argument.value, projection, force_ugrid)

    def get_input_dataset(self, dataset_name: str) -> Optional[DatasetReader]:
        """Get a dataset chosen from the list of available datasets (from DataHandler.available_datasets).

        Args:
            dataset_name: The dataset to retrieve.

        Returns:
            The dataset's reader.
        """
        return self._data_handler.get_input_dataset(dataset_name)

    def get_dataset_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input dataset from a UUID.

        Args:
            uuid: The dataset UUID.

        Returns:
            The dataset name or None.
        """
        return self._data_handler.get_dataset_name_from_uuid(uuid)

    def get_input_dataset_grid(self, dataset_name: str) -> Optional[Grid]:
        """Get the geometry of a dataset chosen from the list of available datasets.

         Available datasets come from DataHandler.available_datasets.

        Args:
            dataset_name: The dataset to retrieve.

        Returns:
            A constrained Grid.
        """
        return self._data_handler.get_input_dataset_grid(dataset_name)

    def set_grid_uuid_for_testing(self, grid_name: str, uuid: str) -> None:
        """Set the UUID for a grid. Used to get grid for a dataset for testing.

        CURRENTLY ONLY USED FOR TESTING.

        Args:
            grid_name: The name of the grid.
            uuid: The UUID.
        """
        self._data_handler.set_grid_uuid_for_testing(grid_name, uuid)

    def get_output_dataset_writer(
        self,
        name: str = 'Dataset',
        dset_uuid: str = None,
        geom_uuid: str = '',
        num_components: int = 1,
        ref_time: Optional[Union[float, datetime.datetime]] = None,
        null_value: float = None,
        time_units: str = 'Days',
        units: str = '',
        dtype: str = 'f',
        use_activity_as_null: bool = False,
        location: str = 'points'
    ) -> DatasetWriter:
        """Get an output dataset writer given a dataset argument value.

        Args:
            name: Dataset argument value.
            dset_uuid: UUID of the dataset
            geom_uuid: UUID of the dataset's geometry
            num_components: The number of data components (1=scalar, 2=vector, 3=3D vector)
            ref_time : The dataset's reference time. Either a Julian float or a Python datetime.datetime
            null_value: The null value of the dataset, if there is one
            time_units: The dataset's time units. One of: 'Seconds', 'Minutes', 'Hours', 'Days'
            units: Units of the dataset values
            dtype: Data type of the dataset's values. One of: 'f', 'd', 'i'
            use_activity_as_null: If True, inactive values (0) will be treated as null values when computing
                timestep mins and maxs. Implies that activity array is on same location as data values (i.e. has same
                shape).
            location: Location of the dataset values. One of the XMDF_DATA_LOCATIONS keys. Note that this does
                not usually need to be set XMS is going to ignore it in most cases. XMS will try to determine the
                dataset location based on the geometries currently loaded (number of nodes, number of points). Here for
                historical reasons.):

        Returns:
            A output dataset writer.
        """
        return self._data_handler.get_output_dataset_writer(
            name=name,
            dset_uuid=dset_uuid,
            geom_uuid=geom_uuid,
            num_components=num_components,
            ref_time=ref_time,
            null_value=null_value,
            time_units=time_units,
            units=units,
            dtype=dtype,
            use_activity_as_null=use_activity_as_null,
            location=location
        )

    def set_output_dataset(self, dataset_writer: DatasetWriter) -> None:
        """Set output dataset for argument.

        Args:
            dataset_writer: The output dataset.
        """
        self._data_handler.set_output_dataset(dataset_writer)

    def _validate_input_dataset(self, argument: DatasetArgument, errors: dict[str, str]) -> DatasetReader | None:
        """Validate a dataset for an argument exists.

        Args:
            argument: The dataset argument.
            errors: Dictionary of errors keyed by argument name. Gets modified if there are errors.

        Returns:
            The dataset, if we could read it.
        """
        if argument.value is None:
            return None

        dataset = self.get_input_dataset(argument.text_value)
        if not dataset:
            errors[argument.name] = 'Could not read data set.'
        return dataset

    def get_input_coverage(self, coverage_name: str) -> GeoDataFrame:
        """Get a coverage chosen from the list of available coverages (from DataHandler.available_coverages).

        Args:
            coverage_name (str): The coverage to retrieve.

        Returns:
            (obj): The coverage object.
        """
        return self._data_handler.get_input_coverage(coverage_name)

    def get_input_coverage_file(self, coverage_name: str) -> str:
        """Get a coverage chosen from the list of available coverages.

        Args:
            coverage_name: The raster to retrieve.

        Returns:
            The coverage file path.
        """
        return self._data_handler.get_input_coverage_file(coverage_name)

    def get_coverage_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input coverage from a UUID.

        Args:
            uuid: The coverage UUID.

        Returns:
            The coverage name or None.
        """
        return self._data_handler.get_coverage_name_from_uuid(uuid)

    def set_output_coverage(self, coverage: GeoDataFrame, argument: CoverageArgument):
        """Set output coverage for argument.

        Args:
            coverage (GeoDataFrame): The output grid.
            argument (CoverageArgument): The coverage argument who's value is the coverage name.
        """
        self._data_handler.set_output_coverage(coverage, argument.value)

    def get_input_raster(self, raster_name: str) -> RasterInput:
        """Get a raster chosen from the list of available rasters (from DataHandler.available_rasters).

        Args:
            raster_name: The raster to retrieve.

        Returns:
            The raster object.
        """
        return self._data_handler.get_input_raster(raster_name)

    def get_input_raster_file(self, raster_name: str) -> str:
        """Get a raster chosen from the list of available rasters (self.available_rasters).

        Args:
            raster_name: The raster to retrieve.

        Returns:
            The raster file path.
        """
        return self._data_handler.get_input_raster_file(raster_name)

    def get_raster_name_from_uuid(self, uuid: str) -> Optional[str]:
        """Get an input raster from a UUID.

        Args:
            uuid: The raster UUID.

        Returns:
            The raster name or None.
        """
        return self._data_handler.get_raster_name_from_uuid(uuid)

    def load_raster_file(self, raster_file: str) -> RasterInput:
        """Load a raster file.

        Args:
            raster_file: The raster file path.

        Returns:
            The raster input.
        """
        return self._data_handler.load_raster_file(raster_file)

    def get_output_raster(self, raster_name: str, raster_extension: str = '.tif') -> str:
        """Get path to write output raster.

        Args:
            raster_name (str): The raster argument who's value is the raster name. For testing the raster name is the
                output file path in the files folder.
            raster_extension: The file extension to add to the raster_name.

        Returns:
            The file path of the new raster.
        """
        return self._data_handler.get_output_raster(raster_name, raster_extension)

    def set_output_raster_file(self, raster_file: str, raster_name: str, raster_format: str = 'GTiff',
                               raster_extension: str = '.tif'):
        """Set output raster for argument.

        Args:
            raster_file: The output raster file path.
            raster_name: The raster argument who's value is the raster name. For testing the raster name is the
                output file path in the files folder.
            raster_format: The short name of the raster format string.  See https://gdal.org/drivers/raster/index.html
                for a list of possible strings.
            raster_extension: The file extension to add to the raster_name.
        """
        self._data_handler.set_output_raster_file(raster_file, raster_name, raster_format, raster_extension)

    def send_output_to_xms(self) -> None:
        """Send output to XMS.

        This should be called from XMS only.
        """
        self._data_handler.send_output_to_xms()

    def fail(self, message: str) -> None:
        """Print a message that the tool has failed and raise a runtime exception.

        Args:
            message: The failure error message to print.
        """
        with self._logger_controller.allow_errors():
            self.logger.error(message)
        raise ToolError(message)

    @property
    def logger(self) -> logging.Logger:
        """Get the tool logger.

        Returns:
            The logger.
        """
        return self._logger_controller.logger

    @logger.setter
    def logger(self, logger: logging.Logger):
        """Set the tool logger.

        Args:
            logger: The logger.
        """
        self._logger_controller = ToolLogController(logger=logger)

    @property
    def echo_output(self) -> bool:
        """Is logging for GUI dialog?

        Returns:
            If logging is for a GUI dialog.
        """
        return self._logger_controller.echo_output

    @property
    def default_wkt(self) -> str:
        """Get the WKT of the default coordinate system.

        Returns:
            The WKT of the default coordinate system.
        """
        return self._data_handler.get_default_wkt()

    @default_wkt.setter
    def default_wkt(self, wkt: str):
        """Set the WKT of the default coordinate system.

        Works only for Python command line. Xms uses the display projection.

        Args:
            wkt: The WKT of the default coordinate system.
        """
        self._data_handler.set_default_wkt(wkt)

    @property
    def vertical_datum(self) -> str:
        """Get the vertical datum of the default coordinate system.

        Returns:
            The vertical datum of the default coordinate system.
        """
        return self._data_handler.get_vertical_datum()

    @vertical_datum.setter
    def vertical_datum(self, datum: str):
        """Set the vertical datum of the default coordinate system.

        Works only for Python command line. Xms uses the display projection datum.

        Args:
            datum: The vertical datum of the default coordinate system.
        """
        return self._data_handler.set_vertical_datum(datum)

    @property
    def vertical_units(self) -> str:
        """Get the vertical units of the default coordinate system.

        Returns:
            The vertical units of the default coordinate system.
        """
        return self._data_handler.get_vertical_units()

    @vertical_units.setter
    def vertical_units(self, units: str) -> None:
        """Set the vertical units of the default coordinate system.

        Works only for Python command line. Xms uses the display projection units.

        Args:
            units: The vertical units of the default coordinate system.
        """
        self._data_handler.set_vertical_units(units)

    @echo_output.setter
    def echo_output(self, echo_output: bool) -> None:
        """Echo output to stdout and stderr?

        Args:
            echo_output: Should output be echoed to stdout and stderr?
        """
        self._logger_controller.echo_output = echo_output

    def set_data_handler(self, data_handler: DataHandler) -> None:
        """Set the tool DataHandler.

        Args:
            data_handler: The data handler.
        """
        self._data_handler = data_handler

    def set_gui_data_folder(self, folder: str) -> None:
        """Set folder which data_handler uses for files.

        Args:
            folder: Path to the folder.
        """
        self._data_handler.file_folder = os.path.abspath(folder)

    def set_output_folder(self, folder: Union[str, Path]) -> None:
        """Set folder which data_handler uses for file output.

        Args:
            folder: Path to the folder.
        """
        self._data_handler.output_folder = os.path.abspath(str(folder))

    def get_testing_output(self) -> str:
        """Get output suitable for testing. No date time stamps.

        Returns:
            Output suitable for testing with no date time stamps.
        """
        # output = io.StringIO()
        # for out in self._output:
        #     output.write(f'[{out[0]}] {out[2]}')
        # return output.getvalue()
        return self._logger_controller.get_testing_output()

    def get_output(self) -> str:
        """Get output suitable for user. Includes date time stamps.

        Returns:
            Output suitable for user with date time stamps.
        """
        # output = io.StringIO()
        # for out in self._output:
        #     output.write(f'[{out[0]}] {out[1]}: - {out[2]}')
        # return output.getvalue()
        return self._logger_controller.get_output()

    def get_arguments_from_results(self, results: dict) -> list[Argument]:
        """Get arguments from a results dictionary.

        Args:
            results (dict): The results dictionary after running the tool.

        Returns:
            List of arguments used from previous tool run.
        """
        arguments = []
        arguments_list = results['arguments']
        for argument in arguments_list:
            class_name = argument.pop('__class__')
            argument.pop('uuid', None)
            if class_name == 'BoolArgument':
                arguments.append(self.bool_argument(**argument))
            elif class_name == 'ColorArgument':
                arguments.append(self.color_argument(**argument))
            elif class_name == 'CoverageArgument':
                arguments.append(self.coverage_argument(**argument))
            elif class_name == 'DatasetArgument':
                arguments.append(self.dataset_argument(**argument))
            elif class_name == 'FileArgument':
                arguments.append(self.file_argument(**argument))
            elif class_name == 'FloatArgument':
                arguments.append(self.float_argument(**argument))
            elif class_name == 'GridArgument':
                arguments.append(self.grid_argument(**argument))
            elif class_name == 'IntegerArgument':
                arguments.append(self.integer_argument(**argument))
            elif class_name == 'RasterArgument':
                arguments.append(self.raster_argument(**argument))
            elif class_name == 'StringArgument':
                arguments.append(self.string_argument(**argument))
            elif class_name == 'TableArgument':
                arguments.append(self.table_argument(**argument))
            elif class_name == 'TimestepArgument':
                arguments.append(self.timestep_argument(**argument))
        for argument in arguments:
            argument.adjust_value_from_results()
        return arguments

    def _log_arguments(self, arguments: list[Argument]) -> None:
        """Prints all the arguments and their values to the logger.

        Args:
            arguments: List of arguments.
        """
        lines = []
        for argument in arguments:
            lines.append(f'\'{argument.name}\': {argument.text_value}')
        inputs = 'Input parameters: {'
        inputs += ', '.join(lines)
        inputs += '}'
        self.logger.info(inputs)
