"""ToolInterface class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy

# 2. Third party modules
import pandas

# 3. Aquaveo modules

# 4. Local modules
from .argument import Argument
from .tool import Tool


class ToolInterface:
    """Provides an interface to get tool arguments values for an interactive interface."""

    def __init__(self, tool: Tool, arguments: list[Argument]):
        """Construct a ToolInterface.

        Args:
            tool: The tool.
            arguments: List of the tool arguments.
        """
        self.tool = tool
        self.arguments = arguments

    def get_interface_values(self) -> list[dict[str, object]]:
        """Get the info to show in an interactive interface.

        Returns: A list of dictionaries with an item for each visible argument.
        """
        interface_info = []
        for argument in self.arguments:
            if argument.show:
                info = argument.get_interface_info()
                if info is not None:
                    interface_info.append(info)
        return interface_info

    def apply_interface_values(self, interface_values: list[dict[str, object]]) -> bool:
        """Apply values from the interactive interface to the arguments.

        Args:
            interface_values: The interface values to apply.

        Returns:
            True if any argument values were changed, False otherwise.
        """
        values_by_name = {interface_value['name']: interface_value for interface_value in interface_values}
        changed = False
        for argument in self.arguments:
            if argument.name in values_by_name:
                value = values_by_name[argument.name]['value']
                if isinstance(value, str) and value == Argument.NONE_SELECTED:
                    # Argument.NONE_SELECTED is like being None
                    value = None
                if not argument.value_equals(value):
                    changed = True
                    argument.value = copy.deepcopy(value)
                    if argument.value is None:
                        argument.value = ''
        changed = self._cascaded_enable_arguments() or changed
        return changed

    def _cascaded_enable_arguments(self) -> bool:
        """Enable tool arguments until no changes are made.

        Returns:
            True if any argument values were changed, False otherwise.
        """
        changed = False
        equal = False
        while not equal:
            old_values = self.get_interface_values()
            self.tool.enable_arguments(self.arguments)
            new_values = self.get_interface_values()
            equal = _interface_lists_equal(old_values, new_values)
            changed = changed or equal
        return changed


def _interface_lists_equal(list_1: list[dict], list_2: list[dict]) -> bool:
    """Compare two lists of dictionaries.

    Args:
        list_1: A list of dictionaries to be compared with list_2.
        list_2: A list of dictionaries to be compared with list_1.

    Returns:
        bool: True if the two lists are equal, False otherwise.
    """
    equal = len(list_1) == len(list_2)
    if equal:
        for i in range(len(list_1)):
            dict_1 = list_1[i]
            dict_2 = list_2[i]
            # compare dictionary keys
            equal = dict_1.keys() == dict_2.keys()
            if equal:
                # compare dictionary values
                for key in dict_1.keys():
                    value1 = dict_1[key]
                    value2 = dict_2[key]
                    # need to compare pandas.DataFrame differently
                    if isinstance(value1, pandas.DataFrame):
                        equal = value1.equals(value2)
                    else:
                        equal = value1 == value2
                    if not equal:
                        break
    return equal
