"""Logger class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass
import io
import logging

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from .exceptions import ToolError


class ToolLogController:
    """Captures logged output for tools."""
    def __init__(self, name='', logger=None):
        """Construct logging class for tool.

        Args:
            name (str): Name to use with logger.
            logger (logging.Logger): The Logger.
        """
        if logger is None:
            if name == '':
                name = __name__
            self._logger = logging.getLogger(name)
            self._handler = ToolLogController._Handler()
            self._logger.setLevel(logging.INFO)
        else:
            self._logger = logger
            self._handler = None

    @property
    def echo_output(self):
        """Determine if output will be echoed to stdout.

        Returns:
            (bool): True if output will be echoed to stdout.
        """
        if self._handler is None:
            return False
        return self._handler.echo_output

    @echo_output.setter
    def echo_output(self, echo_output):
        """Set logger to echo output to stdout.

        Args:
            echo_output (bool): If output should be echoed to stdout.
        """
        if self._handler is not None:
            self._handler.echo_output = echo_output

    @property
    def logger(self):
        """Get logger.

        Returns:
            (logging.Logger): Get logger.
        """
        return self._logger

    def get_testing_output(self):
        """Get output suitable for testing. No date time stamps.

        Returns:
            (str): Output suitable for testing with no date time stamps.
        """
        if self._handler is None:
            return ''
        self._handler.flush()
        output = self._handler._stream.getvalue()
        output = output.replace('$XMS_BOLD$', '')
        return output

    def get_output(self):
        """Get output suitable for user. Includes date time stamps.

        Returns:
            (str): Output suitable for testing with no date time stamps.
        """
        if self._handler is None:
            return ''
        self._handler.flush()
        return self._handler._stream.getvalue()

    def record_output(self) -> 'ToolLogController._RecordOutputManager':
        """Allow output to be recorded in a with statement. Should not be called by tools.

        Need use this with statement for tests to work. This is necessary because the same logger gets used across
        multiple tests and the handler to grab the output needs to be temporary.

        Returns:
            A handler context that can be used to record output in a with statement.
        """
        return ToolLogController._RecordOutputManager(self._logger.name, self._handler)

    def allow_errors(self) -> 'ToolLogController._AllowErrorsManager':
        """Allow errors to logger in a with statement. Should not be called by tools. Instead, raise an exception.

        Returns:
            A handler context that can be used in a with statment to allows errors to be logged.
        """
        return ToolLogController._AllowErrorsManager(self._handler)

    class _Handler(logging.Handler):
        """Handler to save and potentially echo output."""
        def __init__(self):
            """Initialize class for handling a logger message."""
            logging.Handler.__init__(self)
            self.echo_output = True
            self.allow_errors = False
            self._stream = io.StringIO()

        def handle(self, record: logging.LogRecord) -> None:
            """Handle a logged message.

            Args:
                record (logging.LogRecord): The message record.
            """
            print(record.msg, *record.args, file=self._stream)
            if record.levelname == 'ERROR' and not self.allow_errors:
                error = (
                    'Should not call logger.error on a tools logger. '
                    'Instead tools should either call self.fail or raise an exception.'
                )
                raise ToolError(error)
            if self.echo_output:
                print(record.msg, *record.args)

    @dataclass
    class _RecordOutputManager:
        """Context manager to begin and end output recording."""
        logger_name: str
        handler: 'ToolLogController._Handler'

        def __enter__(self):
            logger = logging.getLogger(self.logger_name)
            logger.addHandler(self.handler)

        def __exit__(self, *exit_info) -> bool:
            logger = logging.getLogger(self.logger_name)
            logger.removeHandler(self.handler)
            return False

    class _AllowErrorsManager:
        """Context manager to allow calls to logger.error from Tool class."""
        def __init__(self, handler: 'ToolLogController._Handler'):
            self.handler = handler
            self.old_allow_errors = False

        def __enter__(self):
            self.old_allow_errors = self.handler.allow_errors
            self.handler.allow_errors = True

        def __exit__(self, *exit_info) -> bool:
            self.handler.allow_errors = self.old_allow_errors
            return False
