"""ToolDialog class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import json
import os
import traceback
from typing import List, Optional
import webbrowser

# 2. Third party modules
from PySide2.QtCore import QEvent, QObject, Qt
from PySide2.QtGui import QIcon
from PySide2.QtWebEngineWidgets import QWebEnginePage, QWebEngineSettings, QWebEngineView
from PySide2.QtWidgets import (
    QApplication, QDialog, QDialogButtonBox, QHBoxLayout, QScrollArea, QSplitter, QToolBar, QVBoxLayout, QWidget
)

# 3. Aquaveo modules
from xms.guipy import settings
from xms.guipy.dialogs.help_getter import HelpGetter
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists, get_xms_icon, XmsDlg
from xms.guipy.widgets.widget_builder import setup_toolbar
from xms.tool_core import Argument, Tool, ToolError, ToolInterface  # noqa I100,I201

# 4. Local modules
from xms.tool_gui.argument_layout_helper import ArgumentLayoutHelper
from xms.tool_gui.xms_data_handler import XmsDataHandler

FOCUS_EVENT_TYPE = QEvent.registerEventType()


class ToolDialog(XmsDlg):
    """Dialog for entering values for tool arguments."""
    back_icon = ':/resources/icons/back.svg'
    forward_icon = ':/resources/icons/forward.svg'

    def __init__(self, win_cont: Optional[QWidget], tool: Tool, tool_arguments: List[Argument], title: str = '',
                 description: str = '', tool_url: Optional[str] = None, tool_uuid: str = ''):
        """Initializes the class, sets up the UI.

        Args:
            win_cont (Optional[QWidget]): Parent window.
            tool (Tool): The tool to run
            tool_arguments (List[Argument]): Tool argument list.
            title (str): Dialog title.
            description (str): Description to appear on the side of the dialog.
            tool_url (str): Optional override for the tool help URL.
            tool_uuid (str): The UUID of the tool to use for documentation.
        """
        super().__init__(win_cont, 'xmstool_gui.XmsTool_Dailog')
        self.testing = win_cont is None
        self.tool_arguments = tool_arguments
        self.tool_interface = ToolInterface(tool, tool_arguments)
        self.interface_values = []
        self.web_page = None
        self.web_page_loaded = False
        self.web_load_error = False
        self.setting_error_text = False
        self.tool_url = tool_url
        self.btn_actions = {}  # Web view navigation buttons
        self.title = title
        self.description = description
        self.tool = tool
        self.tool_uuid = tool_uuid
        self.focus_widgets = []

        # Add the simplified XMS tool arguments to a dictionary, and use that to create an
        # ArgumentLayoutHelper, which will be used for the layout
        self._setup_argument_helper()
        self.widgets = dict()

        if len(title) > 0:
            self.setWindowTitle(title)
        else:
            self.setWindowTitle('Tool')
        self.setup_ui()
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())
        self._on_argument_changed(force_change=True)

    def _setup_argument_helper(self):
        """Initialize the argument layout helper."""
        self.argument_layout_helper = ArgumentLayoutHelper(self)
        self.argument_layout_helper.end_do_value_changed.connect(self._on_argument_changed)

    def accept(self):
        """Accept."""
        # First validate tool arguments
        message = self.tool.validate(self.tool_arguments)
        if message:
            title = self.tool.name
            message_with_ok(self, message, title, win_icon=self.windowIcon())
        else:
            super().accept()

    def setup_ui(self):
        """Set up the dialog widgets."""
        # Dialog QVBoxLayout with QTabWidget then QDialogButtonBox
        self._set_layout('', 'top_layout', QVBoxLayout())
        self.widgets['h_layout'] = QHBoxLayout()

        self._setup_ui_arguments()

        # Set up the description pane
        self._add_web_view()
        self.add_splitter()

        self.widgets['h_layout'].addWidget(self.widgets['splitter'])
        self.widgets['top_layout'].addLayout(self.widgets['h_layout'])
        self.widgets['btn_box'] = QDialogButtonBox()

        self.widgets['top_layout'].addWidget(self.widgets['btn_box'])

        # QDialogButtonBox with Ok and Cancel buttons
        self.widgets['btn_box'].setOrientation(Qt.Horizontal)
        self.widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help)
        self.widgets['btn_box'].accepted.connect(self.accept)
        self.widgets['btn_box'].rejected.connect(super().reject)
        self.widgets['btn_box'].helpRequested.connect(self.help_requested)

    def _default_html(self):
        """Returns the default html string for the description window."""
        return """
              <!DOCTYPE html>
              <html>
              <body>

              <h1 style="color:blue;">{0}</h1>
              <p style="color:black;">{1}</p>

              </body>
              </html>
              """

    def _error_html(self):
        """Returns the error html string for the description window."""
        error_html = """
                     <!DOCTYPE html>
                     <html>
                     <head>
                     <title>Use Help Button</title>
                     </head>
                     <body>
                     <h1 style="color:blue;">{0}</h1>
                     <p style="color:red;">Loading web content failed: {1}</p>
                     <p style="color:black;">{2}</p>
                     <p style="color:black;">Use the Help button to load help page in a web browser.{1}</p>
                     </body>
                     </html>
                     """
        return error_html

    def _on_web_page_loaded(self, loaded):
        """Set the description to the tool web page once it loads.

        Args:
            loaded (bool): True if the web page loaded successfully
        """
        if self.setting_error_text:  # pragma no cover - not sure if this is necessary
            return  # Tests don't hit this but it may need to be here. See comment below.

        if loaded:
            self.widgets['web_browser'].setPage(self.web_page)
            self.widgets['navigation_bar'].show()
        else:
            self.setting_error_text = True  # Don't display the error text HTML as the URL
            # Make sure we get the original URL if initial web page load fails.
            url = self.widgets['web_browser'].url().toDisplayString() if self.tool_url is None else self.tool_url
            str_html = self._error_html().format(self.title, url, self.description)
            self.widgets['web_browser'].setHtml(str_html)
            self.setting_error_text = False
            self.web_load_error = True
        self.widgets['web_browser'].setFocusPolicy(Qt.ClickFocus)
        self.web_page_loaded = True

    def _on_url_changed(self, loaded):  # pragma no cover - not testing web navigation
        """Enable/disable navigation buttons when user changes the web view page."""
        # I added the try/except because for some reason this gets called when the dialog is closing and it throws
        # "Internal C++ object (PySide2.QtWidgets.QAction) already deleted."
        try:
            can_go_back = self.widgets['web_browser'].history().canGoBack()
            can_go_forward = self.widgets['web_browser'].history().canGoForward()
            self.widgets['navigation_bar'].widgetForAction(self.btn_actions[self.back_icon]).setEnabled(can_go_back)
            self.widgets['navigation_bar'].widgetForAction(
                self.btn_actions[self.forward_icon]).setEnabled(can_go_forward)
        except Exception:
            pass

    def _on_argument_changed(self, force_change=False):
        """Handles changing an argument by loading new interface values.

        Args:
            force_change (bool): flag to force that the arguments have changed.
        """
        changed = self.tool_interface.apply_interface_values(self.interface_values)
        if changed or force_change:
            self.interface_values = self.tool_interface.get_interface_values()
            clear_layout(self.widgets['arg_layout'])
            self.argument_layout_helper.add_arguments_to_layout(self.widgets['arg_layout'], self.interface_values)
            self.widgets['arg_layout'].addStretch()
            QApplication.postEvent(self, UpdateFocusWidgetsEvent(self))

    def _update_focus_widgets(self):
        """Updates the focus widgets and sets the focus on the appropriate widget."""
        self.set_all_widgets_click_focus()
        focus_widgets, focus_index = self.argument_layout_helper.get_focus_widgets()
        self.focus_widgets = focus_widgets
        focus_widgets.append(self.widgets['btn_box'].button(QDialogButtonBox.Ok))
        focus_widgets.append(self.widgets['btn_box'].button(QDialogButtonBox.Cancel))
        focus_widgets.append(self.widgets['btn_box'].button(QDialogButtonBox.Help))
        widget_count = len(focus_widgets)
        # set event filters to handle the tab key
        for index in range(widget_count):
            current_widget = focus_widgets[index]
            current_widget.installEventFilter(self)
        focus_widgets[focus_index].setFocus(Qt.TabFocusReason)

    def event(self, event: QEvent) -> bool:
        """Handle focus events.

        Args:
            event: an event object containing information about the event.

        Returns:
            True if the event is handled.
        """
        if event.type() == FOCUS_EVENT_TYPE:
            self._update_focus_widgets()
            return True
        return super().event(event)

    def eventFilter(self, watched: QObject, event: QEvent) -> bool:  # noqa: N802
        """Filter to handle tab events.

        Args:
            watched: The object that is being watched for events.
            event: The event that was received by the watched object.

        Returns:
            True if the event is filtered and should not be processed further.
        """
        if event.type() == QEvent.KeyPress:
            if event.key() == Qt.Key_Tab or event.key() == Qt.Key_Backtab:
                if watched in self.focus_widgets:
                    index = self.focus_widgets.index(watched)
                    if event.key() == Qt.Key_Backtab:
                        next_index = (index - 1) % len(self.focus_widgets)
                    else:
                        next_index = (index + 1) % len(self.focus_widgets)
                    self.focus_widgets[next_index].setFocus(Qt.TabFocusReason)
                    return True
        return super().eventFilter(watched, event)

    def set_all_widgets_click_focus(self):
        """Set the focus policy on all widgets to NoFocus."""
        # Find all child widgets of the central widget (recursively)
        all_widgets = self.findChildren(QWidget)

        # Iterate through each widget and set its focus policy to NoFocus
        for widget in all_widgets:
            widget.setFocusPolicy(Qt.ClickFocus)

    def _add_web_view(self):
        """Add the description web view pane."""
        # Create a widget to lay out the navigation bar and web view.
        self.widgets['description_widget'] = QWidget()
        self._set_layout('description_widget', 'description_layout', QVBoxLayout())
        # Create the web view. Initially has the static text defined in the tool class.
        self.widgets['web_browser'] = QWebEngineView()
        self.widgets['web_browser'].urlChanged.connect(self._on_url_changed)
        self.web_page = QWebEnginePage()  # Load the URL in the background
        self.web_page.settings().setAttribute(QWebEngineSettings.LinksIncludedInFocusChain, False)
        self.web_page.settings().setAttribute(QWebEngineSettings.FocusOnNavigationEnabled, False)
        self.web_page.loadFinished.connect(self._on_web_page_loaded)
        # Back and forward navigation buttons
        self._add_navigation_bar()
        self.widgets['description_layout'].addWidget(self.widgets['navigation_bar'])
        self.widgets['description_layout'].addWidget(self.widgets['web_browser'])
        self.widgets['web_browser'].setFocusPolicy(Qt.ClickFocus)
        self.widgets['navigation_bar'].setFocusPolicy(Qt.ClickFocus)
        # Initialize the text in the description web view.
        self._setup_ui_browser_initial_text()

    def add_splitter(self):
        """Adds a QSplitter between the tables so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        self.widgets['splitter'] = QSplitter(self)
        self.widgets['splitter'].setOrientation(Qt.Horizontal)
        self.widgets['splitter'].addWidget(self.widgets['scrollable_area'])
        self.widgets['splitter'].addWidget(self.widgets['description_widget'])
        self.widgets['splitter'].setSizes([50, 50])
        self.widgets['splitter'].setStyleSheet(
            'QSplitter::handle:horizontal { background-color: lightgrey; }'
            'QSplitter::handle:vertical { background-color: lightgrey; }'
        )
        self.widgets['splitter'].setAccessibleName('Splitter')
        self.widgets['splitter'].setCollapsible(0, False)
        self.widgets['splitter'].setCollapsible(1, True)

    def _add_navigation_bar(self):
        """Add the navigation bar for the description web view."""
        self.widgets['navigation_bar'] = QToolBar()
        button_list = [
            [self.back_icon, 'Go Back', self.widgets['web_browser'].back],
            [self.forward_icon, 'Go Forward', self.widgets['web_browser'].forward],
        ]
        self.btn_actions = setup_toolbar(self.widgets['navigation_bar'], button_list)
        self.widgets['navigation_bar'].hide()  # Don't show until the web page is loaded

    def _setup_ui_browser_initial_text(self):
        """Set the initial test for the description pane."""
        self.update_tool_help_url()
        title = f'{self.title} (loading web content...)' if self.tool_url else self.title
        str_html = self._default_html().format(title, self.description)
        self.widgets['web_browser'].setHtml(str_html)
        if self.tool_url:  # Start loading the web page if there is one.
            self.web_page.load(self.tool_url)

    def update_tool_help_url(self):
        """Update the tool help URL used for web page and help button."""
        if self.tool_url is None:
            dialog_help_url = 'https://www.xmswiki.com/wiki/Tool_Dialog_Help'
            getter = HelpGetter(self.tool_uuid, default='', dialog_help_url=dialog_help_url)
            self.tool_url = getter.url()

    def _setup_ui_arguments(self):
        """Set up the general widgets."""
        self.widgets['args_widget'] = QWidget()
        self._set_layout('args_widget', 'arg_layout', QVBoxLayout())
        self.widgets['arg_layout'].addStretch()
        self.widgets['scrollable_area'] = QScrollArea(self)
        self.widgets['scrollable_area'].setWidget(self.widgets['args_widget'])
        self.widgets['scrollable_area'].setWidgetResizable(True)

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (str): Name of parent widget in self.widgets or '' for self
            layout_name (QLay): Name of layout in parent widget
            layout (str): QtLayout to be used
        """
        self.widgets[layout_name] = layout
        if parent_name:
            parent = self.widgets[parent_name]
        else:
            parent = self
        parent.setLayout(self.widgets[layout_name])

    def help_requested(self):
        """Called when the Help button is clicked."""
        if self.tool_url:
            webbrowser.open(self.tool_url)
        else:
            webbrowser.open('https://www.xmswiki.com')

    def get_argument_widget(self, name: str) -> Optional[QWidget]:
        """Get widget for argument by name (used for testing).

        Args:
            name (str): The widget name.

        Returns:
            (Optional[QWidget]): The Qt widget.
        """
        ui_item = self.argument_layout_helper.ui_items.get(name, None)
        if ui_item is not None:
            return ui_item.value_widget
        return None

    def get_argument_label(self, name: str) -> Optional[QWidget]:
        """Get label for argument by name (used for testing).

        Args:
            name (str): The widget name.

        Returns:
            (Optional[QWidget]): The Qt label widget.
        """
        ui_item = self.argument_layout_helper.ui_items.get(name, None)
        if ui_item is not None:
            return ui_item.label_widget
        return None

    def get_argument_widget_names(self) -> List[str]:
        """Get the names of displayed arguments (used for testing).

        Returns:
            (List[str]): The names of the displayed arguments.
        """
        return list(self.argument_layout_helper.ui_items.keys())


class UpdateFocusWidgetsEvent(QEvent):
    """Update focus widgets event."""
    def __init__(self, tool_dialog: ToolDialog):
        """Initialize the event.

        Args:
            tool_dialog: The tool dialog.
        """
        super().__init__(QEvent.Type(FOCUS_EVENT_TYPE))
        self.tool_dialog = tool_dialog


def clear_layout(layout, delete_widgets=True):
    """Clear all widgets under a layout.

    Args:
        layout: The QLayout.
        delete_widgets: Should the widgets be deleted.
    """
    item = layout.takeAt(0)
    while item is not None:
        if delete_widgets:
            widget = item.widget()
            if widget is not None:
                widget.deleteLater()
        child_layout = item.layout()
        if child_layout is not None:
            clear_layout(child_layout, delete_widgets)
        item = layout.takeAt(0)


def _override_default_arguments(tool, json_object):
    """Override a tool's default argument values with those specified in JSON.

    Notes:
        The `description` key/value pair must always be specified and must match the text in the class definition.
        Any other key/value pairs will override the default value for that argument.

    Args:
        tool (Tool): The tool
        json_object (dict): The parsed JSON argument specifier dict

    Returns:
        (list): List of arguments used from previous tool run.
    """
    initial_arguments = [argument.to_dict() for argument in tool.initial_arguments()]
    for initial_argument in initial_arguments:
        arg_name = initial_argument.get('name', '')
        for specified_argument in json_object['arguments']:
            if specified_argument.get('name', '') == arg_name:
                initial_argument.update(specified_argument)
                break
    return tool.get_arguments_from_results({'arguments': initial_arguments})


def _run_results_dialog(module_name, class_name, parent, tool=None):
    """Run a custom, tool-defined results dialog after the tool finishes running.

    Args:
        module_name (str): Import path to the dialog module
        class_name (str): Class namem of the QDialog
        parent (QDialog): The parent dialog
        tool (Tool): The tool
    """
    module = __import__(module_name, fromlist=[class_name])
    klass = getattr(module, class_name)
    dlg = klass(parent)
    dlg.tool = tool
    return dlg.exec()


def run_tool_dialog(query, input_file, output_file, win_cont, tool):
    """Run the tool dialog for a tool.

    Args:
        query (:obj:'xms.api.dmi.Query'): XMS interprocess communication interface
        input_file (str): Path to the tool input file.
        output_file (str): Path to the tool output file.
        win_cont (:obj:'PySide2.QtWidgets.QWidget'): The window container.
        tool (xms.tool.Tool): The tool.

    Returns:
        bool: True if dialog was accepted, False if user canceled
    """
    # Set current working directory to default.
    os.chdir(settings.get_file_browser_directory())

    # use xms data handler
    if query is None:
        tool.set_gui_data_folder(get_test_files_path())
    else:
        data_handler = XmsDataHandler(query)
        data_handler.temp_folder = os.path.dirname(input_file)
        tool.set_data_handler(data_handler)

    # check for saved arguments and load them
    tool_arguments = None
    tool_name = ''
    tool_description = ''
    tool_uuid = ''
    if os.path.isfile(input_file):
        with open(input_file) as json_file:
            json_object = json.load(json_file)
        using_default_override = json_object.get('using_default_override', False)
        if using_default_override:
            tool_arguments = _override_default_arguments(tool, json_object)
        elif 'arguments' in json_object:
            tool_arguments = tool.get_arguments_from_results(json_object)
        tool_name = json_object.get('tool_name', tool.__class__.__name__)
        tool_description = json_object.get('tool_description', '')
        tool_uuid = json_object.get('tool_uuid', '')

    if tool_arguments is None:
        tool_arguments = tool.initial_arguments()
    else:
        if not tool.validate_from_history(tool_arguments):
            icon_path = get_xms_icon()
            icon = None
            if icon_path:
                icon = QIcon(icon_path)
            message = (
                'The arguments in the history do not match the current tool arguments. '
                'The default tool arguments will be used.'
            )
            message_with_ok(win_cont, message, 'Argument Mismatch', win_icon=icon)
            tool_arguments = tool.initial_arguments()
    tool_dialog = ToolDialog(
        win_cont, tool, tool_arguments, title=tool_name, description=tool_description, tool_uuid=tool_uuid
    )
    accepted = False
    if tool_dialog.exec() == QDialog.Accepted:
        print('Running tool \'{0}\' at {1}'.format(tool.name, datetime.datetime.now()))
        accepted = run_tool_with_feedback(win_cont, tool, tool_arguments)
        tool.send_output_to_xms()
        results = tool.results
        if results is not None:
            with open(output_file, 'w') as fp:
                json.dump(results, fp, indent=4)
    return accepted


def run_tool_with_feedback(win_cont, tool, tool_arguments, auto_str='', modal=False):
    """Run a tool using the feedback dialog.

    Args:
        win_cont (:obj:'PySide2.QtWidgets.QWidget'): The window container.
        tool (xms.tool.Tool): The tool.
        tool_arguments (list): The tool arguments.
        auto_str (str): Auto load string
        modal (bool): Flag to run the dialog as modal instead of modeless

    Returns:
        bool: True if dialog was accepted, False if user canceled
    """
    def _run_tool():
        try:
            tool.run_tool(tool_arguments, validate_arguments=False)
        except ToolError:
            # a ToolError should have already been reported
            pass
        except Exception:
            call_stack = traceback.format_exc()
            tool.logger.error(f'Unexpected problem running tool "{tool.name}".  More information:\n{call_stack}')

    tool.echo_output = False
    testing = win_cont is None
    display_text = {
        'title': tool.name,
        'working_prompt': f'Executing "{tool.name}" tool.',
        'error_prompt': 'Error(s) encountered while running tool.',
        'warning_prompt': 'Warning(s) encountered while running tool.',
        'success_prompt': 'Successfully ran tool.',
        'note': '',
        # 'auto_load': 'Close this dialog automatically when finished.',
        'log_format': '- %(message)s',
        'use_colors': True,
        'auto_load': 'testing' if testing else auto_str
    }
    ensure_qapplication_exists()
    worker = ProcessFeedbackThread(_run_tool, None)
    feedback_dlg = ProcessFeedbackDlg(
        display_text=display_text, logger_name=tool.__class__.__module__, worker=worker, parent=win_cont
    )
    if modal:
        feedback_dlg.setModal(True)
    if tool.results_dialog_module:
        worker.processing_finished.connect(
            lambda: _run_results_dialog(tool.results_dialog_module, tool.results_dialog_class, feedback_dlg, tool)
        )
    feedback_dlg.testing = feedback_testing_mode()
    dialog_result = feedback_dlg.exec()
    if feedback_dlg.testing:
        worker.processing_finished.emit()
    return dialog_result


def get_test_files_path():
    """Returns the full path to the 'tests/files' directory.

    Returns:
        (str): See description.
    """
    file_dir = os.path.dirname(os.path.realpath(__file__))
    files_path = os.path.join(file_dir, '..', '..', 'tests', 'files')
    return os.path.abspath(files_path)


def feedback_testing_mode() -> bool:
    """Should feedback dialog be in testing mode."""
    return 'XMSTOOL_GUI_TESTING' in os.environ
