"""Dialog for cleaning shapes tool."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QAbstractItemView
from shapely.geometry import LineString

# 3. Aquaveo modules

# 4. Local modules
import xms.tool_xms.algorithms.geom_funcs_with_shapely as gsh
from xms.tool_xms.gui.clean_shapes_dialog import CleanShapesDialog

# Columns in the primary table
CP_AREA = 3
STR_GEOM_IDX = 'Geometry List Index'
STR_DIST_TO_P = 'Distance to Primary'
STR_ID = 'ID'

# A lot of functions from CleanShapesDialog are not needed because we aren't using the display
to_override = [
    '_update_current_action',
    '_update_viewers',
    '_scroll_to_selection',
    '_sel_first_primary_object',
    '_do_updates',
    '_setup_action_preview',
    '_clear_preview_window'
]


def override_methods_with_pass(method_names):
    """Override.

    Args:
        method_names (list): List of functions in the parent class that we are overriding with "pass"
    """
    def decorator(cls):
        for name in method_names:
            def make_method(name):
                def method(self, *args, **kwargs):
                    pass

                method.__name__ = name
                return method

            setattr(cls, name, make_method(name))
        return cls

    return decorator


@override_methods_with_pass(to_override)
class MergePolysByDistanceInvisibleDialog(CleanShapesDialog):
    """Dialog for the cleaning shapes tool."""
    def __init__(self, parent, tool):
        """Constructor.

        Args:
            parent (:obj:`QObject`): The parent object
            tool (Tool): The tool that opened up this dialog
        """
        super().__init__(parent, False, False, True)
        self._mpbd_ignore_polys = set()
        self.tool = tool
        self._primary_containing_poly = None
        self._overlapped_polys = []

    def do_mpbd_tool(self):
        """Merge all polygons in coverage."""
        if self._process_data() is False:
            return

        tol = self.tool._mpbd_tol

        # set the tolerance
        self.ui.edt_action.setText(str(tol))

        # set the filter distance
        self._max_dist = tol
        self.ui.edt_dist.setText(str(self._max_dist))
        self._on_change_filter()

        # sort by smallest area
        self.ui.tbl_primary.sortByColumn(CP_AREA, Qt.AscendingOrder)

        # set the action to merge
        self.ui.cbx_action.setCurrentText('Merge')
        self._sel_action = 'Merge'

        # set the selection for the secondary spreadsheet (not sure if this is necessary)
        self.ui.grp_secondary.setEnabled(True)
        sm = QAbstractItemView.SingleSelection
        self.ui.tbl_primary.setSelectionMode(sm)

        # total_polys = len(self._ss_df)
        self.tool.logger.info('Merging polygons...')
        num_left = len(self._ss_df)
        while num_left > 0:
            self._sel_first_selectable_primary_object()
            if self._sel_geom_idx is None:
                self.tool.logger.info('Finished merging polygons.')
                break
            elif self._sel_secondary_geom_idx is None:
                self._mpbd_ignore_polys.add(self._sel_geom_idx)  # no polys are within tolerance
                num_left = len(self._ss_df) - len(self._mpbd_ignore_polys)
                self.tool.logger.info(f'{num_left} polygons left to process.')
                continue

            # do the merge
            self._create_action_geoms()
            secondary_attempts = set()

            valid_merge = self._was_merge_valid()
            while valid_merge is False:
                # that merge didn't work - are there others to try?
                secondary_attempts.add(self._sel_secondary_geom_idx)  # add the previous attempt
                cur_sel_row_secondary = self.ui.tbl_secondary.selectedIndexes()[0].row()
                self._sel_first_selectable_secondary_object(cur_sel_row_secondary + 1)
                if self._sel_secondary_geom_idx is None or self._sel_secondary_geom_idx in secondary_attempts:
                    # no more to try
                    self._mpbd_ignore_polys.add(self._sel_geom_idx)
                    break
                self._create_action_geoms()
                valid_merge = self._was_merge_valid()

            if valid_merge:
                self.tool.logger.info(f'Merging {self._sel_geom_idx} and {self._sel_secondary_geom_idx}.')
                self._accept_merge_polys_action()
                if self._sel_geom_idx in self._overlapped_polys:
                    self._overlapped_polys.remove(self._sel_geom_idx)

                self._sel_geom_idx = None
                self._sel_secondary_geom_idx = None

        self.tool.logger.info('Creating coverage.')
        self._create_cov()
        return True

    def _was_merge_valid(self):
        """Merge all polygons in coverage."""
        if self._mod_geoms is None or len(self._mod_geoms) != 1 or self._mod_geoms[0].is_valid is False:
            return False

        if self._primary_containing_poly is not None:
            mod_poly_ls = LineString(gsh.perim_pts_from_sh_poly(self._mod_geoms[0]))
            p_c_poly = self._geoms[self._primary_containing_poly]
            outer_ls = LineString(gsh.perim_pts_from_sh_poly(p_c_poly))
            if mod_poly_ls.intersects(outer_ls):
                return False

            # did this overlap any existing polygons? we will go through with it, but fix it right after
            possible_overlap_polys = self._secondary_ss_df[STR_GEOM_IDX].tolist()
            for poly_geom_idx in possible_overlap_polys:
                if poly_geom_idx == self._sel_secondary_geom_idx or poly_geom_idx == self._primary_containing_poly:
                    continue
                if self._mod_geoms[0].intersects(self._geoms[poly_geom_idx]):
                    self._overlapped_polys.append(poly_geom_idx)
                    if poly_geom_idx in self._mpbd_ignore_polys:
                        self._mpbd_ignore_polys.remove(poly_geom_idx)

        return True

    def _sel_first_selectable_primary_object(self, starting_index=0):
        """Selects an object in the primary spreadsheet that is not set to be ignored."""
        if len(self._overlapped_polys) > 0:
            starting_index = self._get_ss_row_for_geom_idx(True, self._overlapped_polys[0])
        row_count = self.p_model.rowCount()
        self.ui.tbl_primary.selectionModel().blockSignals(True)
        self._iterating_for_selectable = True
        self.ui.tbl_primary.selectRow(starting_index)
        valid_selection = self._is_sel_item_valid(True)
        index = starting_index + 1 if starting_index < row_count else 0
        while valid_selection is False and index != starting_index:
            self.ui.tbl_primary.selectRow(index)
            valid_selection = self._is_sel_item_valid(True)
            if valid_selection is False:
                index = index + 1 if index < row_count else 0
        self._iterating_for_selectable = False
        self.ui.tbl_primary.selectionModel().blockSignals(False)

        if index == starting_index:
            self.ui.tbl_primary.selectionModel().clearSelection()

        self._primary_containing_poly = self._get_containing_poly(self._get_prop_for_selected_row(True, STR_GEOM_IDX))
        self._update_primary_selection()

    def _is_sel_item_valid(self, is_primary):
        """Only allow the item to be selected if we haven't put it in the ignore set and it's not a hole."""
        if is_primary:
            geom_idx = self._get_prop_for_selected_row(True, STR_GEOM_IDX)
            if len(self._overlapped_polys) > 0:
                return geom_idx in self._overlapped_polys
            return geom_idx not in self._mpbd_ignore_polys
        else:
            s_geom_idx = self._get_prop_for_selected_row(False, STR_GEOM_IDX)
            if s_geom_idx in self._mpbd_ignore_polys:
                return False

            dist_to_primary = self._get_prop_for_selected_row(False, STR_DIST_TO_P)
            if dist_to_primary > self._max_dist:
                return False  # too far
            secondary_holes = self._get_hole_polys(s_geom_idx)
            if self._sel_geom_idx in secondary_holes:
                return False  # primary is a hole in the secondary

            # if either is a hole, they must both be, and have the same containing poly
            s_containing_poly = self._get_containing_poly(s_geom_idx)
            return self._primary_containing_poly == s_containing_poly

    def _sel_first_selectable_secondary_object(self, starting_index=0):
        """Selects an object in the secondary spreadsheet."""
        if self._in_setup or len(self._secondary_display) == 0:
            return

        row_count = self.s_model.rowCount()
        self._iterating_for_selectable = True
        self.ui.tbl_secondary.selectionModel().blockSignals(True)
        valid_selection = False
        if starting_index < row_count:
            self.ui.tbl_secondary.selectRow(starting_index)
            valid_selection = self._is_sel_item_valid(False)
            if valid_selection is False:
                if starting_index < row_count - 1:
                    index = starting_index + 1 if starting_index < row_count - 1 else 0
                    while valid_selection is False and index != starting_index:
                        self.ui.tbl_secondary.selectRow(index)
                        valid_selection = self._is_sel_item_valid(False)
                        if valid_selection is False:
                            if index == row_count:
                                break
                            else:
                                index = index + 1

        self._iterating_for_selectable = False
        self.ui.tbl_secondary.selectionModel().blockSignals(False)
        if valid_selection is False:
            self.ui.tbl_secondary.selectionModel().clearSelection()

        self._update_secondary_selection()

    def _transform_all_geom_to_latlon(self):
        """These don't really matter because we aren't displaying."""
        self._geom_latlon_locs = [[[] for _ in geom] for geom in self.tool._geometry_locs]
        self._latlon_extents = [[] for geom in self._geoms]

    def _transform_geom(self, geom):
        """Transform the geometry to latlon for display.

        Args:
            geom (Object): Geometry to transform
        Return:
            geom (Object): Transformed item
            extents (list): Transformed extents
        """
        # doesn't matter because we aren't displaying
        return [], []
